# -------------------------------------------------------------------------------------
# MKDoc::Session
# -------------------------------------------------------------------------------------
# Author : Jean-Michel Hiver <jhiver@mkdoc.com>.
# Copyright : (c) MKDoc Holdings Ltd, 2003
#
# This object / and plug-in attempts to set a session on the target
# machine. It will use either a cookie or user login if the user is
# logged in.
# -------------------------------------------------------------------------------------
package MKDoc::Session;
use flo::HTTP::Header;
use flo::Standard;
use warnings;
use strict;

use base qw /MKDoc::SimplePersistence/;

use constant COOKIE_ID => $ENV{MKD__SESSION} || 'MKDocSessionID';
use constant EXPIRES   => '+1y';


##
# $class->main();
# ---------------
# this is the bit of MKDoc::Session which attempts to set a
# cookie if no session_id was sent.
##
sub main
{
    my $class = shift;
    $class->session_id_sent() && return;
    
    my $cgi    = flo::Standard::cgi();
    my $value  = $class->_gen_id();
    my $cookie = $cgi->cookie (COOKIE_ID) || $cgi->cookie
	( -name    => COOKIE_ID,
	  -value   => $value,
	  -expires => EXPIRES );
    
    my $header = new flo::HTTP::Header;
    $header->set ("Set-Cookie: " . $cookie . "; path=/");

    return;
}


##
# $class->load();
# ---------------
# Instanciates a new MKDoc::Session using the session_id_sent()
# method. If session_id_sent() returns undef, then return undef.
##
sub load
{
    my $class      = shift;
    my $session_id = shift || $class->session_id_sent() || return;
    return $class->SUPER::load ($session_id) || bless {}, $class;
}


##
# $self->save();
# --------------
# Saves the current MKDoc::Session state.
##
sub save
{
    my $self       = shift;
    my $session_id = $self->session_id_sent() || return;
    $self->{id}    = $session_id;
    return $self->SUPER::save();
}


##
# $class->session_id_sent();
# --------------------------
# Returns the session ID which the client has sent,
# FALSE if no session ID was sent.
##
sub session_id_sent
{
    my $self = shift;
    my $cgi  = flo::Standard::cgi();

    my $cgi_cookie = $cgi->cookie (COOKIE_ID());
    return "cookie:$cgi_cookie" if ($cgi_cookie);
    
    my $env_user = $ENV{REMOTE_USER};
    return "user:$env_user" if ($env_user);
    
    return;
}


1;


__END__

