# -------------------------------------------------------------------------------------
# flo::Record::Preference
# -------------------------------------------------------------------------------------
#
#       Author : Jean-Michel Hiver (jhiver@mkdoc.com),
#                Steve Purkis <spurkis@mkdoc.com>
#    Copyright : (c) MKDoc Holdings Ltd, 2002.
#
#       Unauthorized modification, use, reuse, distribution or redistribution
#       of this module is stricly forbidden.
#
#    Description:
#
#       Every record from the 'Preference' table will be blessed into this class
#       so that we can add extra method to access to in the template engine.
#
#       All preferences are associated with a User (aka Editor).
#
#    Synopsis:
#
#       use flo::Record::Preference qw/ LIKE DONT_MIND HATE YES NO /;
#
#       # create + save new pref:
#       my $pref = new flo::Record::Preference
#       $pref->set_user($user)
#            ->set('Fish', LIKES)
#            ->save;
#
#       # check value:
#       do {...} if ($pref->value eq 'something');
#
#       # load existing pref:
#       my $pref = load flo::Record::Preference( ID => $id );
#       my $user = $pref->user;
#
#       # load user's prefs:
#       my @prefs = load flo::Record::Preference( User => $user );
#
#       # delete existing preference:
#       $pref->delete;
#
# -------------------------------------------------------------------------------------

package flo::Record::Preference;

use flo::Standard qw/ table /;

use Exporter;
use strict;
use 5.008_000;


##
# INHERITS FROM
# =============
#
# Exporter
##

use base qw/ Exporter /;


##
# CLASS VARIABLES
# ===============
##

## Version (from CVS revision string).
our $VERSION = (split(/ /, '$Revision: 1.16 $'))[1];


## Boolean switch controlling debug printing to STDERR.
our $DEBUG = 0;


## Things that are ok to Export.
our @EXPORT_OK = qw/ LIKE DONT_MIND HATE YES NO /;


## SQL Table name of this class
use constant TABLE_NAME => 'Preference';


## Flag to represent the User liking something.
use constant LIKE => +1;


## Flag to represent the User not minding something.
use constant DONT_MIND => '0E0';


## Flag to represent the User erhm.. dis-liking something.
use constant HATE => -1;


## Flag to represent the User agreeing to something.
use constant YES => '1';


## Flag to represent the User disagreeing to something.
use constant NO => '0E0';



##
# INSTANCE VARIABLES
# ==================
#
# A Preference contains these fields from the DB:
#
#  ID:          Integer object key
#  Editor_ID:   Integer flo::Record::Editor object key
#  Name:        255 char Name
#  Value:       255 char Value
#
# Of course, you'll always use methods to access these fields, won't you?
# That's good.  It might break if you don't.
##


##
# CONSTRUCTOR
# ===========
##

##
# $pref = new flo::Record::Preference()
# -------------------------------------
#   Instantiates and returns a new object.
##
sub new
{
    my $class = shift;
    $class    = ref $class || $class;
    my $self  = bless {}, $class;
    $self->_initialize(@_);
    return $self;
}


##
# $class->_initialize(@args)
# --------------------------
#   Initializes this object, called by new().  This is empty, and
#   here to be overrided if necessary.
##
sub _initialize
{
    my $self = shift || return;
    return;
}


##
# @objs = load flo::Record::Preference( %args );
# ----------------------------------------------
#   Loads all Preference objects matching given search criteria.
#   Accepts one or more of:
#
#     *  ID           load object by unique id
#     *  Editor       load by editor obj
#        User         load by editor obj
#        Editor_ID    load by editor id
#        User_ID      load by editor id
#        Name         load by name
#        Value        load by value
#
#   Passing an Editor object will clobber the value of Editor_ID.
#   As seen above, 'User' is a synonym for 'Editor'.
#   Additional or undefined fields are ignored.
#
#   Returns one object in scalar context, a list of Preference objects,
#   or undef on failure.
##
sub load
{
    my $class = shift || return;
    my $args  = $class->_filter_search_args(@_) || return;

    warn "  +  loading preferences matching: {" .
      join (' ', map {"$_: $args->{$_}"} keys %$args) . "}\n" if ($DEBUG);

    my $table = $class->table_handle;
    my @prefs = $table->search( %$args )->fetch_all;

    return wantarray
      ? @prefs
      : shift @prefs;
}



##
# CLASS METHODS
# =============
##


##
# $args = $class->_filter_search_args(%args);
# -------------------------------------------
#   Filters search arguments, as defined by $class->load().
#   Returns \%args, or undef on error.
##
sub _filter_search_args
{
    my $class = shift || return;
    my %args  = @_;

    # User == Editor
    foreach my $key (grep /^User/, keys %args)
    {
	my $new_key = $key;
	$new_key    =~ s/User/Editor/;
	$args{$new_key} = delete $args{$key};
    }

    # do we have an editor obj?
    if (my $editor = delete $args{Editor})
    {
	$args{Editor_ID} = $editor->id;
    }

    # don't pass db fields that don't exist to MKDoc::SQL::Table
    for (keys %args)
    {
	delete($args{$_}) unless ((/(ID)|(Editor_ID)|(Name)|(Value)/)
				  and defined($args{$_}));
    }

    return keys(%args) ? \%args : undef;
}


##
# $table = $class->table_handle;
# ------------------------------
#   See $TABLE_NAME.
#   Returns the MKDoc::SQL::Table handle of this object.
##
sub table_handle
{
    my $class = shift;
    my $table = table( $class->TABLE_NAME );
    unless ($table)
    {
	warn "error opening " . $class->TABLE_NAME . " table: $!";
	return;
    }
    return $table;
}



##
# INSTANCE METHODS
# ================
##


##
# $id = $self->id;
# ----------------
#   Returns the value of the 'ID' attribute.
##
sub id
{
    my $self = shift;
    return $self->{ID};
}


##
# $obj = $obj->set_id ($id);
# --------------------------
#   Set the 'ID' attribute to $id.  Returns this object.
##
sub set_id
{
    my $self    = shift;
    $self->{ID} = shift;
    return $self;
}


##
# $id = $self->editor_id;
# -----------------------
#   Returns the value of the 'Editor_ID' attribute.
##
sub editor_id
{
    my $self = shift;
    return $self->{Editor_ID};
}


##
# $obj = $obj->set_editor_id ($editor_id);
# ----------------------------------------
#   Set 'Editor_ID' attribute to $id.  Returns this object.
##
sub set_editor_id
{
    my $self           = shift;
    $self->{Editor_ID} = shift;
    return $self;
}


##
# $obj->editor;
# -------------
#   Returns the flo::Record::Editor object associated with this object.
##
sub editor
{
    my $self = shift;
    return load flo::Record::Editor (ID => $self->editor_id);
}


##
# $obj = $obj->set_editor ($editor);
# ----------------------------------
#   Set 'Editor_ID' attribute to the flo::Record::Editor object's id.
#   Returns this object.
##
sub set_editor
{
    my $self   = shift;
    my $editor = shift;
    return $self->set_editor_id ($editor->id);
}


##
# $obj->user;
# -----------
#   Synonym for editor().
##
sub user
{
    my $self = shift;
    return $self->editor(@_);
}


##
# $obj = $obj->set_user ($user);
# ------------------------------
#   Synonym for set_editor().
##
sub set_user
{
    my $self = shift;
    return $self->set_editor(@_);
}


##
# $name = $self->name;
# --------------------
#   Returns the value of the 'Name' attribute.
##
sub name
{
    my $self = shift;
    return $self->{Name};
}


##
# $obj = $obj->set_name ($name);
# ------------------------------
#   Set the 'Name' attribute to $name.  Returns this object.
##
sub set_name
{
    my $self      = shift;
    $self->{Name} = shift;
    return $self;
}


##
# $value = $self->value;
# --------------------
#   Returns the value of the 'Value' attribute.
##
sub value
{
    my $self = shift;
    return $self->{Value};
}


##
# $obj = $obj->set_value ($value);
# --------------------------------
#   Set the 'Value' attribute to $value.  Returns this object.
##
sub set_value
{
    my $self       = shift;
    $self->{Value} = shift;
    return $self;
}


##
# ($name, $value) = $obj->get;
# ----------------------------
#   Get the value of the 'Name' and 'Value' attributes.
##
sub get
{
    my $self = shift;
    return ($self->name, $self->value);
}


##
# $obj = $obj->set ($name, $value);
# ---------------------------------
#   Sets the 'Name' and 'Value' attributes.  Returns this object.
##
sub set
{
    my $self = shift;
    $self->set_name  (shift)
         ->set_value (shift);
    return $self;
}



##
# $bool = $obj->save;
# -------------------
#   Saves this preference into the database (inserts or updates as appropriate).
#   Returns 1 on success, or undef on error.
##
sub save ($)
{
    my $self = shift || return;

    warn "  +  saving preference " . $self->to_string . " to db\n" if ($DEBUG);

    my $table = $self->table_handle;

    # delete preference?
    if ($self->value eq DONT_MIND or $self->value eq NO) {
	warn "\t  +  preference should not be stored in db\n" if ($DEBUG);
	# if we have an id, exists in db & we should delete it:
	return  $self->id
	  ? $self->delete
	  : 1;
    }

    # to insert or to update... ?
    if ($self->load( ID => $self->id))
    {
	warn "\t  i  updating existing preference\n" if ($DEBUG);
	my $rs = $table->update($self, { 'ID' => $self->id });
	unless ($rs and ($rs != '0E0'))
	{
	    warn "  !  error updating existing preference " . $self->to_string . " in db!\n";
	    return;
	}
    }
    else
    {
	warn "\t  i  inserting new preference\n" if ($DEBUG);
	my $id = $table->insert( $self );
	if ($id and ($id != '0E0'))
	{
	    $self->set_id($id);
	}
	else
	{
	    warn "  !  error inserting new preference " . $self->to_string . " into db!\n";
	    return;
	}
    }

    return 1;
}



##
# $bool = $obj->delete;
# ---------------------
#   Deletes this preference from the database.
#
#   Note: does not update object (potential problem?)  But does
#   delete ID, in case you try & save() it later...  This could
#   really be improved with some more error (ID?) checking, but
#   no time ATM!
#      -spurkis
#
#   Returns 1 on success, or undef on error.
##
sub delete ($)
{
    my $self = shift || return;

    warn "  +  deleting preference ".$self->to_string." from db\n" if ($DEBUG);

    my $table = $self->table_handle;

    # try delete...
    my $rs = $table->delete( ID => $self->id );

    if ((! defined $rs) or $rs eq '0E0')
    {
	warn "  !  error deleting preference ".$self->to_string." from db!" unless ($rs eq '0E0');
	return;
    }

    # delete ID just in case...
    $self->set_id(undef);

    return 1;
}



##
# $str = $obj->to_string;
# -----------------------
#   Returns a string representation of this object.
##
sub to_string ()
{
    my $self = shift || return;
    return ref($self) . '=[' . join (' ', map {"$_: $self->{$_}"} keys %$self) . ']';
}




1;


__END__


