# -------------------------------------------------------------------------------------
# flo::Record::Ticket
# -------------------------------------------------------------------------------------
#
#       Author : Steve Purkis <spurkis@mkdoc.com>
#    Copyright : (c) 2002, MKDoc Holdings Ltd.
#
#      Unauthorized modification, use, reuse, distribution or redistribution
#      of this module is stricly forbidden.
#
# -------------------------------------------------------------------------------------

=head1 NAME

flo::Record::Ticket - OO interface to the I<Tickets> table.

=head1 SYNOPSIS

 use flo::Record::Ticket;

 # to create a new ticket & send email
 my $ticket = new flo::Record::Ticket;
 $ticket->generateId();
 $ticket->ExpiryDate(time()+10000);
 $ticket->Email('spurkis@mkdoc.com');
 $ticket->Action('subscribe');
 $ticket->saveToDb();


 # to validate a new ticket from db:
 my $ticket = new flo::Record::Ticket($id);
 if (defined $ticket) {
     # ticket is valid, do stuff
     $ticket->deleteFromDb();
 } else {
     die "ticket is invalid!";
 }


 # to look up a range of tickets, use
 # flo::Standard and MKDoc::SQL::Table

=head1 DESCRIPTION

An OO interface to the I<Tickets> database table for the MKDoc System.  This class
is intended to be used by whatever templating engine is being used (ie: Petal) as
an API to using Tickets (which are 'random' strings that are used by mkdoc to
confirm things, like the receipt of an email).

Every record from the Tickets table will be blessed into this class automatically
by I<flo::sql::Table>.  Extra methods for manipulating and creating Tickets are
available here.

Exceptions are not dealt with at this level.

=cut


package flo::Record::Ticket;
use flo::Standard qw /general_config cgi path_info raw_path_info table template current_document/;
use MKDoc::SQL::Condition;

use Time::Local;

use strict;
use 5.008_000;
use utf8;


# -------------------------------------------------------------------------------------

=item B< $TABLE_NAME >

SQL Table name of this class.

=cut

use constant TABLE_NAME => 'Tickets';



# -------------------------------------------------------------------------------------

=item B< @ticket_id_chars >

A list of characters to use when generating Ticket Ids.

=cut

our @ticket_id_chars = ('A'..'Z', 'a'..'z', 0..9);



# -------------------------------------------------------------------------------------
# -------------------------------------------------------------------------------------

=back

=head1 INSTANCE VARIABLES

A ticket contains these fields from the DB:

  ID:         25 char key, unique for the life of this ticket
  Email:      Email address associated w/this ticket
  ExpiryDate: When this ticket expires
  Action:     Action to be performed when this ticket is validated

And these additional fields:

  _modified_: out of synch w/db? see $t->modified()
  _in_db_:    record for this ticket in db? see $t->inDb()

Of course, you'll always use methods to access these fields, won't you?
That's good.

=cut



# -------------------------------------------------------------------------------------
# -------------------------------------------------------------------------------------

=head1 CONSTRUCTOR

=over 4

=item B< $t = new flo::Record::Ticket( [ $ticket_id ] ) >

Creates a new flo::Record::Ticket object.  If C<$ticket_id> is passed, the
ticket is populated from the database via C<load_ticket_from_db()> (an error
occurs if $ticket_id is not found in the db).

Returns a new, unmodified object, or undef on error.

=cut

sub new {
    my $class = shift || return;
    my $id    = shift;
    my $self  = {};
    $class    = ref($class) if ref($class); # created from another obj?

    # should we try to populate ticket from database?
    if ($id) {
	$self = load_ticket_from_db($id) || return;
    } else {
	# avoid clobbering MKDoc::SQL::Table's bless_into, just in case
	bless $self, $class;
    }

    # mark as unmodified
    $self->modified(0);

    return $self;
}


sub id
{
    my $self = shift;
    return $self->{ID};
}


sub email
{
    my $self = shift;
    my $res = $self->{Email};
    use Encode;
    Encode::_utf8_on ($res);
    return $res;
    # return $self->{Email};
}


sub expiry_date_text
{
    my $self = shift;
    return $self->{ExpiryDate};
}


# -------------------------------------------------------------------------------------

=item B< $t = [ $t-E<gt> ] load_ticket_from_db( $ticket_id ) >

This is not quite a constructor, but it's close enough.
It loads a ticket from the database given a valid C<$ticket_id> by way of
flo::Standard programming hooks.

May be called from an object or standalone context.

Returns a new, unmodified ticket if successful, or undef on error.

=cut

sub load_ticket_from_db {
    my $thing = shift || return;
    my $id;
    if (ref($thing)) {
	# $thing is likely an object, so disregard it.
	# and get $id from args.
	$id = shift || return;
    } else {
	# $thing is likely the $id:
	$id = $thing;
    }

    # look up ticket_id, flo::Standard stylee
    my $ticket_table = table('Tickets') || return;;
    my $ticket       = $ticket_table->get( $id );

    return unless ($ticket);

    # bless_into *should* bless $ticket into this class, but
    # just in case that changes:
    if ($ticket->can('modified')) {
	# set as unmodified
	$ticket->modified(0);
    }

    # return new ticket obj
    return $ticket;
}


# -------------------------------------------------------------------------------------

=item B< $tkt = load flo::Record::Ticket( %args ) >

Loads all Ticket objects matching given search criteria.
Accepts one or more of:

        ID           load object by unique id
        Email        load by user's email
        ExpiryDate   load by expiry date
        Expiry       load by expiry date
        Action       load by action name

Expiry is a synonym for ExpiryDate.
Additional or undefined fields are ignored.

Returns one object in scalar context, a list in list context,
or undef on failure.

=cut

sub load
{
    my $class = shift || return;
    my $args  = $class->_filter_search_args(@_) || return;
    my $table = $class->table_handle;
    my @prefs = $table->search( %$args )->fetch_all;

    return wantarray
      ? @prefs
      : shift @prefs;
}



# -------------------------------------------------------------------------------------
# -------------------------------------------------------------------------------------

=back

=head1 STATIC METHODS

=over 4

=item B< $args = $class-E<gt>_filter_search_args(%args); >

Filters search arguments, as defined by $class->load().
Returns \%args, or undef on error.

=cut

sub _filter_search_args
{
    my $class = shift || return;
    my %args  = @_;

    # convert Expiry into ExpiryDate (Expiry doesn't exist in this table)
    $args{ExpiryDate} = delete $args{Expiry} if (exists $args{Expiry});

    # don't pass db fields that don't exist to MKDoc::SQL::Table
    for (keys %args) {
	delete($args{$_}) unless ((/(ID)|(Email)|(ExpiryDate)|(Action)/)
				  and (defined $args{$_}));
    }

    # make sure we have an sql date:
    $args{ExpiryDate} = $class->time_as_sql_date($args{ExpiryDate})
      if (exists $args{ExpiryDate} and $args{ExpiryDate} =~ /^\d+$/);

    return keys(%args) ? \%args : undef;
}



=item B< $ids = [ $t-E<gt> ] clear_old_tickets_from_db() >

A utility function that 'reaps' old tickets past their expiry date from the
database.  Uses flo::Standard programming hooks.

May be called from an object or standalone context.

Returns number of rows deleted on success, or undef on error.

=cut

sub clear_old_tickets_from_db {
    # reap old tickets, flo::Standard stylee
    my $ticket_table = table('Tickets') || return;;

    # note: using NOW() doesn't work (it's quoted by MKDoc::SQL::Condition) so
    # we use a mysql date string
    my $cond = new MKDoc::SQL::Condition;
    $cond->add('ExpiryDate', '<', time_as_sql_date(time));
    my $deleted = $ticket_table->delete($cond);

    # return no. tickets reaped.
    return defined($deleted) ? $deleted : 0;
}



# -------------------------------------------------------------------------------------

=item B< $date_str = time_as_sql_date( $time ) >

Converts system C<$time> (returned by time()) to a MySQL date string of the
form: I<YYYY-MM-DD HH:MM:SS>.  No error checking is done.

Returns sql date string, or undef on error.

=cut

sub time_as_sql_date {
    my $time = shift || return;

    # break up $time using localtime
    my ($sec, $min, $hour, $mday, $mon, $year, $wday, $yday, $isdst) = localtime($time);
    $year += 1900;
    $mon  += 1;

    # bung it into a string
    my $date_str = sprintf("%04d-%02d-%02d %02d:%02d:%02d",$year, $mon, $mday, $hour, $min, $sec);

    # and return
    return($date_str);
}



# -------------------------------------------------------------------------------------

=item B< $date_str = sql_date_as_time( $date ) >

Converts a MySQL date string of the form I<YYYY-MM-DD HH:MM:SS> into system time
(ie: as returned by time()).  Error checking is done by I<Time::Local>.

Returns time as an integer, or undef on error.

=cut

sub sql_date_as_time {
    my $date = shift || return;

    # optimization note: move following RE to a class qr//'d variable

    if ($date =~ /(\d{4})-(\d{2})-(\d{2})\s+(\d{2}):(\d{2}):(\d{2})/) {
	# for clarity's sake:
	my ($year, $mon, $mday, $hour, $min, $sec) = ($1, $2, $3, $4, $5, $6);
	# convert back to localtime() format:
	$mon -= 1;
	return timelocal($sec, $min, $hour, $mday, $mon, $year);
    } else {
	warn "  !  couldn't parse expriy date: $date\n";
	return;
    }
}



# -------------------------------------------------------------------------------------

=item B< $table = $class->table_handle >

See $TABLE_NAME.

Returns the MKDoc::SQL::Table handle of this object.

=cut

sub table_handle
{
    my $class = shift;
    my $table = table( $class->TABLE_NAME );
    unless ($table)
    {
	warn "error opening " . $class->TABLE_NAME . " table: $!";
	return;
    }
    return $table;
}


# -------------------------------------------------------------------------------------
# -------------------------------------------------------------------------------------

=back

=head1 PUBLIC METHODS

=over 4

=item B< $id = $t-E<gt>TicketId( [ $id ] ) >

If C<$id> is passed, sets this ticket's id, which must be a 25 char string -- but
you'd be better off using C<generateTicketId()> below to set ticket Id's.

Returns this ticket's id, or undef on error.

=cut

sub TicketId {
    my $self = shift || return;
    my $id   = shift;

    # set id if need be
    if ($id) {
	# validate length
	if (length($id) > 25) {
	    warn "  !  ticket id [$id] too long - truncating to 25 chars!\n";
	    $id = substr($id, 0, 25);
	}

	# set new id:
	$self->{ID} = $id;

	# mark as modified
	$self->modified(1);

	# undefine _in_db_ status
	$self->{_in_db_} = undef;
    }

    # return id
    return $self->{ID};
}



# -------------------------------------------------------------------------------------

=item B< $date = $t-E<gt>ExpiryDate( [ $date ] ) >

If C<$date> is passed, sets this ticket's Expiry Date, which must contain a valid
date returned by time().

Returns this ticket's date as a number (see I<sql_date_as_time()>), or undef on error.

=cut

sub ExpiryDate {
    my $self = shift || return;
    my $date = shift;

    # set expiry date if need be;
    if ($date) {
	# hopefully we have date as digits...
	if ($date =~ /^\d+$/) {
	    # good - now convert digits to date string
	    $date = time_as_sql_date($date);
	} else {
	    warn "  !  can't handle non-digit date!\n";
	    return;
	}

	# set expiry date
	$self->{ExpiryDate} = $date;

	# mark as modified
	$self->modified(1);
    }

    # return expiry date as a number
    return sql_date_as_time($self->{ExpiryDate});
}



# -------------------------------------------------------------------------------------

=item B< $date = $t-E<gt>ExpiryDateText() >

Returns this ticket's expirt date as text (see I<sql_date_as_time()>),
or undef on error.

=cut

sub ExpiryDateText {
    my $self = shift || return;

    # return expiry date as a number
    return $self->{ExpiryDate};
}



# -------------------------------------------------------------------------------------

=item B< $email = $t-E<gt>Email( [ $email ] ) >

If C<$email> is passed, sets this ticket's Email, which must contain a valid email
address (eg. a string < 255 chars).

Returns this ticket's email, or undef on error.

=cut

sub Email {
    my $self  = shift || return;
    my $email = shift;

    # set email if need be
    if ($email) {
	# validate length
	if (length($email) > 255) {
	    warn "  !  ticket email [$email] too long - truncating to 255 chars!\n";
	    $email = substr($email, 0, 255);
	}

	# set new email:
	$self->{Email} = $email;

	# mark as modified
	$self->modified(1);
    }

    # return email
    return $self->{Email};
}



# -------------------------------------------------------------------------------------

=item B< $action = $t-E<gt>Action( [ $action ] ) >

If C<$action> is passed, sets this ticket's Action, which must contain a string
indicating the action this ticket is linked to (ie: 'subscribe', 'remove', etc).

Returns this ticket's action, or undef on error.

=cut

sub Action {
  my $self   = shift || return;
  my $action = shift;

  # set action if need be
    if ($action) {
	# validate length
	if (length($action) > 25) {
	    warn "  !  ticket action [$action] too long - truncating to 25 chars!\n";
	    $action = substr($action, 0, 25);
	}

	# set new action:
	$self->{Action} = $action;

	# mark as modified
	$self->modified(1);

	# undefine _in_db_ status
	$self->{_in_db_} = undef;
    }

    # return action
    return $self->{Action};
}



# -------------------------------------------------------------------------------------

=item B< $id = $t-E<gt>generateId() >

Generates and sets a new Ticket Id (a random 25 char string) for this ticket.  This
method does not ensure that the TicketId generated will be unique.

Returns this ticket's id, or undef on error.

=cut

sub generateId {
    my $self   = shift || return;
    my $tid    = "";

    # generate new ticket id
    for (1..25) {
	my $idx = rand(scalar @ticket_id_chars);
	$tid   .= $ticket_id_chars[$idx];
    }
    $self->{ID} = $tid;

    # mark as modified
    $self->modified(1);

    # return ticket id
    return $self->{ID};
}



# -------------------------------------------------------------------------------------

=item B< $bool = $t-E<gt>saveToDb( [ $force ] ) >

Saves this ticket to the database, if marked as modified.  If C<$force> is true, a
save is attempted even if this ticket is not modified.  A save can only occur if
the ticket has it's I<ID> and I<ExpiryDate> set.  A successful save will result in
this ticket being marked as unmodified.

Returns true on successful save, false on unsuccessful/no save, or undef on error.

=cut

sub saveToDb {
    my $self  = shift || return;
    my $force = shift;

    # save ticket?
    if ($force || $self->modified) {

	# check required fields, and *don't* use methods here...
	unless (defined($self->{ID})) {
	    warn "  !  can't save ticket with no ID!\n";
	    return;
	}
	unless (defined($self->{ExpiryDate})) {
	    warn "  !  can't save ticket with no expiry date!\n";
	    return;
	}

	my $ticket_table = table('Tickets') || return;
	my $status;
	# is the ticket already in the DB?
	if ($self->inDb) {
	    # use SQL update, exception thrown on failure
	    $status = $ticket_table->update($self, {'ID' => $self->{ID}} );
	} else {
	    # use SQL insert, exception thrown on failure
	    $status = $ticket_table->insert($self);
	}

	# mark as unmodified & in db
	$self->modified(0);
	$self->inDb(1);

	# check return status
	#return( (defined $status) ? $status : 99 );
	return( 1 );
    }

    # return false
    return 0;
}



# -------------------------------------------------------------------------------------

=item B< $bool = $t-E<gt>deleteFromDb() >

Attempts to delete this ticket from the database.  A save can only occur if the
ticket has it's I<ID> set.  A successful delete will result in this ticket being
marked as modified.

Returns true on successful delete, false on failure, or undef on error.

=cut

sub deleteFromDb {
    my $self  = shift || return;

    # check required fields, and *don't* use methods here...
    unless (defined($self->{ID})) {
	warn "  !  can't delete ticket with no ID!\n";
	return;
    }

    # use SQL delete
    my $ticket_table = table('Tickets') || return;
    my $deleted = $ticket_table->delete('ID' => $self->{ID});

    # mark as modified & not in db
    $self->modified(1);
    $self->inDb(0);

    # check return status
    return( (defined $deleted) ? $deleted : 0 );
}



# -------------------------------------------------------------------------------------

=item B< $bool = $t-E<gt>modified( [ $val ] ) >

Check to see if this ticket has been modified from the version stored in the db.
If C<$val> is passed, sets this ticket's modified status to this value (boolean).

Returns true if C<_modified_> is set, false if not, or undef on error.

=cut

sub modified {
    my $self  = shift || return;
    my $val   = shift;		# don't forget this may be 0!

    # set modified status if need be
    $self->{_modified_} = $val if (defined $val);

    # return modified status
    return $self->{_modified_};
}



# -------------------------------------------------------------------------------------

=item B< $bool = $t-E<gt>inDb( [ $bool ] ) >

Performs a check to see if this ticket is in the database.  If not defined locally,
queries the db.  This value is set for all tickets loaded from the db, and all
tickets saved to the db.  If C<$bool> is passed, the check is not performed, and
this ticket's C<_in_db_> value is set, (though it's not recommended you do this).

Returns true if C<_in_db_> is set, false if not, or undef on error.

=cut

sub inDb {
    my $self  = shift || return;
    my $val   = shift;		# don't forget this may be 0!

    # set in_db status if need be
    $self->{_in_db_} = $val if (defined $val);

    # check if in db if _in_db_ not set...
    unless (defined $self->{_in_db_}) {
	if (defined $self->{ID}) {
	    if (load_ticket_from_db( $self->{ID} )) {
		$self->{_in_db_} = 1;
	    } else {
		$self->{_in_db_} = 0;
	    }
	} else {
	    $self->{_in_db_} = 0;
	}
    }

    # return in_db status
    return $self->{_in_db_};
}



1;
