# -------------------------------------------------------------------------------------
# flo::editor::Headlines
# -------------------------------------------------------------------------------------
# Author : Jean-Michel Hiver <jhiver@mkdoc.com>.
# Copyright : (c) MKDoc Holdings Ltd, 2003
# -------------------------------------------------------------------------------------
package flo::editor::Headlines;
use flo::Record::Editor;
use Text::Unidecode;
use MKDoc::Config;
use flo::Editor;
use flo::Standard;
use MKDoc::CGI;
use MKDoc::Util::Text2HTML;
use strict;
use utf8;

use constant DEFAULT_BASE_PATH     => '/';
use constant DEFAULT_MAX_HEADLINES => 7;


use base qw /flo::Component
	     flo::editor::Mixin::compute_name
	     flo::editor::Mixin::normalize_name/;


sub preferred_extension { 'html' };


sub _initialize
{
    my $self = shift;
    my $args = $self->cgi_args();
    $self->{title}         = $args->{title}           || '';
    $self->{from_path}     = $args->{'from_path'}     || '';
    $self->{max_headlines} = $args->{'max_headlines'} || '';
    $self->{leaf_only}     = $args->{'leaf_only'} || '';
}


sub leaf_only
{
    my $self = shift;
    return ($self->{leaf_only}) ? 'checked' : undef;
}


sub validate
{
    my $self = shift;
    
    # set up the callback for errors
    local $MKDoc::Ouch::CALLBACK;
    $MKDoc::Ouch::CALLBACK = sub { $self->add_error (@_) };
    
    return $self->validate_title() &
           $self->validate_from_path() &
	   $self->validate_max_headlines();
}


sub validate_title
{
    my $self = shift;
    defined $self->{title} and $self->{title} !~ /^\s*$/ or do {
	new MKDoc::Ouch 'component/headlines/title_empty';
	return 0;
    };
    
    return 1;
}


sub validate_from_path
{
    return 1;
}


sub validate_max_headlines
{
    my $self = shift;
    defined $self->{max_headlines} or do {
	new MKDoc::Ouch 'component/headlines/max_headlines_empty';
	return 0;
    };
    
    my $max_headlines = $self->{max_headlines};
    $max_headlines =~ /^[0-9]+$/ or do {
	new MKDoc::Ouch 'component/headlines/max_headlines_malformed';
	return 0;
    };
    
    $self->{max_headlines} =~ s/^0+//;
    $self->{max_headlines} ||= 0;
    return 1;
}


##
# $self->title;
# -------------
#   Returns the title of that component
##
sub title
{
    my $self = shift;
    return $self->{title};
}


##
# $self->set_title ($new_title);
# ------------------------------
#   $new_title - new title to set
##
sub set_title
{
    my $self = shift;
    $self->{title} = shift;
    $self->{title} = join ' ', split /(?:\n|\r)/, $self->{title};
    $self->{title} =~ s/^\s+//;
    $self->{title} =~ s/\s+$//;
}


##
# $self->from_path;
# -----------------
#   Returns the from_path of that component
##
sub from_path
{
    my $self = shift;
    return $self->{from_path};
}


##
# $self->set_from_path ($new_from_path);
# --------------------------------------
#   $new_from_path - new from_path to set
##
sub set_from_path
{
    my $self = shift;
    $self->{from_path} = shift;
    $self->{from_path} = join ' ', split /(?:\n|\r)/, $self->{from_path};
    $self->{from_path} =~ s/^\s+//;
    $self->{from_path} =~ s/\s+$//;
}


sub from_path_search_value
{
    my $self = shift;
    my $path = $self->from_path() || flo::Standard::current_document()->path();

    if ($path =~ /\*/)
    {
        $path =~ s/^\s+//;
        $path =~ s/\s+$//;

        my @path = ($path =~ /\s/) ? (split /\s+/, $path) : ($path);
        @path = map {
            my $path = lc ($_);
            $path =~ s/[^a-z-0-9\*\/\\-]//g;

            # turn all the backslashes into forward slashes
            $path =~ s/\\/\//g;

            # remove the trailing slash if any (we'll re-add one later)
            $path =~ s/\/$//;

            # if there is no starting slash, prefix with current document's path
            $path = flo::Standard::current_document()->path() . $_ unless ($path =~ /^\//);

            # remove starting slash
            $path =~ s/^\///g;

            # turn foo/*/blah/blurb into a MySQL regex
            $path =~ s/\*/[a-z0-9-]+/g;

            # re-add the slashes
            $path = "/$path/";
        } @path;

        my $path = join '|', @path;
        warn $path;
        return "^($path)\$";
    }
    else
    {
        return "^" . quotemeta ($path) . ".+";
    }
}


sub max_headlines
{
    my $self = shift;
    return $self->{max_default_headlines}      || # backwards
           $self->{max_personalized_headlines} || # compatibility
           $self->{max_headlines};
}


sub set_max_headlines
{
    my $self = shift;
    $self->{max_headlines} = shift || '';
    $self->{max_headlines} = join ' ', split /(?:\n|\r)/, $self->{max_default_headlines} || '';
    $self->{max_headlines} =~ s/^(?:\s|\+)+//;
    $self->{max_headlines} =~ s/\s+$//;
    $self->{max_headlines} = DEFAULT_MAX_HEADLINES
        unless ($self->{max_headlines} =~ /^\+?\d+$/);
}


sub max_personalized_headlines
{
    my $self = shift;
    return $self->max_headlines();
}


sub max_default_headlines
{
    my $self = shift;
    return $self->max_headlines();
}


##
# $self->default_headlines;
# -------------------------
# Returns the default headlines, with no personalization as
# the user is not logged in.
##
sub default_headlines
{
    my $self = shift;
    
    # performs the query and put results in the stash
    my $document_t = flo::Standard::table ('Document');
    
    use MKDoc::SQL::Condition;
    my $cond = new MKDoc::SQL::Condition;
    my $doc  = flo::Standard::current_document();
    $cond->add ('Full_Path', 'REGEXP', $self->from_path_search_value());
    
    my $query = $document_t->select ( {
	cols  => '*',
	where => $cond,
	sort  => [ qw /Date_Created/ ],
	desc  => 1,
    } );

    my @res = $query->fetch_all();
    
    # limit to documents which are showable
    @res = map { $_->is_showable() ? $_ : () } @res;
   
    if ($self->leaf_only())
    {
       @res = map { my @children = $_->children_showable(); @children ? () : $_ } @res;
    }
 
    # limit to max number of headlines
    @res = splice @res, 0, $self->max_default_headlines();
    
    \@res;
}


##
# $self->user_logged_in;
# ----------------------
#   Returns TRUE if a user is logged in, FALSE otherwise.
##
sub is_user_logged_in
{
    return flo::Standard::current_user();
}


##
# $self->personalized_headlines;
# ------------------------------
#   Returns the personalized headlines.
#   If the user is not logged in, returns an empty list.
##
sub personalized_headlines
{
    my $self = shift;
    
    my $user = flo::Standard::current_user() || return [];
    
    $self->{_personalized_headlines} ||= do {
	# This is horrible, but I really don't see how to get around it
	# until we get some kind of RSS search engine
	my $sql  = <<SQL;
SELECT DISTINCT Document.*
FROM   Document, Document_Audience, Audience, Preference_Audience, Editor, Preference_Language
WHERE
        -- join the tables together
        (
                Preference_Language.Language_ID = Document.Lang         AND
                Preference_Language.Editor_ID = Editor.ID               AND
                Preference_Audience.Audience_ID = Audience.ID           AND
                Preference_Audience.Editor_ID = Editor.ID               AND
                Document_Audience.Audience_ID = Audience.ID             AND
                Document_Audience.Document_ID = Document.ID
        )
AND
        -- limit to the specified path
        Full_Path REGEXP ?
AND
        -- limit to the current editor
        Editor.Login = ?
AND
        (
                -- get the wanted languages
                Preference_Language.Language_ID = Document.Lang
        OR
                -- get the wanted audiences
                Preference_Audience.Value       = 1
        )
AND
        -- remove languages which are not wanted
        Preference_Language.Value = 1
ORDER BY Date_Created DESC
SQL
	# $sql .= "\nLIMIT 0, " . $self->max_personalized_headlines;
	my $dbh = MKDoc::SQL::DBH->get;
	my $sth = $dbh->prepare_cached ($sql);
	$sth->execute ($self->from_path_search_value(), $user->login);
	my $query = new MKDoc::SQL::Query (sth => $sth, bless_into => 'flo::Record::Document');
	my @res = $query->fetch_all;
	
	# limit to documents which are showable
	@res = map { $_->is_showable() ? $_ : () } @res;
	
        if ($self->leaf_only())
        {
           @res = map { my @children = $_->children_showable(); @children ? () : $_ } @res;
        }
 
	# limit to max number of headlines
	@res = splice @res, 0, $self->max_personalized_headlines();
	
	\@res;
    };
    
    my $res = $self->{_personalized_headlines};
    return wantarray ? @{$res} : $res;
}


1;
