# -------------------------------------------------------------------------------------
# flo::editor::Html
# -------------------------------------------------------------------------------------
# Author : Jean-Michel Hiver <jhiver@mkdoc.com>.
# Copyright : (c) MKDoc Holdings Ltd, 2003
# -------------------------------------------------------------------------------------
package flo::editor::Html;
use flo::Standard;
use flo::Editor;
use MKDoc::Util::Text2HTML;
use MKDoc::Config;
use strict;
use MKDoc::Ouch;
use Petal::Entities;
use File::Spec;
use MKDoc::XML::Tagger::Preserve;


use base qw /flo::Component
	     flo::editor::Mixin::compute_name
	     flo::editor::Mixin::normalize_name/;


sub preferred_extension { 'html' };


##
# $self->new;
# -----------
#   Returns a new flo::editor::Html object based on CGI input
##
sub new
{
    my $class = shift;
    $class    = ref $class || $class;
    my $self  = bless { @_ }, $class;
    
    my $cgi = $self->{cgi};
    my $param_name = $self->{param_name};
    
    if (defined $cgi)
    {
	if (defined $cgi->param ($param_name . "_add"))
	{
	    $self->{data} = "";
	}
	else
	{
	    $self->{data} = $cgi->param ($param_name);
	}
    }
    else
    {
	$self->{data} = "";
    }
    $self->{class} = 'html';
    return $self;
}


##
# $self->generate_xml;
# --------------------
#   Generates the XML for that component
##
sub generate_xml
{
    my $self = shift;
    my $data = $self->{data};
    Petal::Entities::decode_entities ($data);
    return $data;
}


##
# $self->parse_xml;
# -----------------
#   Parses the XML to reconstruct the component
##
sub parse_xml
{
    my $self = shift;
    my $data = shift;
    $self->{data} = $data;
}


sub as_xhtml { return "<div>" . shift()->html() . "</div>" }


##
# $self->html;
# ------------
#   Returns linkified HTML for that component
##
sub html
{
    my $self = shift;
    my $data = $self->{data};

    my @link = flo::Editor->links();
    my @newl = ();
    for (@link) {
        my $new = { %{$_} };
        $new->{_expr} = delete $new->{expr};
        $new->{_tag}  = 'a';
        $new->{title} = delete $new->{desc};
        push @newl, $new;
    }
    
    return MKDoc::XML::Tagger::Preserve->process_data (['a'], $data, @newl);
}


sub to_html { return shift->html }


##
# $self->validate();
# ------------------
# Validates this component. Returns TRUE if the component's data
# integrity is OK, FALSE otherwise.
##
sub validate
{
    my $self = shift;

    # set up the callback for errors
    local $MKDoc::Ouch::CALLBACK;
    $MKDoc::Ouch::CALLBACK = sub { $self->add_error (@_) };
    
    # return 1 if this html component is empty
    return 1 unless ($self->{data});
    
    my $nolinefeed_data = join '', split /\n|\r/, $self->{data};
    return if ($nolinefeed_data =~ /^\s+$/);
    
    my $ret = 1;
    eval {
	my $tidied_data = $self->html_tidy;
	$self->{data} = $tidied_data;
    };
   
    (defined $@ and $@) and do {
        warn $@;
	new MKDoc::Ouch 'component/html/untidy';
	return 0;
    };
    
    return 1;
}


sub sysexec($$$)
{
    my $cmd = shift;
    my $tmp_dir = shift;
    my $in  = shift;
    Encode::_utf8_off ($in);
    
    my $out = undef;
    my $err = undef;
    
    my $tmp = $$ . '.' . time . '.' . int (rand (10000));
    my $tmp_out = $tmp_dir . '/' . $tmp . '.out';
    my $tmp_err = $tmp_dir . '/' . $tmp . '.err';
    
    if (defined $in)
    {
	# executes the command and writes to temp files
	open FP, "|$cmd 2>$tmp_err >$tmp_out" or
            die "Cannot open pipe to $cmd: $!";
	print FP $in;
	close FP;
    }
    else
    {
	# executes the command and writes to temp files
	system ("$cmd 2>$tmp_err >$tmp_out") or
	    die "Cannot exec $cmd: $!";
	print FP $in;
	close FP;
    }
    
    # reads the command standard output
    open (FP, $tmp_out) or
	die "Cannot read-open file $tmp_out: $!";
    $out = join '', <FP>;
    close FP;
    
    # reads the command error output
    open FP, "<$tmp_err" or
        die "Cannot read-epen file $tmp_err: $!";
    $err = join '', <FP>;
    close FP;
    
    # tidy all the mess and returns $out and $err
    unlink $tmp_out;
    unlink $tmp_err;
    return ($out, $err);
}


##
# $obj->html_tidy;
# ----------------
#   Return a clean version of the HTML.
##
sub html_tidy
{
    my $self   = shift;
    my $string = shift || $self->{data};
    
    my $cfg     = new MKDoc::Config ('GLOBAL');
    my $tidy    = MKDoc::Config->TIDY_COMMAND;  
    my $conf    = MKDoc::Config->TIDY_CONFIG;
    my $tmp_dir = File::Spec->tmpdir();
    
    my ($out, $err) = sysexec ("$tidy -config $conf", $tmp_dir, $string);
    Encode::_utf8_on ($out);
    die $err if ($? != 0 && $? != 256);
    
    # strip everything before <BODY> and after </BODY>
    $out =~ s/^.*<BODY[^>]*>\n?//si;
    $out =~ s\</BODY[^>]*>.*</HTML[^>]*>.*$\\si;
    return $out;
}


1;




