# -------------------------------------------------------------------------------------
# flo::editor::Link
# -------------------------------------------------------------------------------------
# Author : Jean-Michel Hiver <jhiver@mkdoc.com>.
# Copyright : (c) MKDoc Holdings Ltd, 2003
# -------------------------------------------------------------------------------------
package flo::editor::Link;
use flo::Editor;
use flo::Standard;
use MKDoc::CGI;
use MKDoc::Util::Text2HTML;
use strict;

use base qw /flo::Component
	     flo::editor::Mixin::compute_name
	     flo::editor::Mixin::normalize_name/;


sub preferred_extension { 'link' }


sub link
{
    return shift;
}


sub _initialize
{
    my $self = shift;
    my $args = $self->cgi_args()      || return;

    $self->{url}         = $args->{'url'}         || 'http://';
    $self->{title}       = $args->{'title'}       || '';
    $self->{description} = $args->{'description'} || '';
}


sub validate
{
    my $self = shift;
    
    # set up the callback for errors
    local $MKDoc::Ouch::CALLBACK;
    $MKDoc::Ouch::CALLBACK = sub { $self->add_error (@_) };
    
    return $self->validate_url()   &
           $self->validate_title() &
	   $self->validate_description();
}


sub validate_url
{
    my $self = shift;
    $self->{url} and $self->{url} ne 'http://' or do {
	new MKDoc::Ouch 'component/link/url_empty';
	return 0;
    };
    
    return 1;
}


sub validate_title
{
    my $self = shift;
    $self->{title} or do {
	new MKDoc::Ouch 'component/link/title_empty';
	return 0;
    };
    
    return 1;
}


sub validate_description
{
    my $self = shift;
    $self->{description} or do {
	new MKDoc::Ouch 'component/link/description_empty';
	return 0;
    };
    
    return 1;
}


sub full_url
{
    my $self = shift;
    my $url  = shift;
    return $url if ($url =~ /^(http|https|mailto|news|gopher):/);
    
    if ($url =~ /^\//)
    {
	return $ENV{SERVER_NAME} . $url;
    }
    else
    {
	return $ENV{SERVER_NAME} . flo::Standard::raw_path_info() . $url;
    }
}


sub url
{
    my $self = shift;
    $self->{url} = shift || return $self->{url};
}


sub url_clean
{
    my $self = shift;
    my $url = $self->url;
    $url =~ s/\&(?!amp;)/&amp;/g;
    return $url;
}


sub title
{
    my $self = shift;
    $self->{title} = shift || return $self->{title};
}


sub description
{
    my $self = shift;
    $self->{description} = shift || return $self->{description};
}


sub parse_xml
{
    my $self = shift;
    my $data = shift || return;
   
    $data =~ /^<a href/ and do {
        # backwards compatibility horribleness
        my ($url, $description, $title) = $data =~ /\<a href=\"(.*?)\" title=\"(.*?)\"\>(.*)\<\/a\>/;
        $self->{url}         = xml2text ($url)          || '';
        $self->{description} = xml2text ($description)  || '';
        $self->{title}       = xml2text ($title)        || '';
        return $self;
    };
    
    return $self->SUPER::parse_xml ($data, @_);
}


sub uri
{
    my $self = shift;
    my $url  = $self->url();
    return $url if ($url =~ /^[A-Za-z]+\:/);
    
    $url =~ /^\// and return do {
	my $cgix = flo::Standard::cgi()->new();
	for ($cgix->param()) { $cgix->delete ($_) }
	$cgix->path_info ($url);
	$cgix->self_url;
    };
    
    return do {
	my $cgix = flo::Standard::cgi()->new();
	my $path = flo::Standard::path_info();
	my $relpath = $url;
	$path = File::Spec->canonpath ($path . $relpath);
	$cgix->path_info ($path);
	$cgix->self_url;
    };
}


1;

