# -------------------------------------------------------------------------------------
# flo::editor::RSS
# -------------------------------------------------------------------------------------
# Author : Jean-Michel Hiver (jhiver@mkdoc.com).
# Copyright : (c) MKDoc Holdings Ltd, 2003
# -------------------------------------------------------------------------------------
package flo::editor::RSS;
use strict;
use warnings;
use MKDoc::HTTP::Response;
use Cache::FileCache;
use XML::RSS;
use Encode;


use base qw /flo::Component/;


use constant ALL_HEADLINES     => 'all';
use constant AS_FOUND          => 'as_found';
use constant DATE_CREATED_ASC  => 'date_created_asc';
use constant DATE_CREATED_DSC  => 'date_created_dsc';
use constant DATE_MODIFIED_ASC => 'date_modified_asc';
use constant DATE_MODIFIED_DSC => 'date_modified_dsc';
use constant ALPHABETICAL_ASC  => 'alphabetical_asc';
use constant ALPHABETICAL_DSC  => 'alphabetical_dsc';


sub validate
{
    my $self = shift;
    
    # set up the callback for errors
    local $MKDoc::Ouch::CALLBACK;
    $MKDoc::Ouch::CALLBACK = sub { $self->add_error (@_) };
    
    return $self->validate_uri()   &
           $self->validate_title() &
	   $self->validate_max()   &
	   $self->validate_template();
}


sub validate_uri
{
    my $self = shift;
    $self->{uri} || do {
	new MKDoc::Ouch 'component/rss/uri_empty';
	return 0;
    };
    
    return 1;
}


sub validate_title
{
    my $self = shift;
    defined $self->{title} and $self->{title} !~ /^\s*$/ or do {
	new MKDoc::Ouch 'component/rss/title_empty';
	return 0;
    };
    
    return 1;
}


sub validate_max
{
    my $self = shift;
    defined $self->{max} and $self->{max} ne '' or do {
	new MKDoc::Ouch 'component/rss/max_empty';
	return 0;
    };
    
    $self->{max} =~ s/^0+//;
    $self->{max} eq '' and do {
	$self->{max} = 0;
	return 1;
    };
    
    defined $self->{max} and ( $self->{max} =~ /^[0-9]+$/ or $self->{max} eq ALL_HEADLINES ) or do {
	new MKDoc::Ouch 'component/rss/max_malformed';
	return 0;
    };
    
    return 1;
}


sub validate_template
{
    my $self = shift;
    $self->{template} || do {
	new MKDoc::Ouch 'component/rss/template_empty';
	return 0;
    };
    
    for ($self->templates()) { $self->template() eq $_->{value} and return 1 }
    
    new MKDoc::Ouch 'component/rss/template_not_found';
    return 0;
}


sub _initialize
{
    my $self = shift;
    my $args = $self->cgi_args()          || return;
    $self->{uri}      = $args->{uri}      || '';
    $self->{title}    = $args->{title}    || '';
    $self->{max}      = $args->{max}      || '';
    $self->{template} = $args->{template} || '';
}


sub max_eq
{
    my $self = shift;
    my $max  = $self->{max};
    my $val  = shift || '';
    ($max eq $val) ? 'selected' : undef;
}


sub sort_eq
{
    my $self = shift;
    my $sort = $self->{sort};
    my $val  = shift || '';
    ($sort eq $val) ? 'selected' : undef;
}


sub template_eq
{
    my $self = shift;
    my $tmpl = $self->{tmpl};
    my $val  = shift || '';
    ($tmpl eq $val) ? 'selected' : undef;
}


sub uri
{
    my $self = shift;
    return $self->{uri};
}


sub set_uri
{
    my $self = shift;
    $self->{uri} = shift;
}


sub title
{
    my $self = shift;
    return $self->{title};
}


sub set_title
{
    my $self = shift;
    $self->{title} = shift;
}


sub max
{
    my $self = shift;
    return $self->{max};
}


sub set_max
{
    my $self = shift;
    $self->{max} = shift;
}


sub sort
{
    my $self = shift;
    return $self->{sort};
}


sub set_sort
{
    my $self = shift;
    $self->{sort} = shift;
}


sub template
{
    my $self = shift;
    return $self->{template};
}


sub set_template
{
    my $self = shift;
    $self->{template} = shift;
}


sub is_503
{
    my $self = shift;
    my $resp = $self->response();
    return $resp->Status() =~ /^503/;
}


sub is_404
{
    my $self = shift;
    my $resp = $self->response();
    return $resp->Status() =~ /^404/;
}


sub items
{
    my $self = shift;

    my $resp = $self->response() || return;
    my $body = $resp->Body()     || return;
    
    my $rss  = new XML::RSS;
    eval { $rss->parse ($body) };
    (defined $@ and $@) and do {
	warn $@;
	return;
    };
    
    my @items = map {
	my $link  = $_->{'link'};
	my $title = $_->{'title'};
	my $desc  = $_->{'description'};
	
	Encode::_utf8_on ($link);
	Encode::_utf8_on ($title);
	Encode::_utf8_on ($desc);
	
	{
	    uri         => $link,
	    title       => $title,
	    description => $desc,
	}
    } @{$rss->{'items'}};
    
    # let's keep a few numbers if necessary
    ($self->max() ne ALL_HEADLINES) and do {
	my $max_index_1 = $self->max() - 1;
	my $max_index_2 = @items - 1;
	my $index = ($max_index_1 > $max_index_2) ? $max_index_2 : $max_index_1;
	@items = splice (@items, 0, $self->max());
    };
    
    return wantarray ? @items : \@items;
}


sub __on_save
{
    my $self  = shift;
    my $uri   = $self->uri() || return;
    my $cache = $self->rss_cache();
    $cache->get ($uri) and return;
    
    my $resp  = new MKDoc::HTTP::Response();
    $resp->Status ('503 Service Unavailable');
    $resp->Content_Type ('application/xml');
    $resp->Retry_After (24 * 3600);
    $resp->X_Retries ('0');
    my $data = $resp->GET();
    
    $cache->set ($uri, $data);
}


sub __on_load
{
    my $self  = shift;
    my $uri   = $self->uri() || return;
    my $cache = $self->rss_timestamp_cache();
    $cache->set ($uri, time());
}


sub response
{
    my $self  = shift;
    my $uri   = $self->uri() || return;
    my $cache = $self->rss_cache();
    my $data  = $cache->get ($uri);
    Encode::_utf8_on ($data);
    
    return new MKDoc::HTTP::Response ($data);
}


sub rss_cache
{
    my $self = shift;
    my %opt  = (
        cache_root => MKDoc::Config->DATA_DIR,
	namespace  => 'rss',
       );
    
    return new Cache::FileCache ( \%opt );
}


sub rss_timestamp_cache
{
    my $self = shift;
    my %opt  = (
        cache_root => MKDoc::Config->DATA_DIR,
	namespace  => 'rss_timestamp',
       );
    
    return new Cache::FileCache ( \%opt );
}


# returns this object as embeddable HTML using Petal
sub as_xhtml
{
    my $self = shift;
    my $type = $self->type();
    my $tmpl = $self->template();
    my $template = new Petal (
	file   => "component/$type/$tmpl",
	lang   => $self->language(),
	input  => 'XML',
	output => 'XHTML'
       );
    
    my $res = $template->process (self => $self);
    Encode::_utf8_on ($res);
    return $res;
}


1;


__END__
