# -------------------------------------------------------------------------------------
# flo::plugin::Account::Details
# -------------------------------------------------------------------------------------
# Author : Steve Purkis <spurkis@mkdoc.com>
# Copyright : (c) 2003, MKDoc Holdings Ltd.
# -------------------------------------------------------------------------------------
package flo::plugin::Account::Details;
use flo::Standard qw /cgi/;
use flo::HTTP::Header;
use flo::Record::Ticket;
use flo::Record::Editor;
use MKDoc::Config;
use Carp;
use Petal;
use Petal::Mail;

use strict;
use 5.008_000;
use utf8;

require flo::plugin::Account::Preferences;

use base qw /flo::Plugin/;


sub template_path { '/account/details' };


sub activate
{
    my $self = shift;
    return unless ($self->SUPER::activate());
    
    my $cgi = flo::Standard::cgi();
    my $id  = $cgi ->param ('t') || return;
    $self->{ticket} = $self->validate_ticket_id ($id) || return;
    
    return 1;
}


sub http_get
{
    my $self = shift;
    my $cgi  = flo::Standard::cgi();
    $self->_initialize_cgi_fields();

    $self->render_http (
	self       => $self,
	object     => $self->parent,
	__input__  => 'XML',
	__output__ => 'XHTML',
       );
    return 'TERMINATE';
}


sub _initialize_cgi_fields
{
    my $self = shift;
    my $cgi  = flo::Standard::cgi();
    $cgi->param ('login') || $cgi->param ('first_name') || $cgi->param ('family_name') && return;
    $self->_initialize_cgi_fields_login();
#    $self->_initialize_cgi_fields_first_name();
#    $self->_initialize_cgi_fields_family_name();
}


sub _initialize_cgi_fields_login
{
    my $self = shift;
    my $cgi  = flo::Standard::cgi();
    my $mail = $self->{ticket}->email();
    $mail    =~ s/\@.+//;
    $mail    = lc ($mail);
    $mail    =~ s/[^a-z]//g;
    $mail  ||= 'login';
    $cgi->param ('login', $self->_initialize_cgi_fields_login_find_free_login ($mail));
}


sub _initialize_cgi_fields_first_name
{
    my $self = shift;
    my $cgi  = flo::Standard::cgi();
    my $mail = $self->{ticket}->email();
    $mail    =~ s/\@.+//;
    $mail    = lc ($mail);
    $mail    =~ s/^[^a-z]//g;
    $mail    =~ s/[^a-z]$//g;
    $mail  ||= '';
    ($mail)  = split /[^a-z]+/, $mail, 2;
    $mail    =~ s/(\w+)/\u\L$1/g;
    $cgi->param ('first_name', $mail);
}


sub _initialize_cgi_fields_family_name
{
    my $self = shift;
    my $cgi  = flo::Standard::cgi();
    my $mail = $self->{ticket}->email();
    my $junk;
    $mail    =~ s/\@.+//;
    $mail    = lc ($mail);
    $mail    =~ s/^[^a-z]//g;
    $mail    =~ s/[^a-z]$//g;
    $mail  ||= '';
    ($junk, $mail)  = split /[^a-z]+/, $mail, 2;
    $mail    =~ s/(\w+)/\u\L$1/g;
    $cgi->param ('family_name', $mail || $cgi->param ('first_name'));
    return $mail;
}


sub _generate_random_password
{
    return join '', map { chr (ord ('a') + int rand 26) } 1..8;
}


sub _initialize_cgi_fields_login_find_free_login
{
    my $self   = shift;
    my $login  = shift || '';
    length ($login) > 12 && do { $login = substr ($login, 0, 12) };

    my $user_t = flo::Standard::table ('Editor');
    $user_t->get (Login => $login) || return $login;
    
    my $count = 2;
    my $new_login = $login . $count;
    while ($user_t->get (Login => $new_login))
    {
	$count++;
	$new_login = $login . $count;
    }
    
    return $new_login;
}


sub http_post
{
    my $self = shift;
    $self->_validate_and_create_account();
    return $self->http_get if ($self->has_errors());
    
    $self->_create_homepage();    
    $self->_send_mail();
    $self->{ticket}->deleteFromDb();

    my $cgix   = flo::Standard::cgi();
    if ($cgix->param ('goto'))
    {
       # hack... JM 2004-01-05
       my $domain = $cgix->param ('goto');
       my $login  = $cgix->param ('login');
       my $passwd = $cgix->param ('password');
       $domain =~ s/:\/\//:\/\/$login:$passwd\@/;
       print $cgix->redirect ($domain);
       return 'TERMINATE';
    }
    else
    {   
        return $self->http_get;
    }
}


sub _send_mail
{
    my $self = shift;

    eval {
        my $mail = new Petal::Mail (
            file => '/account/emails/details',
            language => $self->language(),
        );

        $mail->send (
	    self       => $self,
	    object     => $self->parent(),
	    __input__  => 'XML',
	    __output__ => 'XML',
	   );
    };

    defined $@ and $@ and do {
	warn $@;
	new MKDoc::Ouch 'mail/cannot_create';
	return 0;
    };

    return 1;
}


sub validate_ticket_id
{
    my $class     = shift || return;
    my $ticket_id = shift || return;
    
    # clear expired tickets:
    flo::Record::Ticket::clear_old_tickets_from_db;
    
    # try to load ticket from db
    my $ticket = new flo::Record::Ticket ($ticket_id);
    
    return $ticket;
}


sub _validate_and_create_account
{
    my $self   = shift || return;
    my $ticket = $self->{ticket};
    my $cgix   = flo::Standard::cgi();
    
    ($cgix->param ('password') eq $cgix->param ('password_verify')) or do {
	new MKDoc::Ouch 'user/password/mismatch';
	return 0;
    };
    
    $self->{user} = new flo::Record::Editor (
        Login       => $cgix->param ('login'),
	Password    => $cgix->param ('password'),
	Email       => $ticket->email(),
	First_Name  => $cgix->param ('first_name'),
	Family_Name => $cgix->param ('family_name'),
	Disabled    => 0,
       ) || return;
    
    # OK This is really wierd... For some reason, creating a flo::Record::Editor
    # only in this module screws up unicode first and last name, but not in other
    # places in the code that do the same thing...
    # horrible fix below
    my $first_name  = $cgix->param ('first_name');
    my $family_name = $cgix->param ('family_name');
    my $sql = 'UPDATE Editor SET First_Name = ?, Family_Name = ? WHERE Login = ?';
    my $dbh = MKDoc::SQL::DBH->get();
    my $sth = $dbh->prepare ($sql);
    $sth->execute ($first_name, $family_name, $self->{user}->login());
    
    return 1;
}


sub _create_homepage
{
    my $self = shift;
    $self->{user} || confess "How did I get there?";
    
    my $location = MKDoc::Config->ACCOUNT_HOMEPAGE_LOCATION() || return;
    
    # compute what a munged name might be
    my $munged_name = $self->{user}->login();
    
    # this'll be munged further by MKDoc lower level libs
    $munged_name =~ s/\///g;
    $munged_name =~ s/ /-/g;
    
    # $location is a template which can use time variables as well as login
    my @time = localtime (time());
    my %args = (
	'Y' => 1900 + $time[5],
	'm' => 1    + $time[4],
	'd' =>        $time[3],
	'L' =>        $self->{user}->login(),
	'n' =>        $munged_name,
       );
    
    my %table = (
	'01' => 'january',
	'02' => 'february',
	'03' => 'march',
	'04' => 'april',
	'05' => 'may',
	'06' => 'june',
	'07' => 'july',
	'08' => 'august',
	'09' => 'september',
	'10' => 'october',
	'11' => 'november',
	'12' => 'december'
       );
    $args{'m'} = "0$args{m}" if (length ($args{'m'}) == 1);
    $args{'d'} = "0$args{d}" if (length ($args{'d'}) == 1);
    $args{'B'} = $table{$args{'m'}};
    
    $location =~ s/\%(\w+)/$args{$1}/g;
    $self->_mkdir_p ($location);
}



sub _mkdir_p
{
    my $self     = shift;
    
    my $location = shift;
    $location    =~ s/^\///;
    $location    =~ s/\/$//;
    my @location = split /\//, $location;
    
    for (my $i = 1; $i < @location; $i++)
    {
	my @ancestor = @location;
	@ancestor = splice @ancestor, 0, $i;
	my $path = join "/", @ancestor;
	$self->_create_if_not_exists ("/$path/");
    }
    
    my $body      = undef;
    my $body_file = MKDoc::Config->ACCOUNT_HOMEPAGE_CONTENT();
    if ($body_file)
    {
	open FP, "<$body_file" || die "Cannot read-open $body_file";
	$body = join '', <FP>;
	close FP;
    }
    
    $self->_create_if_not_exists ("/$location/" => $body, $self->{user}->real_name());
    $self->{'home'} = flo::Standard::table ('Document')->get (Full_Path => "/$location/");
    
    # grant editor rights for this document
    my $base_doc_t = flo::Standard::table ('Base_Document');
    $base_doc_t->insert (
	Document_ID => $self->{home}->id(),
	Editor_ID   => $self->{user}->id()
       );
}


sub _create_if_not_exists
{
    my $self   = shift;
    my $path   = shift;
    my $body   = shift;
    my $title  = shift;
    
    my $doc_t  = flo::Standard::table ('Document');
    $doc_t->get ($path) and return;
    
    $path =~ s/^\///;
    $path =~ s/\/$//;
    
    my $parent_path = ();
    my $create_name = undef;
    
    if ($path =~ /\//)
    {
	my @parent_path = split /\//, $path;
	$create_name = pop (@parent_path);
	$parent_path = join '/', @parent_path;
	$parent_path = "/$parent_path/";
    }
    else
    {
	$create_name = $path;
	$parent_path = '/';
    }
    
    my $parent = $doc_t->get (Full_Path => $parent_path) || confess "Cannot retrieve $parent_path";
    $parent->get_child ($create_name) and return; # nothing to do...
    
    # otherwise try to create the document
    new flo::Record::Document (
	Title                     => $title || $create_name,
	Name                      => $create_name,
	Lang                      => $self->language(),
	Keywords                  => $create_name,
	Description               => $create_name,
	Editor_Created_ID         => $self->{user}->id(),
	Editor_Last_Modified_ID   => $self->{user}->id(),
	Parent_ID                 => $parent->id(),
	Template                  => 'default',
	Cache_Control             => 10,
	Body                      => $body,
	Sort_By                   => flo::Record::Document::SORT_POSITION,
	Order_By                  => flo::Record::Document::ORDER_ASC,
	Date_Created              => $self->_mysql_now(),
	Date_Last_Modified        => $self->_mysql_now(),
       ) || do {
	   use Data::Dumper;
	   die Dumper ($self->errors());
       };
}


sub _mysql_now
{
    my $self = shift;
    my ($sec, $min, $hour, $mday, $mon, $year, $wday, $yday, $isdst) = localtime (time);
    $mon++;
    $year += 1900;
    $sec  = "0$sec"  if (length ($sec)  == 1);
    $min  = "0$min"  if (length ($min)  == 1);
    $hour = "0$hour" if (length ($hour) == 1);
    $mday = "0$mday" if (length ($mday) == 1);
    $mon  = "0$mon"  if (length ($mon)  == 1);    
    
    return "$year-$mon-$mday $hour:$min:$sec";
}


sub user
{
    my $self = shift;
    return $self->{user};
}


sub user_domain
{
    my $self = shift;
    return MKDoc::Config->USER_DOMAIN;
}


sub user_domain_notrailingslash
{
    my $self   = shift;
    my $domain = $self->user_domain();
    $domain    =~ s/\/$//;
    
    return $domain;
}


1;
