# -------------------------------------------------------------------------------------
# flo::plugin::Admin::Move
# -------------------------------------------------------------------------------------
# Author : Jean-Michel Hiver <jhiver@mkdoc.com>
# Copyright : (c) MKDoc Holdings Ltd, 2003
#
# Lets an editor delete the current document. Redirects to the parent document.
# -------------------------------------------------------------------------------------
package flo::plugin::Admin::Move;
use MKDoc::Config;
use MKDoc::Error;
use flo::Standard;
use Encode;
use strict;
use warnings;
use 5.008_000;
use Carp;

use base qw /flo::Plugin/;

sub template_path { 'admin/move' }


##
# $self->activate;
# ----------------
# Returns TRUE if this plugin can be activated, FALSE
# otherwise.
##
sub activate
{
    my $self = shift;
    return unless ($self->SUPER::activate (@_));
    my $current_document = flo::Standard::current_document();
    return if ($current_document->is_root());
    return $current_document->Can_Admin;
}


sub http_get
{
    my $self = shift;
    my $doc  = flo::Standard::current_document();
    $self->render_http (
	self       => $self,
	__input__  => 'XML',
	__output__ => 'XHTML',
       );
    
    return 'TERMINATE';
}


sub _validate_defined
{
    my $self = shift;
    my $to = shift;
    defined $to and $to or do {
	new MKDoc::Ouch 'document/move/no_target';
	return 0;
    };
    return 1;
}


sub _validate_exists
{
    my $self = shift;
    my $to   = shift;
    my $document_t = flo::Standard::table ('Document');
    $document_t->get ($to) or do {
	new MKDoc::Ouch 'document/move/no_target';
	return 0;
    };
    return 1;
}


sub _validate_perms
{
    my $self = shift;
    my $to   = shift;
    my $document_t = flo::Standard::table ('Document');
    my $doc  = $document_t->get ($to);
    $doc->Can_Admin() or do {
	new MKDoc::Ouch 'document/move/denied';
	return 0;
    };
    return 1;
}


sub http_post
{
    my $self  = shift;
    my $cgix  = flo::Standard::cgi()->new();
    my $to    = $cgix->param ('to');
    my $doc   = flo::Standard::current_document();
    my $lang  = $doc->language();
    
    $self->_validate_defined ($to) || return $self->http_get();
    $self->_validate_exists  ($to) || return $self->http_get();
    $self->_validate_perms   ($to) || return $self->http_get();

    my $document_t = flo::Standard::table ('Document');
    my $new_parent = $document_t->get ($to);
    $doc->move_to ($new_parent);
    
    $self->has_errors() and return $self->http_get();
    $self->redirect();
    return 'TERMINATE';
}


sub redirect
{    
    my $self = shift;
    my $cgix = flo::Standard::cgi()->new();
    my $path_info = $cgix->path_info();
    ($path_info) = $path_info =~ /^\/(.*)\//;
    $path_info = (defined $path_info) ? "/$path_info/" : '/';
    $cgix->path_info ($path_info);
    $cgix->delete ($cgix->param());
    print $cgix->redirect ($cgix->self_url);
}


sub apply_move
{
    my $self = shift;
    my $to   = shift;
    my $here = flo::Standard::current_document();

    my $document_t = flo::Standard::table ('Document');
    my $there = $document_t->get ($to);
    
    $here->set_parent ($there);
    $document_t->modify ($here);
}


sub movable_document_list
{
    my $self = shift;
    
    use MKDoc::SQL::Condition;
    my $document_t = flo::Standard::table ('Document');
    my $document   = flo::Standard::current_document();
    my $condition  = new MKDoc::SQL::Condition();
    $condition->add ('Full_Path', 'NOT LIKE', $document->path() . '%');
    
    my $query = $document_t->select (
	cols   => [ qw /ID Parent_ID Cache_Control Template Description Keywords
			Date_Created Date_Last_Modified Editor_Created_ID Editor_Last_Modified_ID
			Name Full_Path Title Lang Rights Publisher Source Relation Coverage
			Sibling_Position Sort_By Order_By/ ], 
	sort   => [ 'Full_Path' ],
	desc   => 0,
	where  => $condition,
       );
    
    my @res = ();
    while (my $doc = $query->next())
    {
	push @res, $doc if ($doc->Can_Admin());
    }
    
    return wantarray ? @res : \@res;
}


1;
