# -------------------------------------------------------------------------------------
# MKDoc::Config
# -------------------------------------------------------------------------------------
# Author : Jean-Michel Hiver <jhiver@mkdoc.com>.
# Copyright : (c) Jean-Michel Hiver, 2003.
# -------------------------------------------------------------------------------------
package MKDoc::Config;
use strict;
use warnings;
use Carp;
use File::Spec;


sub sillyness { $Petal::Hash::MODIFIERS };


$Petal::Hash::MODIFIERS->{'config:'} = sub {
    my $hash  = shift;
    my $key   = shift;
    return MKDoc::Config->$key();
};


our $CONFIG     = undef;
our $LISTCONFIG = undef;


sub _reset_globals
{
    $CONFIG = undef;
    $LISTCONFIG = undef;
}


sub parsefile_hashref
{
    my $class = shift;
    
    $LISTCONFIG ||= do {
	if ($ENV{MOD_PERL}) { Apache->push_handlers ("PerlCleanupHandler", \&MKDoc::Config::_reset_globals) };
	{};
    };
    
    my $file = shift;
    $LISTCONFIG->{$file} ||= do {	
	my %hash  = ();
	
	open(FP, "<:utf8", "$file") or die "Cannot read-open $file. Reason: $@!";
	while (<FP>)
	{
	    my ($key, $val) = _parseline ($_);
	    $key ||= next;
	    $val ||= 1;
	    $hash{$key} = $val;
	}
	close FP;
	
	\%hash;
    };
    
    return $LISTCONFIG->{$file};
}


sub _parseline
{
    $_ = shift;
    chomp; chomp; # miam
    s/\#.*//;
    s/^\s*//;
    s/\s*$//;
    return split /\s+/, $_, 2;
}


sub get_env
{
    my $key = shift;
    if ($ENV{MOD_PERL})
    {
	my $r = Apache->request;
	return $ENV{$key} || $r->dir_config ($key);
    }
    else
    {
	return $ENV{$key};
    }
}


sub config_lines
{
    my $class = shift;
    my $file  = shift;
    my @res   = ();
    -e $file || die "$file does not exist";
    
    open FP, "<$file" or die "cannot read-open $file. Reason: $!";
    my @lines = <FP>;
    close FP;

    while ($_ = shift @lines)
    {
	chomp();
	s/^\s*#.*//;
        s/#.*//;
        s/\s+$//;

        if (s/^Include\s+//)
        {
            push @res, $class->config_lines ($_);
        }
        else
        {
	    push @res, $_ if ($_ !~ /^\s*$/);
        }
    }
    return @res;
}


sub MKDOC_DIR     { return get_env ('MKDOC_DIR')        || confess 'MKDOC_DIR is not defined'        }
sub SITE_DIR      { return get_env ('SITE_DIR')         || confess 'SITE_DIR is not defined'         }
sub PUBLIC_DOMAIN { return get_env ('PUBLIC_DOMAIN')    || confess 'PUBLIC_DOMAIN is not defined'    }
sub USER_DOMAIN   { return get_env ('USER_DOMAIN')      || confess 'USER_DOMAIN is not defined'      }
sub PLUGIN_LIST   { return get_env ('MKD__PLUGIN_LIST') || confess 'MKD__PLUGIN_LIST is not defined' }


# Variables you can use:
# %Y - year
# %m - month
# %d - day
# %B - english readable month
# %L - login
# for example: /%Y/%m/%L
# will create 2000/01/bruno
sub ACCOUNT_HOMEPAGE_LOCATION
{
    my $class = shift;
    return get_env ('MKD__ACCOUNT_HOMEPAGE_LOCATION');
}


# Pointer to a file which is default MKDoc content to insert.
# this is optional.
sub ACCOUNT_HOMEPAGE_CONTENT
{
    my $class = shift;
    return get_env ('MKD__ACCOUNT_HOMEPAGE_CONTENT');
}


sub URI_HIDDEN_REGEX
{
    my $class = shift;
    return get_env ('MKD__URI_HIDDEN_REGEX') || qr /\/(hidden|private)-/;
}


sub URI_MODIFIED_REGEX
{
    my $class = shift;
    return get_env ('MKD__URI_MODIFIED_REGEX') || qr/\/modified-\d\d\d\d-\d\d-\d\d-\d\d-\d\d-\d\d\//;
}


sub URI_DELETED_REGEX
{
    my $class = shift;
    return get_env ('MKD__URI_DELETED_REGEX') || qr/\/deleted-\d\d\d\d-\d\d-\d\d-\d\d-\d\d-\d\d\//;
}


sub TEMPLATE_DIR
{
    my $class = shift;
    return get_env ('MKD__TEMPLATE_DIR') || $class->MKDOC_DIR . '/templates';
}


sub TEMPLATE_DIR2
{
    my $class = shift;
    return get_env ('MKD__TEMPLATE_DIR2') || $class->SITE_DIR . '/resources/templates';
}


sub LANGUAGE_LIST
{
    my $class = shift;
    return get_env ('MKD__LANGUAGE_LIST') || $class->MKDOC_DIR . '/conf/languages.conf';
}


sub LANGUAGE_LIST_RTL
{
    my $class = shift;
    return get_env ('MKD__LANGUAGE_LIST_RTL') || $class->MKDOC_DIR . '/conf/languages_rtl.conf';
}



sub XHTML_FILTER_LIST
{
    my $class = shift;
    return get_env ('MKD__XHTML_FILTER_LIST') || $class->MKDOC_DIR . '/conf/xhtml_filter.conf';
}


sub ROBOTS_LIST
{
    my $class = shift;
    return get_env ('MKD__ROBOTS_LIST') || $class->MKDOC_DIR . '/conf/robots.conf';    
}


sub TIDY_COMMAND
{
    my $class = shift;
    return get_env ('MKD__TIDY') || 'tidy';
}


sub TIDY_CONFIG
{
    my $class = shift;
    return get_env ('MKD__TIDY_CONF') || $class->MKDOC_DIR . '/conf/tidy.conf';
}


sub IMAP_SERVER        { return get_env ('MKD__IMAP_SERVER')                                                          }
sub IMAP_USER          { return get_env ('MKD__IMAP_USER')                                                            }
sub IMAP_PASSWORD      { return get_env ('MKD__IMAP_PASSWORD')                                                        }
sub IMAP_PORT          { return get_env ('MKD__IMAP_PORT')                                                            }
sub IMAP_POST_URI_HINT { return get_env ('MKD__IMAP_POST_URI_HINT')     || 'post'                                     }
sub IMAP_VIEW_URI_HINT { return get_env ('MKD__IMAP_POST_URI_HINT')     || 'view'                                     }

sub PHOTO_SMALL_HINT   { return get_env ('MKD__PHOTO_SMALL_HINT')       || 'small'                                    }
sub PHOTO_SCALED_HINT  { return get_env ('MKD__PHOTO_SCALED_HINT')      || 'scaled'                                   }

sub CACHE_DIR          { return get_env ('MKD__GLOBAL_CACHE_DIR') ||
                                get_env ('MKD__CACHE_DIR')        ||
                                MKDoc::Config->SITE_DIR . '/cache' }

sub DATA_DIR          { MKDoc::Config->SITE_DIR . '/data'  }

sub FILE_DISK_PATH     { get_env ('MKD__GLOBAL_FILE_DISK_PATH')         || MKDoc::Config->SITE_DIR . '/static/files'  }
sub IMAGE_DISK_PATH    { get_env ('MKD__GLOBAL_IMAGE_DISK_PATH')        || MKDoc::Config->SITE_DIR . '/static/images' }
sub FILE_HTTP_PATH     { get_env ('MKD__GLOBAL_FILE_HTTP_PATH')         || '/.static/files'                           }
sub IMAGE_HTTP_PATH    { get_env ('MKD__GLOBAL_IMAGE_HTTP_PATH')        || '/.static/images'                          }


##
# $class->new ($prefix);
# ----------------------
#   $prefix - An optional prefix argument, i.e. flo.plugin
##
sub new
{
    $CONFIG ||= do {
	if ($ENV{MOD_PERL}) { Apache->push_handlers ("PerlCleanupHandler", \&MKDoc::Config::_reset_globals) };
	my $config =  { _make_config() };
	$config;
    };
    
    my $class = shift;
    $class = ref $class || $class;
    
    my $self = bless { prefix => shift }, $class;
    return $self;
}


sub _make_config {
    'flo.plugin.Discussion.List.slice_thickness'    => get_env ('MKD__PLUGIN_DISCUSSION_PER_PAGE')  || 50,
    'flo.plugin.Photo.scaled_max_x'                 => get_env ('MKD__PLUGIN_PHOTO_SCALED_X')       || 512,
    'flo.plugin.Photo.scaled_max_y'                 => get_env ('MKD__PLUGIN_PHOTO_SCALED_Y')       || 384,
    'flo.plugin.Photo.thumb_max_x'                  => get_env ('MKD__PLUGIN_PHOTO_THUMB_X')        || 256,
    'flo.plugin.Photo.thumb_max_y'                  => get_env ('MKD__PLUGIN_PHOTO_THUMB_Y')        || 128,
    'flo.plugin.Poll.cookie_name'                   => get_env ('MKD__PLUGIN_POLL_COOKIE')          || 'MKD_Poll_',
    'flo.plugin.Search.per_page'                    => get_env ('MKD__PLUGIN_SEARCH_PER_PAGE')      || 10,
    'flo.plugin.Sitemap.MAX_DEPTH'                  => get_env ('MKD__PLUGIN_SITEMAP_DEPTH')        || 2,
    'User.Preferences.general.newsletter-daily'     => 'newsletter-daily',
    'User.Preferences.general.newsletter-weekly'    => 'newsletter-weekly',
    'User.Preferences.general.newsletter-monthly'   => 'newsletter-monthly',
};


##
# $self->get ($key);
# ------------------
#   $key - value for which you want to fetch the value
#
#   Returns the value matching $key
##
sub get
{
    my $self = shift;    
    my $key  = shift || '';
    $key = $self->{prefix} . '.' . $key if (defined $self->{prefix});
    my $val = $CONFIG->{$key} || return;
    return (ref $val and ref $val eq 'CODE') ? $val->() : $val
}


##
# $class->keys;
# -------------
#   Returns an array of keys for this config file
##
sub keys
{
    my $self   = shift;
    my $prefix = $self->{prefix};
    return map { s/^$prefix\.// ? $_ : () } keys %{$CONFIG};
}


##
# $self->hash;
# ------------
#   Returns a hash computed from the keys() and get()
#   methods. Returns a hash in list context or a hashref
#   in scalar context.
##
sub hash
{
    my $self = shift;
    my %hash = map { $_ => $self->get ($_) } $self->keys;
    
    return wantarray ? %hash : \%hash;
}


1;


__END__
