# -----------------------------------------------------------------------------
# MKDoc::Site::Deploy::DB
# -----------------------------------------------------------------------------
#
#       Author : Jean-Michel Hiver <jhiver@mkdoc.com>
#                Steve Purkis <spurkis@mkdoc.com>
#      Version : $Id: DB.pm,v 1.1.2.9 2004/01/02 14:15:18 jhiver Exp $
#    Copyright : (c) MKDoc Holdings Ltd, 2001
#
#      Unauthorized modification, use, reuse, distribution or redistribution
#      of this module is stricly forbidden
#
#    Description:
#
#      Tools to setup a database for an MKDoc Site given config information
#      (usually from setup.cfg, but any old config hash will do).
#
#    Synopsis:
#
#      use MKDoc::Site::Deploy::DB;
#
#      my $dbs = new MKDoc::Site::Deploy::DB;
#
#      $dbs->set_config      ( \%config )
#          ->set_site_dir    ( '/path/to/mkdoc-site' )
#          ->set_interactive ( 0 )
#          ->set_verbose     ( 0 );
#
#      $dbs->setup() || die "$@";
#
# -----------------------------------------------------------------------------
package MKDoc::Site::Deploy::DB;

use strict;
use warnings;

use File::Spec;
use File::Path;

use MKDoc::Site::Deploy::DB::Driver;
use MKDoc::Site::Deploy::DB::Schema;
use Carp;

use base qw/MKDoc::Site::Deploy/;


sub class
{
    my $class = shift;
    return ref $class || $class;
}


sub new
{
    my $class = shift->class;
    return bless { @_ }, $class;
}


##
# CLASS VARIABLES
# ===============
##

## Version (from CVS revision string).
our $VERSION = (split(/ /, '$Revision: 1.1.2.9 $'))[1];



##
# CONSTRUCTOR
# ===========
#
# $obj = new MKDoc::Site::Deploy::DB( %args )
# -------------------------------------------
# Inherited from MKDoc::Site::Deploy.
##


##
# Class Methods
# =============
##

##
# $class->setup();
# ----------------
# Instanciates a MKDoc::Site::DB object, configures it
# and runs it so that the database schema is deployed
# properly.
#
# Changed setup() to be a class method.
# JM. 2002.12.13
##
sub setup
{
    my $class = shift->class;
    my $self = $class->new();
    
    my $config = eval {
	use MKDoc::Site::Config;
	new MKDoc::Site::Config;
    };
    
    $self->set_site_dir ( $config->{SITE_DIR} );
    $self->set_config ( $config );
    $self->set_interactive (1);
    
    $self->say ("Setting up MKDoc Site database... ");
    $self->setup_filesystem()    || return;
    $self->setup_db_connection() || return;
    $self->setup_db_tables()     || return;
    $self->say ("Database setup complete.");
}


##
# Instance Methods
# =============================================================================
#
# Accessors & Instance Variables
# -----------------------------------------------------------------------------
# Inherited from MKDoc::Site::Deploy.
##


##
# File-system related
# -----------------------------------------------------------------------------
##

##
# $path = $obj->su_dir
# --------------------
# Get the path to the 'su' directory (under the mkdoc site dir),
# where database definition files are stored.
##
sub su_dir
{
    my $self = shift;
    confess ("no mkdoc site dir specified!") unless $self->site_dir;
    return File::Spec->catdir ($self->site_dir, 'su');
}


##
# $path = $obj->driver_pl
# -----------------------
# Get the path to the 'driver.pl' file (under the 'su' dir), that
# holds the database driver perl script.  See generate_db_driver().
##
sub driver_pl
{
    my $self = shift;
    return File::Spec->catfile($self->su_dir, 'driver.pl');
}



##
# Setup methods
# -----------------------------------------------------------------------------
##


##
# $obj = $obj->setup_filesystem()
# -------------------------------
# ATM, this just makes sure the 'su' dir exists.
#
# Returns this object or undef on error.
##
sub setup_filesystem
{
    my $self = shift;

    $self->say("Setting up filesystem:");
    $self->check_su_dir() || return;

    return $self;
}


##
# $obj = $obj->setup_db_connection()
# ----------------------------------
# Generates the db driver for this site, and creates database definitions
# in the 'su' directory.
#
# Returns this object or undef on error.
##
sub setup_db_connection
{
    my $self = shift;

    $self->say("Setting up database connection layer:");
    $self->generate_and_save_driver() || return;
    $self->load_db_driver()           || return;
    $self->save_table_definitions()   || return;

    return $self;
}


##
# $obj = $obj->setup_db_tables
# ----------------------------
# Creates required tables in the database, unless existing tables
# should be preserved.
#
# Returns this object, or undef on error.
##
sub setup_db_tables
{
    my $self = shift;

    $self->say("Setting up database tables:");

    $self->check_existing_tables()        || return;
    my $tables = $self->existing_tables() || return;

    if (@$tables)
    {
	$self->say("  !  Warning: if this is an upgrade, some required\n",
		   "     tables may not have been defined.  I don't know\n",
		   "     how to upgrade database structures yet, so you\n",
		   "     must check this manually.\n");
    }
    else
    {
	$self->create_tables()   || return;
	$self->populate_tables() || return;
    }

    return $self;
}


##
# $obj = $obj->_check_su_dir()
# ----------------------------
# Checks to see if the 'su' directory exists.  If not, asks the user
# about creating it (if in interactive mode).
#
# Returns this object, or undef on error.
##
sub check_su_dir
{
    my $self   = shift;
    my $su_dir = $self->su_dir() || return;
    
    $self->say("  +  checking 'su' directory [$su_dir]");
    unless (-d $su_dir)
    {
	$self->say("\t  !  Warning: [$su_dir] does not exist");
	if ($self->should_we_create_su_dir)
	{
	    $self->say("\t  +  creating [$su_dir] dir");
	    eval { mkpath($su_dir, 0, 0755) };
	    confess "couldn't create [$su_dir] - $@" if ($@);
	}
	else
	{
	    $self->say("\t  !  not creating [$su_dir] dir");
	    confess ("su dir [$su_dir] does not exist!");
	}
    }
    
    return $self;
}


##
# $obj = $obj->load_db_driver()
# -----------------------------
# Loads the db driver for this site.
# Returns this object or undef on error.
##
sub load_db_driver
{
    my $self      = shift;
    my $driver_pl = $self->driver_pl;
    
    $self->say("  +  loading driver\n");
    
    eval "require '$driver_pl';";
    
    if ($@)
    {
	$self->say("  !  Driver Error: $@");
	return $self->_error("Driver Error: $@");
    }

    return $self;
}


##
# $obj = $obj->save_table_definitions()
# -------------------------------------
# Save db table definitions defined in MKDoc::Site::Deploy::DB::Schema in
# the 'su' directory.
#
# Returns this object or undef on error.
##
sub save_table_definitions
{
    my $self   = shift;
    my $su_dir = $self->su_dir;

    $self->say("  +  saving table definitions in $su_dir");
    lib::sql::Table->save_state($su_dir);

    return $self;
}


##
# $obj = $obj->generate_and_save_driver()
# ---------------------------------------
# Creates driver with generate_db_driver(), and saves it to the file
# specified by this object's driver_pl.
#
# Returns this object or undef on error.
##
sub generate_and_save_driver
{
    my $self = shift;
    
    $self->say("  +  generating database driver\n");
    my $driver_pl   = $self->driver_pl()                       || return;
    my $driver      = new MKDoc::Site::Deploy::DB::Driver;
    my $driver_code = $driver->generate_driver($self->config)  || return;
    $self->save_file($driver_pl, $driver_code)                 || return;

    return $self;
}


##
# $bool = $obj->check_existing_tables()
# -------------------------------------
# Checks database for existing tables, and erases them if we should
# erase them.
#
# Returns this object, or undef on error.
##
sub check_existing_tables
{
    my $self = shift;

    $self->say("  +  checking for existing tables:");

    my $tables = $self->existing_tables() || return;

    if (@$tables) {
	$self->say("\t  !  Warning: old tables exist!");

	if ($self->should_we_keep_tables)
	{
	    $self->say("\t  +  keeping existing tables");
	}
	else
	{
	    $self->drop_tables(@$tables);
	}
    }
    else
    {
	$self->say("\t  +  no tables found\n");
    }

    return $self;
}


##
# @tables = $obj->existing_tables()
# ---------------------------------
# Checks database for existing tables and returns them.
##
sub existing_tables
{
    my $self = shift;

    my $dbh    = lib::sql::DBH->get;
    my $tables = $dbh->selectcol_arrayref('SHOW TABLES');

    return $tables;
}


##
# $obj = $obj->drop_tables( @tables )
# -----------------------------------
# Drops @tables from the database.  Assumes they exist.
##
sub drop_tables
{
    my $self   = shift;
    my @tables = @_;

    $self->say("\t  +  dropping existing tables:");

    my $dbh = lib::sql::DBH->get;
    foreach my $table_name (@tables)
    {
	$self->say("\t\t  -  $table_name");
	$dbh->do("DROP TABLE $table_name");
    }

    return $self;
}


##
# $obj = $obj->create_tables
# ---------------------------
# Creates required tables in the database.
# Returns this object, or undef on error.
##
sub create_tables
{
    my $self = shift;

    $self->say("  +  creating tables");
    lib::sql::Table->create_all;    # tries to create all the tables

    return $self;
}


##
# populate_tables($config)
# ------------------------
# Creates 'admin' + config-defined users, and root document
# owned by config-defined user.
# TODO: check config parameters
##
sub populate_tables
{
    my $self   = shift;
    my $config = $self->config;

    $self->say("  +  populating tables from config:");

    my $editor_table        = lib::sql::Table->table('Editor');
    my $document_table      = lib::sql::Table->table('Document');
    my $base_document_table = lib::sql::Table->table('Base_Document');
    
    my $amount = int (rand (2)) + 3;
    my @rand = qw /a i u e o
		   ka ki ku ke ko
		   sa shi su se so
		   ta tchi tsu te to
		   na ni nu ne no
		   ha hi fu he ho
		   ma mi mu me mo
		   ya yu yo
		   la li lu le lo
		   wa wo/;
    
    $::ADMIN_PASSWD = join '', map { $rand[int rand (scalar @rand)] } 1..$amount;
    
    $self->say("\t  +  adding 'admin' user\n");
    $editor_table
      ->insert(
	       Login       => 'admin',
	       Password    => $::ADMIN_PASSWD,
	       Email       => $config->{ORGANIZATION_EMAIL},
	       First_Name  => '',
	       Family_Name => $config->{ORGANIZATION_NAME},
	       Disabled    => 0,
	      );
    
    $self->say("\t  +  adding '$config->{USER_LOGIN}' user\n");
    $editor_table
      ->insert(
	       Login       => $config->{USER_LOGIN},
	       Password    => $config->{USER_PASS},
	       Email       => $config->{USER_EMAIL},
	       First_Name  => $config->{USER_FIRST_NAME},
	       Family_Name => $config->{USER_LAST_NAME},
	       Disabled    => 0,
	      );
    # assume auto-inc ID's, we inserted 2 editors:
    my $editor_id = 2;

    $self->say("\t  +  creating root document owned by $config->{USER_LOGIN}\n");
    $document_table
      ->insert(
	       Template                => 'default',
	       Cache_Control           => 10,    # what's this mean?
	       Description             => 'enter a description here',
	       Keywords                => 'enter keywords here',
	       Date_Created            => current_date(),
	       Date_Last_Modified      => current_date(),
	       Editor_Created_ID       => $editor_id,
	       Editor_Last_Modified_ID => $editor_id,
	       Name                    => '',
	       Title                   => 'Root document',
	       Lang                    => 'en',
	       Sort_By                 => 'Title',
	       Order_By                => 0,     # what's this mean?
	      );
    # assume auto-inc ID's, we inserted 1 doc:
    my $doc_id = 1;

    $self->say("\t  +  setting $config->{USER_LOGIN}'s base document\n");
    $base_document_table
      ->insert(
	       Editor_ID   => $editor_id,
	       Document_ID => $doc_id,
	      );

    return 1;
}


##
# User interaction methods
# -----------------------------------------------------------------------------
#
# Note: these could potentially go into an 'AskUser' class of their own.
#
##

##
# $bool = $obj->should_we_create_su_dir()
# ---------------------------------------
# Asks the user if they want to create the 'su' directory.
# (if interactive mode is on).
#
# Returns true if we should, false if not.
##
sub should_we_create_su_dir
{
    my $self   = shift;
    my $su_dir = $self->su_dir;

    my $q = ("The 'su' directory [$su_dir] where I store all the db\n" .
	     "configuration does not exist. Do you want me to create it?");

    my $ans = $self->ask_user($q, 'Y', 'n');

    if (defined($ans) and $ans eq 'N') { return 0; }
    else                               { return 1; }
}


##
# $bool = $obj->should_we_keep_tables()
# -------------------------------------
# Asks the user if they want to keep or erase existing db tables
# (if interactive mode is on).
#
# Returns true if tables should be kept, false if not.
##
sub should_we_keep_tables
{
    my $self = shift;

    my $q = ("It appears that your database is not empty.\n" .
	     "Do you want to (K)eep your current database or (E)rase it?");

    my $ans = $self->ask_user($q, 'K', 'e');

    if (defined($ans) and $ans eq 'K') { $::KEEP_TABLES = 1; return 1; }
    else                               { $::KEEP_TABLES = 0; return 0; }
}


##
# Package Subroutines
# =============================================================================
#
# save_file($file, $data)
# -----------------------
# Save $data to $file, dies on error.
#
sub save_file
{
    my $self = shift;
    my $file = shift;
    my $data = shift;

    $self->say("  +  saving $file\n");
    open (DRIVER, ">$file") || return $self->_error("  !  Error writing to $file: $!\n");
    print DRIVER ($data)    || return $self->_error("  !  Error writing to $file: $!\n");
    close DRIVER            || return $self->_error("  !  Error writing to $file: $!\n");

    return 1;
}


##
# $date = current_date;
# ---------------------
#   Returns the current date in MySQL format
#   TODO: this belongs in an sql library
##
sub current_date
{
    # Prefill date fields with current date
    my ($sec, $min, $hour, $mday, $mon, $year) = localtime (time);
    $mon++;
    $year += 1900;

    if (length ($mon)  == 1) { $mon  = "0" . $mon  }
    if (length ($mday) == 1) { $mday = "0" . $mday }

    return "$year-$mon-$mday $hour:$min:$sec";
}


1;

__END__
