# -------------------------------------------------------------------------------------
# flo::Component
# -------------------------------------------------------------------------------------
# Author : Jean-Michel Hiver <jhiver@mkdoc.com>
# Copyright : (c) MKDoc Holdings Ltd, 2002
# 
# Discussion content component. Should let the user add hyperlinks to pages
# -------------------------------------------------------------------------------------
package flo::Component;
use MKDoc::XML::Dumper;
use Unicode::Normalize;
use flo::Standard;
use Encode;
use strict;
use Carp;


##
# $self->clone();
# ---------------
# Returns a shallow clone of this component.
##
sub clone
{
    my $self = shift;
    my $other = bless { %{$self} } => ref $self;
    return $other;
}


##
# $self->copy_to ($document);
# ---------------------------
# Copies $self to $document. Saves $document.
##
sub copy_to
{
    my $self   = shift;
    my $target = shift;
    $target->add_component ($self);
    $target->save();
}


##
# $self->move_to ($document);
# ---------------------------
# Moves $self to $document. Saves $document
# as well as the $document which $self was into.
##
sub move_to
{
    my $self = shift;
    my $target = shift;
    $self->copy_to ($target);
    
    my $parent = $self->parent();
    $parent->del_component ($self);
    $parent->save();
}


##
# $self->validate();
# ------------------
# Returns TRUE if this component validates, FALSE otherwise.
# This method is to be overriden, it always returns TRUE by default.
##
sub validate
{
    return 1;
}


##
# $self->has_errors();
# --------------------
# Returns TRUE if the current plugin object holds error that needs
# to be displayed, false otherwise.
##
sub has_errors
{
    my $self = shift;
    my $errors = $self->errors;
    return scalar @{$errors};
}


##
# $self->errors();
# ----------------
# Returns a list of errors.
##
sub errors
{
    my $self = shift;
    return (defined $self->{'.errors'}) ?
        $self->{'.errors'} :
	[];
}


##
# $self->add_error (@_);
# ----------------------
# Adds all the errors in @_ to the current plugin object.
##
sub add_error
{
    my $self = shift;
    $self->{'.errors'} ||= [];
    push @{$self->{'.errors'}}, @_;
}


##
# $self->template();
# ------------------
# Returns the name of the template associated with this object.
##
sub template
{
    my $self = shift;
    return $self->{template};
}


##
# $self->set_template ($tmpl);
# ----------------------------
# Sets the name of the template associated with this object.
##
sub set_template
{
    my $self = shift;
    return $self->{template};
}


##
# $self->templates;
# -----------------
# Returns a structure which can be used within a template to choose
# a document template from a select box.
#
# See MKDoc::Util::TemplateFinder::list_structure()
##
sub templates
{
    my $self = shift;
    my $type = $self->type();
    my $tmpl = shift || $self->template() || 'default';
    my @res  = MKDoc::Util::TemplateFinder::list_structure ("component/$type", $tmpl);
    return wantarray ? @res : \@res;
}


##
# $self->selected_template;
# -------------------------
# Returns the template which is currently selected.
# See MKDoc::Util::TemplateFinder::list_structure_selected()
##
sub selected_template
{
    my $self = shift;
    my $selected = MKDoc::Util::TemplateFinder::list_structure_selected ($self->templates());
    return $selected;
}


##
# $self->unselected_templates;
# ----------------------------
# Returns the templates which are not currently selected.
#
# See MKDoc::Util::TemplateFinder::list_structure_unselected()
##
sub unselected_templates
{
    my $self = shift;
    return MKDoc::Util::TemplateFinder::list_structure_unselected ($self->templates());
}


sub new
{
    my $class = shift;
    $class    = ref $class || $class;
    my $self  = bless { @_ }, $class;
    $self->_initialize();
    return $self;
}


sub cgi_args
{
    my $self   = shift;
    my $cgi    = $self->{cgi} || return;
    my $prefix = $self->{param_name} . '_';
    
    my %hash = map {
	my $param = $_;
	my $arg   = $param;
	($arg =~ s/^$prefix//) ? ($arg => $cgi->param ($param)) : ();
    } $cgi->param();
    
    return wantarray ? %hash : \%hash;
}


sub link
{
    my $self = shift;
    my @links = flo::Editor::_ETERNAL_()->access ( class => [ 'link' ] );
    
    my $self_title_lc = lc ($self->title());
    $self_title_lc =~ s/\s+/ /g;
    $self_title_lc =~ s/^\s+//;
    $self_title_lc =~ s/\s+$//;
    foreach my $link (@links)
    {
	my $link_title_lc = lc ($link->title());
	$link_title_lc =~ s/\s+/ /g;
	$link_title_lc =~ s/^\s+//;
	$link_title_lc =~ s/\s+$//;
	return $link if ($self_title_lc eq $link_title_lc);
    }
    
    return;
}


sub is_saved
{
    my $self = shift;
    return $self->{is_saved};
}


sub activate
{
    my $self  = shift;
    my $class = ref $self || $self;
    $class =~ s/^.*:://;
    $class = uc ($class);
    defined $ENV{"MKD__DISABLE_$class"} and return;
    return 1;
}


# Returns a string that helps identifying the type of the
# component.
sub type
{
    my $self = shift;
    my $class = ref $self || $self;
    $class =~ s/^.*\:\://;
    return lc ($class);
}


##
# $self->edit;
# ------------
# Return the edit HTML chunk for this component
##
sub edit
{
    my $self = shift;
    my $type = $self->type();
    my $tmpl = new Petal ( file => "editor/$type", lang => $self->language() );
    my $res  = $tmpl->process ( self => $self );
    Encode::_utf8_on ($res);
    return $res;
}


##
# $self->block_name();
# --------------------
# Used for the templates that are used to edit this component.
##
sub block_name
{
    my $self = shift;
    return $self->{param_name};
}


##
# $self->set_block_name ($new_block_name);
# ----------------------------------------
# Used by flo::Editor to prefix this component fields with the
# proper prefix.
##
sub set_block_name
{
    my $self = shift;
    $self->{param_name} = shift;
}


##
# $self->id;
# ----------
#   Returns the identifier of the current component, which should
#   be the identifier of the document followed by a dash, followed
#   be the position of the component in the page. I.e. 1-3
##
sub id
{
    my $self = shift;
    my $parent_id = $self->parent->id;
    my $position  = $self->position;
    return "$parent_id-$position";
}


##
# $self->parent_id;
# -----------------
#   Returns the ID of the parent document which holds this component.
##
sub parent_id
{
    my $self = shift;
    return $self->{parent_id} || flo::Standard::current_document()->id();
}


##
# $self->parent;
# --------------
#   Returns the parent document which holds this component.
##
sub parent
{
    my $self = shift;
    my $document_table = flo::Standard::table ('Document');
    return $document_table->get ($self->parent_id);
}


##
# $self->equals ($component);
# ---------------------------
#   $component - component to compare $self with.
#
#   Returns TRUE if $self and $component have the same id,
#   FALSE otherwise.
##
sub equals
{
    my $self = shift;
    my $component = shift;
    return $self->id eq $component->id;
}


##
# $self->ancestors;
# -----------------
#   Returns a list of ancestors for this component.
##
sub ancestors
{
    my $self = shift;
    my $parent = $self->parent;
    return (wantarray) ? () : [] unless (defined $parent);
    
    my @res = ($parent->ancestors, $parent);
    return (wantarray) ? @res : \@res;
}


##
# $self->position;
# ----------------
#   Returns the position of this component comparatively with
#   its siblings.
##
sub position
{
    my $self = shift;
    return $self->{position};
}


##
# $self->set_position ($pos);
# ---------------------------
#   $pos - new position to set
#
#   Sets the position of this component comparatively with
#   its siblings.
##
sub set_position
{
    my $self = shift;
    $self->{position} = shift;
}


##
# $self->path;
# ------------
#   Returns the path_info of this component.
##
sub path
{
    my $self = shift;
    return $self->parent->path . $self->name;
}


##
# $self->title;
# -------------
#   Returns the title of this component.
##
sub title
{
    my $self = shift;
    exists $self->{'title'} and return $self->{'title'};
    return $self->name;
}


##
# $self->root;
# ------------
#   Returns the document root.
##
sub root
{
    my $self = shift;
    return $self->parent->root;
}


##
# $self->is_only_child;
# ---------------------
#   Returns TRUE if this component has no sibling components,
#   FALSE otherwise.
##
sub is_only_child
{
    my $self = shift;
    my @components = $self->parent->components;
    return @components > 1;
}


##
# $self->extension;
# -----------------
# Returns the extension to use for the http name of
# this component. Defaults to 'html'.
##
sub extension { return 'html' }


##
# $self->extension_is ($stuff);
# -----------------------------
# Returns TRUE if this component's extension is $stuff,
# FALSE otherwise.
##
sub extension_is
{
    my $self = shift;
    my $ext  = shift || '';
    return (lc ($self->extension()) eq lc ($ext)) or
           ($self->extension() =~ /\.\Q$ext\E$/);
}


##
# $self->uri();
# -------------
# This returns the URI for the public address of this object.
# This address is not necessary on the same domain (i.e. link components).
#
# By default it's an alias to mkdoc_uri() however this can be overriden.
##
sub uri
{
    my $self = shift;
    return $self->mkdoc_uri (@_);
}


##
# $self->mkdoc_uri();
# -------------------
# This returns the URI for the private address of the object.
# This address is necessary on the same domain and represents the location
# of the object within the MKDoc system.
##
sub mkdoc_uri
{
    my $self = shift;
    my $cgix = flo::Standard::cgi()->new;
    $cgix->delete ($_) for ($cgix->param);
    $cgix->path_info ( $self->path );
    while (@_)
    {
	my ($key, $val) = (shift, shift);
	$cgix->param ($key, $val);
    }
    return $cgix->self_url;
}


##
# $self->name;
# ------------
#   Returns the name associated with this component,
#   with the extension.
##
sub name
{
    my $self = shift;
    return $self->{uri_name};
}


##
# $self->generate_xml;
# --------------------
#   Serializes the object to XML
##
sub generate_xml
{
    my $self  = shift;
    $self->__on_save() if ($self->can ('__on_save'));
    
    my $cgi   = delete $self->{cgi};
    my $param = delete $self->{param_name};

    my $xml = MKDoc::XML::Dumper->perl2xml ($self);

    $self->{cgi} = $cgi;
    $self->{param_name} = $param;
    
    $xml =~ s/^<.*?<perl>/<perl>/sm;
    $xml = NFC ($xml);
    return $xml;
}


##
# $self->parse_xml;
# -----------------
#   Parses the XML and sets the object attributes from the
#   parsed data
##
sub parse_xml
{
    my $self = shift;
    eval {
        my $xml  = shift;
        my $obj  = MKDoc::XML::Dumper->xml2perl ($xml);
        $self->{$_} = $obj->{$_} for (keys %{$obj});
        $self->__on_load() if ($self->can ('__on_load'));
    };
    # $@ and warn $@;
    # Carp::cluck ($@);
}


##                                                               #
#  ALL THESE DEFAULT TO THE PARENT DOCUMENT METHODS, BUT SHOULD  #
#  BE PROPERLY RE-IMPLEMENTED LATER ON                           #
#                                                               ##

sub contributors
{
    my $self = shift;
    return $self->parent->contributors;
}


sub editor_created_id
{
    my $self = shift;
    return $self->parent->editor_created_id;
}


sub editor_created
{
    my $self = shift;
    return $self->parent->editor_created;
}


sub editor_last_modified_id
{
    my $self = shift;
    return $self->parent->editor_last_modified_id;
}


sub editor_last_modified
{
    my $self = shift;
    return $self->parent->editor_last_modified;
}


sub date_created
{
    my $self = shift;
    return $self->parent->date_created;
}


sub date_created_w3c
{
    my $self = shift;
    return $self->parent->date_created_w3c;
}


sub date_last_modified
{
    my $self = shift;
    return $self->parent->date_last_modified;
}


sub date_last_modified_w3c
{
    my $self = shift;
    return $self->parent->date_last_modified_w3c;
}


sub align
{
    my $self = shift;
    return $self->parent->align;
}


sub align_opposite
{
    my $self = shift;
    return $self->parent->align_opposite;
}


sub direction
{
    my $self = shift;
    return $self->parent->direction;
}


# returns this object as embeddable HTML using Petal
sub as_xhtml
{
    my $self = shift;
    my $type = $self->type();
    my $template = new Petal (
	file   => "component/$type",
	lang   => $self->language(),
	input  => 'XML',
	output => 'XHTML'
       );

    my $res = $template->process (self => $self);
    Encode::_utf8_on ($res);
    return $res;
}


sub language
{
    my $self = shift;
    return $self->parent()->language();
}


sub lang
{
    my $self = shift;
    return $self->language();
}


sub user
{
    return flo::Standard::current_user();
}


##                              #
#  HORRIBLE DEPRECATED METHODS  #
#                              ##

sub html    { return "HTML METHOD IS DEPRECATED" }
sub to_html { return shift->html                 }


sub Align
{
    my $self = shift;
    carp ref $self . "::Align is deprecated";
    return $self->align;
}


sub Direction
{
    my $self = shift;
    carp ref $self . "::Direction is deprecated";
    return $self->direction;
}


1;
