# -------------------------------------------------------------------------------------
# flo::Editor
# -------------------------------------------------------------------------------------
# Author      : Jean-Michel Hiver (jhiver@mkdoc.com).
# Copyright   : (c) MKDoc Holdings Ltd, 2001
# Description : A class that is used to manage a pluggable component system for MKDoc
# -------------------------------------------------------------------------------------
package flo::Editor;
use MKDoc::Config;
use flo::Standard qw /general_config cgi path_info raw_path_info table template current_document/;
use MKDoc::CGI;
use fields;
use strict;
use 5.008_000;
use utf8;

use constant BLOCK => 'edit_block';


our $IMP = {};
our $MAP = undef;
our $PAM = undef;

sub _reset_globals
{
    $flo::Editor::MAP = undef;
    $flo::Editor::PAM = undef;
}


$::MKD_flo_Editor_LINKS = [];
sub _LINKS_()       { return $::MKD_flo_Editor_LINKS      }
sub _PUSH_LINKS_(@) { push @{$::MKD_flo_Editor_LINKS}, @_ }
sub _SET_LINKS_($)  { $::MKD_flo_Editor_LINKS = shift     }


$::MKD_flo_Editor_ETERNAL = undef;
sub _ETERNAL_()      { return $::MKD_flo_Editor_ETERNAL   }
sub _SET_ETERNAL_($) {
    $::MKD_flo_Editor_ETERNAL = shift;
}


sub _MAP_()
{
    $MAP ||= do {
	$ENV{MOD_PERL} and Apache->push_handlers ("PerlCleanupHandler", \&flo::Editor::_reset_globals);
	_construct_component_list();
    };
    
    return $MAP;
}


sub _construct_component_list
{
    _construct_component_list_import();
    
    my $res = {};
    foreach my $module (sort keys %{$IMP})
    {
	unless ($module->can ('activate') and not $module->activate())
	{
	    my $type = $module->type();
	    $res->{$type} = $module;
	}
    }
    
    return $res;
}


sub _construct_component_list_import
{
    return if (scalar keys %{$IMP});
    
    foreach my $include_dir (@INC)
    {
	my $dir = "$include_dir/flo/editor";
	if (-e $dir and -d $dir)
	{   
	    opendir DD, $dir or do {
		warn "Cannot open directory $dir. Reason: $!";
		next;
	    };
	    
	    my @modules = map { s/\.pm$//; $_ }
	                  grep /\.pm$/,
			  grep !/^\./,
			  readdir (DD);
	    
	    closedir DD;
	    
	    foreach my $module (@modules)
	    {
                $module =~ /^(\w+)$/;
                $module = $1;
		eval "use flo::editor::$module";
		$@ and warn "Cannot import module $module. Reason: $@";
		$IMP->{"flo::editor::$module"} = 1;
	    }
	}
    }
}


sub _PAM_()
{
    my $map = _MAP_();
    return { reverse %{$map} };
}


##
# new $class;
# -----------
# Instanciates and initializes a new flo::Editor object.
# Attempts to initialize using MKDoc::CGI->get object parameters
#
# Parameters names should look like block_1_text, block_2_image, etc
##
sub new
{
    my $class = shift;
    $class    = ref $class || $class;
    return _ETERNAL_() if _ETERNAL_();
    _SET_ETERNAL_ ($class->_new (@_));
    return _ETERNAL_();
}


sub _new
{
    my $class = shift;
    $class    = ref $class || $class;
    my $self  = bless { @_ }, $class;
    $self->{block} ||= {};
    $self->_initialize;
    return $self;
}


##
# $self->links;
# -------------
# This subroutine returns all the links of the current document
# and caches them into $LINKS, these links can then be used by
# MKDoc::XML::Tagger to linkify the whole document.
#
# It may look like a horrible hack (and it is), but since the
# component system needs to be redesigned it does not bother
# me that much anyway
##
sub links
{
    my $class = shift;
    return @{_LINKS_()} if (scalar @{_LINKS_()});

    my @links = ();
    
    my $self = _ETERNAL_();
    my @links_components = $self->access ( class => [ 'link' ] );
    
    # add the links for the child documents of that document
    my $document_table = flo::Standard::table ('Document');
    my $document = $document_table->get ( Full_Path => path_info );
    
    if (defined $document)
    {
	# retrieve the links for that document
	foreach my $link_component (@links_components)
	{
	    push @links, {
		href => $link_component->url,
		desc => $link_component->description,
		expr => $link_component->title,
	    } if ($link_component->url ne '' and $link_component->title ne '');
	}
	
	# plus we want to hyperlink all the children
	foreach my $child ($document->children)
	{
	    push @links, {
		href => $child->uri,
		desc => $child->description,
		expr => $child->title,
		lang => $child->lang,
	    }
	}
    }
    
    @links = sort { length ($b->{expr}) <=> length ($a->{expr}) } @links;
    _SET_LINKS_ (\@links);
    return @{_LINKS_()};
}


##
# $obj->access ( class  => [ 'text, link' ],
#                number => 1,
#                remove => 0 );
# -----------------------------
#   Lets you query and access an editor component
##
sub access
{
    my $self = shift;
    
    # search @_ for 'negate'
    my $component_negate = 'no';
    for (my $i = 0; $i < @_; $i++)
    {
	if (defined $_[$i] and $_[$i] eq 'negate')
	{
	    my ($key, $val) = splice @_, $i, 1;
	    $component_negate = 'yes';
	}
    }
    
    # search @_ for 'number' followed by something
    my $component_number = undef;
    for (my $i = 0; $i < @_; $i++)
    {
	if (defined $_[$i] and $_[$i] eq 'number')
	{	
	    my ($key, $val) = splice @_, $i, 2;
	    $component_number = $val if ($val =~ /^\d+$/);
	}
    }
    
    # search @_ for 'remove' followed by something
    my $component_remove = 'no';
    for (my $i = 0; $i < @_; $i++)
    {
	if (defined $_[$i] and $_[$i] eq 'remove')
	{
	    my ($key, $val) = splice @_, $i, 1;
	    $component_remove = 'yes';
	}
    }
    
    # search @_ for 'class' followed by something
    # this is for backwards compatibility
    for (my $i = 0; $i < @_; $i++)
    {
	if (defined $_[$i] and $_[$i] eq 'class')
	{
	    my ($key, $val) = splice @_, $i, 2;
	    push @_, (ref $val) ? @{$val} : $val;
	}
    }
    
    # all the rest is component classes
    my @component_class  = @_;
    
    if ($component_negate eq 'yes')
    {
	my @component_not = ();
	if ($component_number) { @component_not = $self->access ('number', $component_number, @component_class) }
	else                   { @component_not = $self->access (@component_class)                              }
	
	my @component_all = $self->access();
	my @result = ();
	
      LOOP:
	foreach my $component (@component_all)
	{
	    foreach my $component_not (@component_not)
	    {
		$component->equals ($component_not) and next LOOP;
	    }
	    push @result, $component;
	}
	
	return @result;
    }
    
    else
    {
	# first of all, let's capture the components given their type
	# into a pool of matching components
	my @pool = ();
	
	foreach my $id (sort { $a <=> $b } keys %{$self->{block}})
	{
	    my $component = $self->{block}->{$id};
	    my $type = ref $component;
	    
	    defined _PAM_()->{$type} || next;
	    
	    if (@component_class)
	    {
		push @pool, map { ( defined _MAP_()->{$_} and _MAP_()->{$_} eq $type ) ? $id : () } @component_class;
	    }
	    else
	    {
		push @pool, $id;
	    }
	}
	
	# if the component number is specified, then we should deal
	# with that component only
	if ($component_number)
	{
	    my $id = $pool[$component_number - 1];
	    return unless (defined $id);
	    if ($component_remove eq 'yes') { return delete $self->{block}->{$id} }
	    else                            { return $self->{block}->{$id}        }
	}
	else
	{
	    if ($component_remove eq 'yes') { return map { delete $self->{block}->{$_} or {} } @pool }
	    else                            { return map { $self->{block}->{$_} } @pool              }
	}
    }
}


sub count
{
    my $self = shift;
    my @comp = $self->access();
    return scalar @comp;
}



##
# $self->_initialize;
# -------------------
# Initializes the newly created object
##
sub _initialize
{
    my $self = shift;
    my $cgi  = $self->{cgi} || return;
    
    $self->_initialize_initialize ($cgi);
    $self->_initialize_delete ($cgi);
    $self->_initialize_up ($cgi);
    $self->_initialize_down ($cgi);
    $self->_initialize_add ($cgi);
    $self->_initialize_uri_names ($cgi);
}


##
# $self->_initialize_uri_names;
# -----------------------------
# Initializes the uri_name of each component. This step MUST
# be reproductible, i.e. if ran twice it should initialize
# components' URIs the same way.
##
sub _initialize_uri_names
{
    my $self = shift;
    
    # this hash contains the names which have been
    # already used by another component
    my %names = ();
    foreach my $component ($self->access())
    {
	my $name = $component->can ('preferred_uri_name') ?
	    $component->preferred_uri_name() :
	    $component->type();

	my $ext = $component->can ('preferred_extension') ?
	    $component->preferred_extension() :
	    $component->type();	    
	
	$name = $self->_compute_unique_uri_name ($name, \%names);
	$component->{uri_name} = "$name.$ext";
    }
}


##
# $self->_compute_unique_uri_name ($name, \%names);
# -------------------------------------------------
# Compute a unique uri name which is not in \%names,
# add it to \$names, and returns it.
##
sub _compute_unique_uri_name
{
    my $self  = shift;
    my $name  = shift;
    my $names = shift;
    
    my $res   = $name;
    my $count = 1;
    
    while (defined $names->{$res}) { $res = $name . '-' . ++$count }
    $names->{$res} = 1;
    return $res;
}


##
# $self->_initialize_initialize;
# ------------------------------
# Scans the CGI parameters to set all the components
##
sub _initialize_initialize
{
    my $self = shift;
    my $cgi  = shift;
    
    # This loop sets all the components by scanning the CGI parameters
    foreach my $cgi_param_name ($cgi->param)
    {
	my $qBLOCK = quotemeta (BLOCK) . "_";

	# if this CGI param is appropriate because it starts by BLOCK
	if ($cgi_param_name =~ /^$qBLOCK([0-9]+)_([a-zA-Z]+)(?:_.*)?$/)
	{
	    my ($id, $type) = $cgi_param_name =~ /^$qBLOCK([0-9]+)_([a-zA-Z]+)(?:_.*)?$/;
	    
	    # if $self->{block}->{$id} is already defined, then this component
	    # has been instanciated already and we can safely move on the next
	    # parameter
	    next if (defined $self->{block}->{$id});

	    my $map = _MAP_();
	    
	    my $class = $map->{$type};
	    my $file  = $class;
	    $file =~ s/::/\//g;
	    $file .= ".pm";
	    require $file;
	    import  $file;
	    
	    my $param_name = $qBLOCK . $id . '_' . $type;
	    $self->{block}->{$id} = $class->new ( cgi => $cgi, param_name => $param_name );
	}
    }
}


##
# $self->_initialize_delete;
# --------------------------
#   Scans the CGI parameters to check if a component has
#   to be removed
##
sub _initialize_delete
{
    my $self = shift;
    my $cgi  = shift;
    
    # Then we need to re-scan the cgi parameters to check if deleting
    # a component is requested
    foreach my $cgi_param_name ($cgi->param)
    {
	my $qBLOCK = quotemeta (BLOCK) . "_";

	if ($cgi_param_name =~ /^$qBLOCK([0-9]+)_([a-zA-Z]+)_delete$/)
	{
	    my ($id, $type) = $cgi_param_name =~ /^$qBLOCK([0-9]+)_([a-zA-Z]+)_delete$/;
	    delete $self->{block}->{$id};
	}
    }
    
    # we may have deleted some elements, thus we need to squash the list
    # of identifiers ex 1 2 3 5 => 1 2 3 4
    my @components = map { $self->{block}->{$_} } sort { $a <=> $b } keys %{$self->{block}};
    $self->{block} = {};
    
    for (my $i = 0; $i < @components; $i++)
    {
	my $component = $components[$i];
	my $type   = _PAM_()->{ref $component};
	my $new_id = $i + 1;

	$component->{param_name} = BLOCK . "_" . $new_id . "_" . $type;
	$self->{block}->{$new_id} = $component;
    }
}


##
# $self->_initialize_up;
# ----------------------
#   Scans the CGI parameters to check if a component has
#   to be moved up
##
sub _initialize_up
{
    my $self = shift;
    my $cgi  = shift;
   
    # Then we need to re-scan the cgi parameters to check if a component
    # has to move up.
    foreach my $cgi_param_name ($cgi->param)
    {
	my $qBLOCK = quotemeta (BLOCK) . "_";
	
	if ($cgi_param_name =~ /^$qBLOCK([0-9]+)_([a-zA-Z]+)_up$/)
	{
	    my ($id, $type) = $cgi_param_name =~ /^$qBLOCK([0-9]+)_([a-zA-Z]+)_up$/;
	    if ($id > 1)
	    {
		my $id_1 = $id;
		my $id_2 = $id - 1,
		
		my $component_1 = $self->{block}->{$id_1};
		my $component_2 = $self->{block}->{$id_2};
		
		my $type_1 = _PAM_()->{ref $component_1};
		my $type_2 = _PAM_()->{ref $component_2};

		$self->{block}->{$id_1} = $component_2;
		$self->{block}->{$id_2} = $component_1;

		$component_1->{param_name} = BLOCK . "_" . $id_2 . "_" . $type_1;
		$component_2->{param_name} = BLOCK . "_" . $id_1 . "_" . $type_2;
	    }
	}
    }
}


##
# $self->_initialize_up;
# ----------------------
#   Scans the CGI parameters to check if a component has
#   to be moved down
##
sub _initialize_down
{
    my $self = shift;
    my $cgi  = shift;
    
    # Then we need to re-scan the cgi parameters to check if a component
    # has to move down
    foreach my $cgi_param_name ($cgi->param)
    {
	my $qBLOCK = quotemeta (BLOCK) . "_";
	
	if ($cgi_param_name =~ /^$qBLOCK([0-9]+)_([a-zA-Z]+)_down$/)
	{
	    my ($id, $type) = $cgi_param_name =~ /^$qBLOCK([0-9]+)_([a-zA-Z]+)_down$/;
	    if ($id < scalar keys %{$self->{block}})
	    {
		my $id_1 = $id;
		my $id_2 = $id + 1,
		
		my $component_1 = $self->{block}->{$id_1};
		my $component_2 = $self->{block}->{$id_2};
		
		my $type_1 = _PAM_()->{ref $component_1};
		my $type_2 = _PAM_()->{ref $component_2};
		
		$self->{block}->{$id_1} = $component_2;
		$self->{block}->{$id_2} = $component_1;
		
		$component_1->{param_name} = BLOCK . "_" . $id_2 . "_" . $type_1;
		$component_2->{param_name} = BLOCK . "_" . $id_1 . "_" . $type_2;
	    }
	}
    }
}


##
# $self->_initialize_add;
# -----------------------
#   Scans the CGI parameters to check if a component has
#   to be added
##
sub _initialize_add
{
    my $self = shift;
    my $cgi  = shift;
    
    # Then we need to re-scan the cgi parameters to check if a component
    # has to be added
    if (defined $cgi->param ('_new_component_add'))
    {
	my $cgi_param_name = $cgi->param ('_new_component');
	my $qBLOCK = quotemeta (BLOCK) . '_';
	my ($id, $type) = $cgi_param_name =~ /^$qBLOCK([0-9]+)_([a-zA-Z]+)$/;
	my $nb = $cgi->param ('_new_component_amount');
	my $map = _MAP_();
	my $class = $map->{$type};
	my $file  = $class;
	$file =~ s/::/\//g;
	$file .= ".pm";
	require $file;
	import  $file;
	
	$nb = 0 unless ($nb =~ /^[0-9]+$/);
	$nb = 100 if ($nb > 100);
	while ($nb > 0)
	{
	    $self->{block}->{$id} = $class->new ( cgi => $cgi, param_name => BLOCK . "_" . $id . "_" . $type );
	    $id++;
	    $nb--;
	}
    }
}


sub next_id
{
    my $self = shift;
    my @ids  = sort { $b <=> $a } keys %{$self->{block}};
    return (scalar @ids) ? 1 + $ids[0] : 1;
}


sub add_component
{
    my $self      = shift;
    my $parent    = shift;
    my $component = shift;
    $component = $component->clone();
    
    my $next_id                = $self->next_id();
    $component->{parent_id}    = $parent->id();
    $component->{position}     = $next_id;
    $self->{block}->{$next_id} = $component;
}


sub del_component
{
    my $self = shift;
    my $component = shift;
    
    my @components = map  { $self->{block}->{$_} }
                     sort { $a <=> $b }
		     keys %{$self->{block}};
    
    my @list = ();
    for (@components)
    {
	push @list, $_ unless ($_->equals ($component));
    }
    
    my $count = 0;
    $self->{block} = {};
    foreach my $cmp (@list)
    {
	$count++;
	$self->{block}->{$count} = $cmp;
    }
}


sub component
{
    my $self = shift;
    my $id   = shift;
    return $self->{block}->{$id};
}


##
# $self->edit;
# ------------
#   Returns the body html for component edition
##
sub edit
{
    my $self = shift;
    my @out  = map  { $self->{block}->{$_}->edit }
               sort { $a <=> $b }
               keys %{$self->{block}};
    
    my @add = ();
    my @id  = keys %{$self->{block}};
    my $next_id = @id + 1;
    
    my $map = _MAP_();
    my $pam = _PAM_();
    foreach (sort keys %{$pam})
    {
	my $type = $pam->{$_};
	push @add, { label => $type, name => BLOCK . "_" . $next_id . "_" . $type };
    }
    
    # build the add component box
    my $doc  = flo::Standard::current_document();
    my $tmpl = new Petal ( file => 'editor/box', lang => $doc->language() );
    push @out, $tmpl->process ( option_list => \@add, document => $doc );
    return join "\n", @out;
}


##
# $self->generate_xml;
# --------------------
#   Generates easily parsable pseudo-xml for database storage
##
sub generate_xml
{
    my $self = shift;
    my @out  = ();
    my $count = 0;
    foreach my $id (sort { $a <=> $b } keys %{$self->{block}})
    {
	my $component = $self->{block}->{$id};
	
	$count++;
	my $type  = _PAM_()->{ref $component};
	my $start = qq |<block id="flo_block_$count" type="$type">|;
	my $comp  = $component->generate_xml;
	my $stop  = qq |</block>|;
	push @out, ($start, $comp, $stop);
    }
    return join "\n", @out;
}


##
# $self->parse_xml;
# -----------------
#   Parses the database pseudo-xml and turns it back into
#   proper memory structure
##
sub parse_xml
{
    my $self = shift;
    my $xml  = shift || "";
    my $doc_id = shift;
    my @xml  = split /\n/, $xml;
    
    my $cgi  = $self->{cgi};
    my @current_data = ();
    my $current_id   = undef;
    my $current_type = undef;
    foreach my $line (@xml)
    {
	if ($line =~ /^<block id=\"flo_block_[0-9]+\" type=\".*\">/)
	{
	    ($current_id, $current_type) = $line =~ /^<block id=\"flo_block_([0-9]+)\" type=\"(.*?)\">/;
	}
	elsif ($line =~ /^<\/block>/)
	{
	    if (defined $current_id and defined $current_type)
	    {
		my $map = _MAP_();
		my $class = $map->{$current_type} || next;
		my $file = $class;
		$file =~ s/::/\//g;
		$file .= ".pm";
		require $file;
		import  $file;
		
		my $component = $class->new (cgi => $cgi, param_name => BLOCK . "_" . $current_id . "_" . $current_type);
		
		$component->parse_xml (join "\n", @current_data);
		$component->{param_name} = BLOCK . "_" . $current_id . "_" . $current_type;
		$component->{parent_id}  = $doc_id;
		$component->{position}   = $current_id;
		$component->{is_saved}   = 1;
		$self->{block}->{$current_id} = $component;
		@current_data = ();
	    }
	}
	else { push @current_data, $line }
    }
    
    $self->_initialize_uri_names ($cgi);
}


##
# $self->html;
# ------------
#   Returns the HTML that represents all the components.
#   This method is deprecated and should not be used anymore
##
sub html
{
    my $self = shift;
    my @out  = ();
    foreach my $id (sort { $a <=> $b } keys %{$self->{block}})
    {
	my $type = _PAM_()->{ref $self->{block}->{$id}};
	push @out, ( qq |<div id="flo_block_$id" class="$type">|,
		     $self->{block}->{$id}->html,
		     qq |</div>| );
    }
    my $res = join "\n", @out;
    # return $self->_make_links ($res);
    return $res;
}


sub validate
{
    my $self = shift;
    my $cgi  = shift;
    
    my $ok   = 1;
    foreach my $component (values %{$self->{block}})
    {
	$ok = 0 unless ($component->validate());
    }
    
    return $ok;
}


1;
