# -------------------------------------------------------------------------------------
# flo::Plugin
# -------------------------------------------------------------------------------------
# Author : Jean-Michel Hiver
# Copyright : (c) 2001, MKDoc Holdings Ltd
#
# Abstract class for MKDoc plugins.  The plugin processor (MKDoc::process)
# executes a list of plugins until one handles the HTTP request.
# -------------------------------------------------------------------------------------
package flo::Plugin;
use flo::Object;
use flo::Standard;
use flo::HTTP::Header;
use MKDoc::Ouch;
use Petal;
use strict;
use warnings;
use Carp;
use base qw /flo::Object/;
use Data::Dumper;


##
# Petal modifier 'plugin:'
# ------------------------
# Instanciates a plugin of class plugin_type and returns it. This is handy if
# you need to access the attributes methods of another plugin from the plugin
# you're on.
#
# Example: petal:define="details flo::plugin::Account::Details"
##
$Petal::Hash::MODIFIERS->{'plugin:'} = sub {
    my $hash  = shift;
    my $class = shift;
    my $file  = $class;
    $file =~ s/::/\//g;
    $file .= '.pm';
    
    for (@INC)
    {
	(-e "$_/$file") and do {
	    require $file;
	    last;
	};
    }
    
    return unless defined $INC{$file}; 
    return eval "$class->new()";
};


# equals has a special meaning for plugins
sub equals
{
    my $self  = shift;
    my $thing = shift;
    return ref $self eq ref $thing;
}


##
# would_activate
# --------------
# Returns TRUE if this plugin would activate if its address was requested.
# Returns FALSE otherwise.
# For example: <span
#                petal:define="details flo::plugin::Account::Details"
#                petal:condition="details/would_activate"
#              > ./.account.details can be accessed from here! </span>
#
# For convenience, would_activate always returns FALSE if invoked on the same 
# class of plugin that is currently being activated.
# 
# i.e. if you were using the piece of code above in the account details plugin
# itself, details/would_activate would always return false.
##
sub would_activate
{
    my $self  = shift;
    return if ($self->is_current_plugin());
    
    my $class = ref $self || $self;
     
    # let's lie about what the path_info is...
    my $path_info = flo::Standard::path_info();
    $path_info =~ s/\/.*?$/\//;
    $path_info .= $self->name() if ($self->name());    
    
    local *flo::Standard::path_info;
    *flo::Standard::path_info = sub { $path_info };

    local *flo::Standard::raw_path_info;
    *flo::Standard::raw_path_info = sub { $path_info };
    
    # if the plugin wouldn't want to activate,
    # then return false.
    return unless ($self->activate());

    # from then on, if the plugin happens to be in the
    # plugin list, then it would definitely activate and
    # we return true.
    my @plugin = MKDoc::Config->config_lines ( MKDoc::Config->PLUGIN_LIST );
    for my $plugin (@plugin)
    {
        return 1 if ($plugin eq $class);
    }

    # if we get there, the plugin could not activate since it
    # wasn't in the plugin list so it could never be triggered
    return;
}


##
# is_current_plugin
# -----------------
# Returns TRUE if this plugin is the current plugin, FALSE otherwise.
# For example: <span
#                petal:define="details flo::plugin::Account::Details"
#                petal:condition="details/is_current_plugin"
#              > Hey, you want to change your account details don't you! </span>
#
# Indeed for this reason self/is_current_plugin is always TRUE.
##
sub is_current_plugin
{
    my $class = shift;
    $class = ref $class || $class;
    return $MKDoc::CurrentPlugin eq $class;
}

sub is_current_plugin_nowarning
{
    $MKDoc::CurrentPlugin; 
}


# $class->main();
# ---------------
# This class method is called by the MKDoc module.
# Unless overriden it instantiates the current plugin and
# checks to see if it should be activated.  If so it processes
# the request by calling run(), otherwise it does nothing.
#
# Returns 'TERMINATE' if the request was handled by this plugin,
# false if not, or undef on error.
sub main
{
    my $class = shift || return;
    my $self  = $class->new (@_);
    return ($self->activate) ? do
    {
	# set up the callback for errors
	local $MKDoc::Ouch::CALLBACK;
	$MKDoc::Ouch::CALLBACK = sub { $self->add_error (@_) };
    
	# set up the flo::Editor object
	local $::MKD_flo_Editor_ETERNAL;
	$::MKD_flo_Editor_ETERNAL = do {
	    my $document = flo::Standard::current_document();
	    my $editor   = _new flo::Editor();
	    $editor->parse_xml ($document->{Body});
	    $editor;
	};

	$self->run();
    }
    :
    undef;
}


# $class->new;
# ------------
# Very basic constructor which instantiates a new object.
# Calls the _initialize method.
sub new
{
    my $class = shift;
    $class    = ref $class || $class;
    my $self  = bless { @_ }, $class;
    $self->_initialize;
    return $self;
}


# $obj = $obj->_initialize (@args);
# ---------------------------------
# Initializes this object, called by new().  This is empty, and
# here to be overrided if necessary. Returns $self, or undef on
# failure.
sub _initialize
{
    my $self = shift;
    return $self;
}


# $self->run();
# -------------
# Runs the current plugin.
sub run
{
    my $self = shift;    
    if ($ENV{REQUEST_METHOD} =~ /^POST$/) { $self->http_post (@_) }
    else                                  { $self->http_get  (@_) }
}


##
# is_get
# ------
# Returns true if the query is a GET query, FALSE otherwise.
#
# Example: <span petal:condition="self/is_get">
#          You're doing a GET, aren't you?
#          </span>
##
sub is_get { $ENV{REQUEST_METHOD} =~ /^GET$/ || $ENV{REQUEST_METHOD} =~ /HEAD/ }


##
# is_post
# -------
# Returns true if the query is a POST query, FALSE otherwise.
#
# Example: <span petal:condition="self/is_get">
#          You're doing a POST, aren't you?
#          </span>
##
sub is_post { $ENV{REQUEST_METHOD} =~ /^POST$/ }


# $self->http_get();
# ------------------
# Processes an HTTP GET request.
sub http_get
{
    my $self = shift;
    $self->render_http (
	self       => $self,
	object     => $self->parent,
	__input__  => 'XML',
	__output__ => 'XHTML',
       );
    return 'TERMINATE';
}


# $self->http_post();
# -------------------
# Processes an HTTP POST request.
sub http_post
{
    my $self = shift;
    return $self->http_get (@_);
}


# $self->activate();
# ------------------
# Returns TRUE if this plugin needs to catch the request,
# FALSE otherwise.
sub activate
{
    my $self = shift;

    my $name = $self->name();
    warn "$self doesn't seem to have a name?" unless (defined $name);
    my $q_http_name   = quotemeta ($name);
    my $raw_path_info = flo::Standard::raw_path_info() || return;
    
    my $current_document = flo::Standard::current_document();
    return unless (defined $current_document);
    return $raw_path_info =~ /.*\/$q_http_name$/;
}


##
# name
# ----
# Returns the HTTP NAME of the this plugin. For example, for the sitemap it's
# likely to return something like '.sitemap.html'. Almost all plugins HTTP name
# start with a dot since it is necessary to make a clear distinction between the
# address of a plugin (i.e. http://example.com/.some.plugin) and the address of
# a file attachment (i.e. http://example.com/some-file.txt).
##
sub name
{
    my $self = shift;
    
    my $env_name = $self->_name_env();
    my $def_name = $self->_name_default();
    return $ENV{$env_name} || $def_name;
}


sub _name_env
{
    my $class = shift;
    $class = ref $class || $class;
    $class =~ s/^flo::plugin:://;
    $class =~ s/\.pm$//;
    $class = uc ($class);
    $class =~ s/::/_/g;
    return "MKD__" . $class . "_HTTP";
}


sub _name_default
{
    my $class = shift;
    $class = ref $class || $class;
    $class =~ s/^flo::plugin:://;
    $class =~ s/\.pm$//;
    $class = lc ($class);
    $class =~ s/::/./g;
    return ".$class";
}


##
# path;
# -----
# Returns the path of this plugin, i.e. '/foo/bar/.plugin'
##
sub path
{
    my $self   = shift;
    my $parent = $self->parent;
    return $self->parent->path . $self->name;
}


##
# uri <key1, value1, ...>
# ---------------------
# Returns the URI of this plugin, i.e.
# http://example.com/.plugin
#
# It is possible to pass parameters, for example self/uri --foo 'bar'
# could produce something like:
#
# http://example.com/.plugin?foo=bar
##
sub uri
{
    my $self = shift;
    my $cgix = flo::Standard::cgi()->new;
    $cgix->delete ($_) for ($cgix->param);
    while (@_)
    {
	my ($key, $val) = (shift, shift);
	$cgix->param ($key, $val);
    }
    $cgix->path_info ($self->path);
    return $cgix->self_url;
}


##
# title
# -----
# Returns the title of the document this plugin is associated with.
##
sub title
{
    my $self = shift;
    return $self->parent->title;
}


##
# $self->parent;
# --------------
# Returns the data object (here a document) which this plugin is currently
# associated with.
##
sub parent
{
    my $self = shift;
    my $document = flo::Standard::current_document();
    return $document;
}


##
# $self->lang;
# ------------
# Returns the language of the current plugin, which is directly inherited
# from the parent language.
##
sub lang
{
    my $self = shift;
    return $self->parent->lang;
}


sub language { return shift()->lang (@_) }


##
# $self->lang_label;
# ------------------
# Returns the current language label for that document,
# i.e. 'British English'
##
sub lang_label
{
    my $self   = shift;
    my $lang   = $self->Lang;
    my $hash   = MKDoc::Config->parsefile_hashref (MKDoc::Config->LANGUAGE_LIST);
    return $hash->{$lang};
}


##
# $self->align;
# -------------
# Returns the alignment of this object depending on the language,
# i.e. 'left' or 'right'
##
sub align
{
    my $self   = shift;
    return ($self->direction eq 'ltr') ? 'left' : 'right';
}


##
# $self->align_opposite;
# ----------------------
# Returns 'left' when align() returns 'right', returns 'right' otherwise
##
sub align_opposite
{
    my $self = shift;
    return ($self->direction eq 'ltr') ? 'right' : 'left';
}


##
# $self->direction;
# -----------------
# Returns the direction depending on the language, i.e. 'left' or 'right'
##
sub direction
{
    my $self = shift;
    my $hash = MKDoc::Config->parsefile_hashref (MKDoc::Config->LANGUAGE_LIST_RTL);
    my $lang = $self->language();
    return exists $hash->{$lang} ? 'rtl' : 'ltr';
}


##
# $self->ancestors;
# -----------------
# Returns the ancestors of that document, i.e.
# its parent, grand-parent, grand-grand parent, etc...
# the root element being the first element of the
# array which is being returned.
#
# This method returns an array reference in scalar context,
# or a list otherwise.
##
sub ancestors
{
    my $self   = shift;
    my $parent = $self->parent;
    if (defined $parent)
    {
	my @res = ($parent->ancestors, $parent);
	return (wantarray) ? @res : \@res;
    }
    else
    {
	return wantarray ? () : [];
    }
}


##
# $self->content_type;
# --------------------
# Returns the content-type associated with this component
# plugin
##
sub content_type { return "text/html; charset=UTF-8" }


##
# $self->user();
# --------------
# Returns the current user.
##
sub user
{
    return flo::Standard::current_user();
}


##
# $self->has_errors();
# --------------------
# Returns TRUE if the current plugin object holds error that needs
# to be displayed, false otherwise.
##
sub has_errors
{
    my $self = shift;
    my $errors = $self->errors;
    return scalar @{$errors};
}


##
# $self->errors();
# ----------------
# Returns a list of errors.
##
sub errors
{
    my $self = shift;
    return (defined $self->{'.errors'}) ?
        $self->{'.errors'} :
	[];
}


##
# $self->add_error (@_);
# ----------------------
# Adds all the errors in @_ to the current plugin object.
##
sub add_error
{
    my $self = shift;
    $self->{'.errors'} ||= [];
    push @{$self->{'.errors'}}, @_;
}


##
# $header = $self->render_http (%args);
# -------------------------------------
#   Renders the current object with render().
#   Wraps the rendered document in a flo::HTTP::Header object.
#
#   Returns header object on success, or undef on error.
#   See render(), template_path().
##
sub render_http
{
    my $self   = shift || return;
    my $data   = $self->render (@_) || return;
    my $header = new flo::HTTP::Header;

    {
        use bytes;
        $header->set ("Content-Type: "   . $self->content_type);
        no bytes;
    }
    
    $header = $header->header;
    Encode::_utf8_off ($header);
    Encode::_utf8_off ($data);
    
    use bytes;
    $ENV{REQUEST_METHOD} ||= 'GET';
    if ($ENV{REQUEST_METHOD} =~ /^HEAD$/i) { print $header         }
    else                                   { print $header . $data }
}


##
# $doc = $self->render (%args);
# -----------------------------
#   Renders the current object, passing %args to the template being used.
#   Intercepts '__input__' and '__output__' to set Petal processing vars.
#
#   Returns the rendered document on success, or undef on error.
#   See template_path().
##
sub render
{
    my $self = shift;
    my $hash = (ref $_[0]) ? shift : { @_ };
    $Petal::DISK_CACHE = 1;
    $Petal::MEMORY_CACHE = 1;
    $Petal::INPUT  = $hash->{__input__}  || 'XML';
    $Petal::OUTPUT = $hash->{__output__} || 'XML';
    
    my $template = new Petal
        language => $self->language(),
	file     => $self->template_path();
    
    my $data = $template->process ( @_ );
    return $data;
}


sub cgi
{
    my $self = shift;
    return flo::Standard::cgi();
}


sub env
{
    my $self = shift;
    return \%ENV;
}


sub root
{
    my $self = shift;
    my $document_t = flo::Standard::table ('Document');
    return $document_t->get ( Full_Path => '/' );
}


sub config_class
{
    my $self = shift;
    my $what = shift || return; 
    return MKDoc::Config->$what;
}


1;
