# -------------------------------------------------------------------------------------
# flo::plugin::Discussion::IMAP
# -------------------------------------------------------------------------------------
#
#       Author : Jean-Michel Hiver <jhiver@mkdoc.com>
#    Copyright : (c) MKDoc Holdings Ltd, 2002
#
#      Unauthorized modification, use, reuse, distribution or redistribution
#      of this module is stricly forbidden
#
# -------------------------------------------------------------------------------------
package flo::plugin::Discussion::IMAP;
use MKDoc::Config;
use flo::plugin::Discussion::Message;
use Mail::IMAPClient;
use Text::Wrap;
use flo::Standard qw /cgi path_info raw_path_info table template current_document/;
use warnings;
use strict;
use Carp;
use 5.008_000;
use Encode qw /encode decode _utf8_on is_utf8/;

use base qw /flo::Object/;
use vars qw /%Threads @Stack $Current %Fetch/;


sub new
{
    my $class = shift;
    $class = ref $class || $class;
    $::MKD_IMAP ||= do {
	my $self = bless {}, $class;
	$self->_initialize();
	$self;
    };
    
    return $::MKD_IMAP;
}


sub _initialize
{
    my $self = shift;
    my %config = $self->_get_imap_config();
    $self->_connect_imap (%config);
    $self->_configure_imap_namespace;
}


sub _get_imap_config
{
    my $self = shift;
    my $config = $self->config;
    my $server   = MKDoc::Config->IMAP_SERVER;
    my $user     = MKDoc::Config->IMAP_USER;
    my $password = MKDoc::Config->IMAP_PASSWORD;
    my $port     = MKDoc::Config->IMAP_PORT;
    my %config = (
	Server   => $server,
	User     => $user,
	Password => $password,
	Port     => $port,
       );
    
    return wantarray ? %config : \%config;
}


sub _connect_imap
{
    my $self = shift;
    my $imap = Mail::IMAPClient->new (@_) or confess "Cannot connect: $@";
    $imap->Uid (1);
    $imap->IsConnected     or confess ("\$imap not connected");
    $imap->IsAuthenticated or confess ("\$imap not authenticated");
    $self->set_imap ($imap);
}


sub _configure_imap_namespace
{
    my $self = shift;
    my $imap = $self->imap;
    my ($has_ns_capability) = grep /^NAMESPACE$/, $imap->capability;
    if ($has_ns_capability)
    {
	my $ns_command = join '', $imap->tag_and_run ('NAMESPACE');
	my ($prefix, $delimiter) = $ns_command =~ /(NIL|\".*?\")/gsm;
	unless ($prefix eq 'NIL')
	{
	    $prefix =~ s/^\"//;
	    $prefix =~ s/\"$//;
	    $self->set_imap_prefix ($prefix);
	    
	    $delimiter =~ s/^\"//;
	    $delimiter =~ s/\"$//;
	    $self->set_imap_delimiter ($delimiter);
	    return;
	}
    }
    
    $self->set_imap_prefix ('');
    $self->set_imap_delimiter ('.');
}


sub DESTROY
{
    my $self = shift;
    my $imap = $self->imap;
    $imap->disconnect if (defined $imap);
}


sub imap
{
    my $self = shift;
    confess ('foo') unless (ref $self);
    return $self->{imap_client};
}


sub set_imap
{
    my $self = shift;
    $self->{imap_client} = shift;
}


sub imap_prefix
{
    my $self = shift;
    return $self->{imap_prefix};
}


sub set_imap_prefix
{
    my $self = shift;
    $self->{imap_prefix} = shift;
}


sub imap_delimiter
{
    my $self = shift;
    return $self->{imap_delimiter};
}


sub set_imap_delimiter
{
    my $self = shift;
    $self->{imap_delimiter} = shift;
}


sub mbox
{
    my $self = shift;
    return $self->{mbox};
}


sub prefixed_mbox
{
    my $self = shift;
    my $mbox = shift || $self->mbox;
    return $self->imap_prefix . $mbox;
}


sub set_mbox
{
    my $self = shift;
    $self->{mbox} = shift;
    $self->imap()->select ($self->prefixed_mbox);
}


sub mbox_exists
{
    my $self = shift;
    my $imap = $self->imap;
    my $folder = $imap->Folder;
    return $imap->exists ($folder);
}


sub mbox_create
{
    my $self = shift;
    my $mbox = shift;
    my $imap = $self->imap;
    $imap->create ($self->prefixed_mbox ($mbox));
}


sub post
{
    my $self    = shift;
    my $message =  shift;
    my $imap    = $self->imap;
    my $mbox    = $self->mbox;
    my $mail = $self->_post_construct_mail ($message);
    return $imap->append ($self->prefixed_mbox, $mail);
}


sub _post_clean_header
{
    my $self = shift;
    my $res  = shift || '';
    $res =~ s/(\n|\r)+/ /gsm;
    $res =~ s/[\x00-\x08]//g;
    $res =~ s/[\x0B-\x0C]//g;
    $res =~ s/[\x0E-\x1F]//g;
    $res =~ s/^\s+//;
    $res =~ s/\s+$//;
    return $res;
}


sub _post_clean_body
{
    my $self = shift;
    my $res  = shift;
    my @data = split /\n/sm, $res;
    
    my @result  = ();
    my @current = ();
    for (@data)
    {
	chomp();
	chomp();
	/^\s*\>/ and do {
	    if (@current)
	    {
		my $current = Text::Wrap::wrap ('', '', @current);
		push @result, split /\n/sm, $current;
		@current = ();
	    }
	    push @result, $_;
	    next;
	};
	
	/^\s*$/ and do {
	    if (@current)
	    {
		my $current = Text::Wrap::wrap ('', '', @current);
		push @result, split /\n/sm, $current;
		@current = ();
	    }
	    push @result, $_;
	    next;
	};
	
	push @current, $_;
    }
    
    if (@current)
    {
	my $current = Text::Wrap::wrap ('', '', @current);
	push @result, split /\n/sm, $current;
	@current = ();
    }
    
    return join "\n", @result;
}


sub _post_construct_mail
{
    my $self = shift;
    my $message = shift;
    my $imap    = $self->imap;
    my $mbox    = $self->mbox;
    
    my $realname    = $self->_post_clean_header ($message->{realname});
    my $email       = $self->_post_clean_header ($message->{email});
    my $subject     = $self->_post_clean_header ($message->{subject});
    my $body        = $self->_post_clean_body   ($message->{message});
    my $language    = $self->_post_clean_header ($message->{language});
    my $in_reply_to = $message->{in_reply_to};
    my $references  = $message->{references};
    my $now         = $imap->Rfc822_date (time);
    
    my $imap_user = new MKDoc::Config ('flo.plugin.Discussion.IMAP')->get ('user');
    my $to  = $imap_user . '+' . $mbox . '@' . $ENV{SERVER_NAME};
    
    my $from_encoded    = encode ('MIME-Header', "$realname <$email>");
    my $to_encoded      = encode ('MIME-Header', $to);
    my $subject_encoded = encode ('MIME-Header', $subject);
    
    my $rand1  = join '', map { chr (ord ('A') + int (rand (26))) } 1..8;
    my $rand2  = join '', map { chr (ord ('A') + int (rand (26))) } 1..8;
    my $msg_id = $email;
    $msg_id =~ s/.*\@//;
    $msg_id = "$rand1.$rand2\@$msg_id";
    
    my $rhost = $ENV{REMOTE_HOST} || $ENV{REMOTE_ADDR};
    my $mail = <<EOF;
Return-Path: <$email>
Received: from $rhost [$ENV{REMOTE_ADDR}]
        by $ENV{SERVER_NAME} with MKDoc-Forum;
        $now
Delivered-To: $to
From: $from_encoded
To: $to_encoded
Subject: $subject_encoded
Message-ID: <$msg_id>
EOF

    if ($references)
    {
	$mail .= <<EOF;
References: $references
EOF
    }

    $mail .= <<EOF;
Mime-Version: 1.0
Content-Type: text/plain; charset=UTF-8
Content-Transfer-Encoding: 8bit
Content-Disposition: inline
Content-Language: $language
EOF

    if ($in_reply_to)
    {
	$mail .= <<EOF;
In-Reply-To: $in_reply_to
EOF
    }
    
    $mail .= <<EOF;
User-Agent: $ENV{HTTP_USER_AGENT}
Sender: $email
Date: $now
EOF

    $mail =~ s/^(\r|\n)+//sm;
    $mail .= "\n$body";
    Encode::_utf8_off ($mail);
    return $mail;
}


sub message_body
{
    my $self = shift;
    my $uid  = shift || confess "\$uid not specified";
    my $imap = $self->imap;
    my $body_string = $imap->body_string ($uid);
    my @lines = split /\n/sm, $body_string;
    pop (@lines) if ($lines[$#lines] =~ /^\s*\)\s*$/);
    return join "\n", @lines;
}


sub message_header
{
    my $self = shift;
    my $uid  = shift || confess "\$uid not specified";
    my $imap = $self->imap;
    return $imap->parse_headers ($uid, 'ALL')
}


sub message_exists
{
    my $self = shift;
    my $uid  = shift || return;
    my $res  = $self->select_headers ("UID $uid");
    (scalar (keys %{$res})) ? return 1 : return;
}


sub select_headers
{
    my $self      = shift;
    my $condition = shift || 'ALL';
    my @headers   = @_;
    @headers = ('ALL') unless (scalar @headers);
    
    my $imap = $self->imap;
    return $imap->parse_headers (scalar ($imap->search ($condition)) , @headers)
}


sub select_threaded_messages
{
    my $self = shift;
    my $imap = $self->imap;
    my $condition = shift || 'ALL';
    
    # bulk-load message headers to avoid loading them one by
    # one later on. It's a bit of a kludge but it increases
    # performance greatly.
    $::MKD_MESSAGES_HEADERS_TEMP = $self->select_headers (
	$condition,
	qw/From To Date Subject In-Reply-To Content-Language/
       );
    
    my ($res_string) = grep /\*\s+THREAD/, $imap->tag_and_run ("UID THREAD REFERENCES US-ASCII $condition");
    my $res = $self->_messages_threaded ($res_string);
    $self->_messages_parentify ($res);
    
    # message headers have been set in each message object,
    # they are now properly threaded, thus this variable becomes
    # unecessary.
    $::MKD_MESSAGES_HEADERS_TEMP = {};
    return $res;
}


sub slice_threaded
{
    my $self  = shift;
    my $slice = shift;
    my $uids = [ @{$slice->{uids}} ];
    return new flo::plugin::Discussion::Message ( children => [] ) unless (defined $uids);
    return new flo::plugin::Discussion::Message ( children => [] ) unless (scalar @{$uids});
    if (scalar @{$uids} == 1) { return $self->select_threaded_messages ("UID $uids->[0]") }
    if (scalar @{$uids} == 2) { return $self->select_threaded_messages ("OR UID $uids->[0] UID $uids->[1]") }
    
    my $condition = '';
    while (@{$uids} != 2)
    {
	my $uid = shift (@{$uids});
	$condition .= "OR UID $uid ";
    }
    $condition .= "OR UID $uids->[0] UID $uids->[1]";
    return $self->select_threaded_messages ($condition);
}


sub slice_info
{
    my $self      = shift;
    my $thickness = shift;
    my $slice_num = shift;
    my $uid_array = shift;
    
    my $first_slice_num = 1;
    my $last_slice_num  = scalar @{$uid_array} / $thickness;
    $last_slice_num = int ($last_slice_num) + 1
        unless ($last_slice_num == int ($last_slice_num));
    
    my $lower_bound;
    my $upper_bound;
    
    # if there is only one slice, then things are easy enough
    if ($first_slice_num == $last_slice_num)
    {
	$lower_bound = 0;
	$upper_bound = $#{$uid_array};
    }
    
    # if there is more than one slice, but we're on the first slice
    elsif ($slice_num == $first_slice_num)
    {
	$lower_bound = 0;
	$upper_bound = $slice_num * $thickness - 1;
    }
    
    # if there is more than one slice, but we're on the last slice
    elsif ($slice_num == $last_slice_num)
    {
	$lower_bound = ($slice_num - 1) * $thickness;
	$upper_bound = $#{$uid_array};
    }
    
    # we're somewhere in the middle
    else
    {
	$lower_bound = ($slice_num - 1) * $thickness;
	$upper_bound = $slice_num * $thickness - 1;
    }
    
    my @slice = @{$uid_array}[$lower_bound .. $upper_bound];
    return {
	number     => $slice_num,
	thickness  => $thickness,
	uids       => \@slice,
    };
}


sub slicing_structure
{
    my $self = shift;
    my $thickness = shift;
    my $slice_num = shift;
    my $uid_array = shift;
    
    my $number_of_slices = scalar @{$uid_array} / $thickness;
    $number_of_slices = int ($number_of_slices) + 1
        if (int ($number_of_slices) != $number_of_slices);
    
    my @struct = ();
    for (1..$number_of_slices)
    {
	my $slice = $self->slice_info ($thickness, $_, $uid_array);
	$slice->{current} = ($slice_num == $_) ? 1 : 0;
	push @struct, $slice;
    }
    
    unless (scalar @struct)
    {
	push @struct, {
	    number    => 1,
	    thickness => $thickness,
	    uids      => [],
	};
    }
    
    return (wantarray) ? @struct : \@struct;
}


sub uid_most_recent_first
{
    my $self = shift;
    my $imap = $self->imap;
    my @res  = $imap->sort ('REVERSE DATE', 'US-ASCII', 'ALL');
    return (wantarray) ? @res : \@res;
}


sub _messages_parentify
{
    my $self    = shift;
    my $message = shift;
    my $parent  = shift;
    
    if (defined $parent) { $message->{parent} = "$parent" }
    foreach (@{$message->{children}}) { $self->_messages_parentify ($_, $message) }
}


sub _messages_threaded
{
    my $self = shift;
    my $thread_info_string = shift;
    my @tokens = map { (defined $_) ? $_ : () } $thread_info_string =~ /(\()|(\))|(\d+)/g;
    
    # a bit of initialization...
    %Threads = ();
    $Current = new flo::plugin::Discussion::Message ( children => [] );
    @Stack = ($Current);
    
    # builds the tree using the tokens
    my $first = $Current;
    foreach (@tokens)
    {
	if (/\(/)    { $self->_messages_threaded_open_parenthesis()  }
	elsif (/\)/) { $self->_messages_threaded_close_parenthesis() }
	else         { $self->_messages_threaded_number ($_)         }
    }
    
    $self->_messages_threaded_collapse_empty_threads ($first);
    return $first;
}


sub _messages_threaded_collapse_empty_threads
{
   my $self = shift;
   my $thread = shift;

   return ($thread) unless (defined $thread->{children});
   
   my @new_children = ();
   push @new_children, $self->_messages_threaded_collapse_empty_threads ($_) for (@{$thread->{children}});
   $thread->{children} = \@new_children;
   
   (defined $thread->{uid}) ? return ($thread) : return @{$thread->{children}};
}


sub _messages_threaded_open_parenthesis
{
    my $self = shift;
    my $message = new flo::plugin::Discussion::Message;
    $Current->{children} ||= [];
    push @{$Current->{children}}, $message;
    $Current = $message;
    push @Stack, $message;
}


sub _messages_threaded_close_parenthesis
{
    my $self = shift;
    pop (@Stack);
    $Current = $Stack[$#Stack];
}


sub _messages_threaded_number
{
    my $self  = shift;
    my $number = shift;
    if (not defined $Current->uid)
    {
	$Current->{uid} = $number;
    }
    else
    {
	my $message = new flo::plugin::Discussion::Message ($number);
	$Current->{children} ||= [];
	push @{$Current->{children}}, $message;
	$Current = $message;
    }
}


1;
