# -------------------------------------------------------------------------------------
# flo::plugin::Search
# -------------------------------------------------------------------------------------
# Author : Jean-Michel Hiver <jhiver@mkdoc.com>
# Copyright : (c) MKDoc Holdings Ltd, 2003
#
# A plugin that deals with search requests and builds search result pages
# -------------------------------------------------------------------------------------
package flo::plugin::Search;
use MKDoc::Stemmer;
use MKDoc::Config;
use MKDoc::Error;
use flo::Standard;
use strict;
use warnings;
use 5.008_000;
use Carp;

use base qw /flo::Plugin/;

sub template_path { 'search' }

sub _name_default { '.search.html' }

sub run
{
    my $self = shift;
    local $MKDoc::CGI::FULL_URI;
    $MKDoc::CGI::FULL_URI = 1;
    $self->render_http (
	self       => $self,
	__input__  => 'XML',
	__output__ => 'XHTML',
       );
    
    return 'TERMINATE';
}


sub keywords
{
    my $self  = shift;
    my $cgi   = flo::Standard::cgi();
    my $words = $cgi->param ('q') || '';
    $self->{'.keywords'} = MKDoc::Stemmer::split ($words);
    return wantarray ? @{$self->{'.keywords'}} : $self->{'.keywords'};
}


sub keywords_prepared
{
    my $self  = shift;
    my @keywords = $self->keywords();
    my $where = 'WHERE (Document_Index.Keyword=?';
    shift (@keywords);
    while (@keywords)
    {
	shift (@keywords);
	$where .= ' OR Document_Index.Keyword=?';
    }
    $where .= ')';
    return $where;
}


sub number_of_results
{
    my $self = shift;
    $self->results();
    return $self->{'.number_of_results'};
}


sub results
{
    my $self = shift;
    my @keywords = $self->keywords();
    scalar @keywords || return [];
    
    $self->{'.results'} ||= do {
	my $limit  = $self->per_page();
	my $offset = ($self->page() - 1) * $limit;
	
	my $dbh   = lib::sql::DBH->get();
	my $narrw = $self->narrow_prepared();
	my $where = $self->keywords_prepared();
	my $sql   = <<EOF;
SELECT
  Document.ID,
  SUM(CASE Document_Index.Column_Name
      WHEN 'Publisher'   THEN 3
      WHEN 'Coverage'    THEN 3
      WHEN 'Relation'    THEN 3
      WHEN 'Lang'        THEN 3
      WHEN 'Title'       THEN 9
      WHEN 'Keywords'    THEN 8
      WHEN 'Description' THEN 6
      WHEN 'Name'        THEN 7
      WHEN 'Full_Path'   THEN 5
      WHEN 'Rights'      THEN 3
      ELSE 1 END) As Score
FROM Document, Document_Index
$where $narrw AND Document.ID = Document_Index.Record_ID
GROUP BY (ID)
ORDER BY Score DESC
-- LIMIT $offset, $limit
EOF
	
	my $sth = $dbh->prepare ($sql);
	push @keywords, $self->narrow() . '%' if ($self->narrow());
	$sth->execute (@keywords);
	
	my $doc_t = flo::Standard::table ('Document');
	my @res = ();
	while (my $arryref = $sth->fetchrow_arrayref())
	{
	    my $doc = $doc_t->get ($arryref->[0]);
	    $doc->{score} = $arryref->[1];
	    push @res, $doc if ($doc->is_showable());
	}
	
	$self->{'.number_of_results'} = scalar @res;
	@res = splice @res, $offset, $limit;
	\@res;
    };
    
    return wantarray ?
        @{$self->{'.results'}} :
	$self->{'.results'};
}


sub per_page
{
    my $self = shift;
    $self->{'.per_page'} ||= do {
	no warnings;
	my $cgi  = flo::Standard::cgi();
	my $per_page = 0 + $cgi->param ('per_page');
	($per_page > 0) ? $per_page : $self->config()->{'per_page'};
    };
    
    return $self->{'.per_page'};
}


sub page
{
    my $self = shift;
    $self->{'.page'} ||= do {
	no warnings;
	my $cgi  = flo::Standard::cgi();
	my $page = 0 + $cgi->param ('page');
	$page > 0 ? $page : 1;
    };
    
    return $self->{'.page'};
}


sub number_of_pages
{
    my $self = shift;
    $self->{'.number_of_pages'} ||= do {
        my $number_of_results = $self->number_of_results() || '0';
        my $per_page          = $self->per_page()          || '10';
	my $number_of_pages   = $number_of_results / $per_page;
	(int ($number_of_pages) == $number_of_pages) ?
            $number_of_pages :
	    int ($number_of_pages + 1);
    };
    
    return $self->{'.number_of_pages'};
}


sub narrow_prepared
{
    my $self = shift;
    my $cgix = flo::Standard::cgi()->new();
    return ($cgix->param ('narrow')) ?
        'AND Document.Full_Path LIKE ?' :
	'';
}


sub narrow
{
    my $self = shift;
    my $cgix = flo::Standard::cgi()->new();
    return $cgix->param ('narrow');
}


sub navbar
{
    my $self = shift;
    $self->{'.navbar'} ||= do {
	my $number_of_pages = $self->number_of_pages();
	my $current_page    = $self->page();
	
	my $left_space  = $current_page - 1;
	my $right_space = $number_of_pages - $current_page;
	$left_space  = 10 if ($left_space > 10);
	$right_space = 10 if ($right_space > 10);
    
	my $cgix = flo::Standard::cgi()->new;
	my @res  = ();
	
	foreach my $page ( $current_page - $left_space .. $current_page - 1)
	{
	    $cgix->param ('page', $page);
	    push @res, { page => $page, uri => $cgix->self_url };
	}
	
	push @res, { page => $current_page, uri => '' };
	
	foreach my $page ( $current_page + 1 .. $current_page + $right_space )
	{
	    $cgix->param ('page', $page);
	    push @res, { page => $page, uri => $cgix->self_url };
	}
	
	(@res == 1) ? [] : \@res;
    };
    
    return wantarray ?
        @{$self->{'.navbar'}} :
	$self->{'.navbar'};
}


1;


__END__
