#!/usr/bin/perl
package MKDoc::Misc::Newsletter;
use strict;
use warnings;
use Text::Wrap;
use MKDoc;
use flo::Record::Document;
use Encode;
use base qw /flo::Plugin/;
use Petal::Mail;

sub template_path { '/newsletter' }


sub root
{
    my $document_t = flo::Standard::table ('Document');
    return $document_t->get (1);
}


sub user
{
    my $self = shift;
    return $self->{user};
}


sub send 
{
    my $self = shift;
    my $mail = new Petal::Mail (
        file => '/newsletter',
        language => $self->language() || 'en',
    );
    
    $mail->send (
	self       => $self,
	__input__  => 'XML',
	__output__ => 'XML',
    );
    
    return 1; 
}


sub results
{
    my $self = shift;    
    my $user_login = $self->user()->login();
    my $date_since = $self->compute_date_since();
    
    my $sql  = <<SQL;
SELECT DISTINCT Document.Title, Document.Date_Created, Document.Full_Path, Document.Description,
                Editor.First_Name, Editor.Family_Name, Editor.Login,
                Creator.First_Name as Creator_First_Name, Creator.Family_Name as Creator_Family_Name
FROM   Document, Document_Audience, Audience, Preference_Audience, Editor, Editor as Creator, Preference_Language
WHERE
        -- join the tables together
        (
                Preference_Language.Language_ID = Document.Lang         AND
                Preference_Language.Editor_ID = Editor.ID               AND
                Preference_Audience.Audience_ID = Audience.ID           AND
                Preference_Audience.Editor_ID = Editor.ID               AND
                Document_Audience.Audience_ID = Audience.ID             AND
                Document_Audience.Document_ID = Document.ID		AND
		Document.Editor_Created_ID = Creator.ID
        )
AND
        -- limit to the current editor
        Editor.Login = ?
AND
        (
                -- get the wanted languages
                Preference_Language.Language_ID = Document.Lang
        OR
                -- get the wanted audiences
                Preference_Audience.Value       = 1
        )
AND
        -- remove languages which are not wanted
        Preference_Language.Value = 1
AND
        -- get only documents which are recent enough
       Document.Date_Created > ?

ORDER BY Document.Full_Path
SQL

    my $dbh = lib::sql::DBH->get();
    my $sth = $dbh->prepare_cached ($sql);
    $sth->execute ($user_login, $date_since);
    
    my @res = ();
    while (my $h = $sth->fetchrow_hashref)
    {
	my $doc = bless $h, 'flo::Record::Document';
	next unless ($doc->is_showable());
	
	Encode::_utf8_on ($h->{Description});
	Encode::_utf8_on ($h->{Title});
	$Text::Wrap::columns = 72;
	$Text::Wrap::columns = 72;
	$h->{Full_Path} =~ s/^\//$::PUBLIC_DOMAIN/;
	$h->{Date_Created} =~ s/\s+.*//;
	$h->{Description} = join '', fill ('    ', '    ', $h->{Description});
	push @res, $h;
    }
    
    return \@res;
}


package MKDoc::Misc::Newsletter::Daily;
use strict;
use warnings;
our @ISA = qw /MKDoc::Misc::Newsletter/;

sub compute_date_since
{
    my $self = shift;
    my ($sec, $min, $hour, $mday, $mon, $year, $wday, $yday, $isdst) = localtime (time - 24 * 3600);
    $mon += 1;
    $year += 1900;
    return "$year-$mon-$mday";
}

sub is_daily { 1 }


package MKDoc::Misc::Newsletter::Weekly;
use strict;
use warnings;
our @ISA = qw /MKDoc::Misc::Newsletter/;

sub compute_date_since
{
    my $self = shift;
    my ($sec, $min, $hour, $mday, $mon, $year, $wday, $yday, $isdst) = localtime (time - 7 * 24 * 3600);
    $mon += 1;
    $year += 1900;
    return "$year-$mon-$mday";
}

sub is_weekly { 1 }


package MKDoc::Misc::Newsletter::Monthly;
use strict;
use warnings;
our @ISA = qw /MKDoc::Misc::Newsletter/;

sub compute_date_since
{
    my $self = shift;
    my ($sec, $min, $hour, $mday, $mon, $year, $wday, $yday, $isdst) = localtime (time);
    $year += 1900;
    ($mon == 0) and do {
	$year--;
	$mon = 12;
    };
    
    return "$year-$mon-$mday";
}

sub is_monthly { 1 }



package main;
use strict;
use warnings;
use Carp;
use MKDoc::Handler::Initialize;
use Text::Wrap;

use constant MONTHLY => 'newsletter-monthly';
use constant WEEKLY  => 'newsletter-weekly';
use constant DAILY   => 'newsletter-daily';

our $PERIODICITY;
our $DBH;

$::PUBLIC_DOMAIN ||= $ENV{PUBLIC_DOMAIN} || die 'PUBLIC_DOMAIN is not defined';
 

MKDoc::Handler::Initialize::handler();

main();


sub main
{
    my $newsletter  = undef;
    my $periodicity = periodicity();
    
    print "Sending newsletter with periodicity: $periodicity\n"; 
    foreach my $user (users_which_want_newsletter())
    {
	print "\n\t$user->{Login}";
	my $newsletter = instantiate_newsletter ($user) || next;
        $newsletter->send();
        print "\t$user->{First_Name} $user->{Family_Name} <$user->{Email}>";
    }
}


sub instantiate_newsletter
{
    my $user = shift;
    my $periodicity = periodicity();
    my %args = ( user => $user );
    my $news = undef;
    ($periodicity eq DAILY)   and $news = new MKDoc::Misc::Newsletter::Daily (%args);
    ($periodicity eq WEEKLY)  and $news = new MKDoc::Misc::Newsletter::Weekly (%args);
    ($periodicity eq MONTHLY) and $news = new MKDoc::Misc::Newsletter::Monthly (%args);
    
    my $res = $news->results();
    @{$res} or return;
    return $news;
}


sub periodicity
{
    $PERIODICITY ||= do {
	my $res = undef;
	$_ = shift (@ARGV) || 'none';
	/^(day|week|month)$/ or die "Usage: $0 (day|week|month)";
	/day/   and $res = DAILY;
	/week/  and $res = WEEKLY;
	/month/ and $res = MONTHLY;
	$res;
    };
    
    return $PERIODICITY;
}


sub users_which_want_newsletter
{
    my $sql = join "\n",
        qq |SELECT DISTINCT Editor.*                 |,
	qq |FROM   Editor, Preference                |,
	qq |WHERE                                    |,
	qq |    Preference.Editor_ID = Editor.ID AND |,
	qq |    Preference.Name = ?              AND |,
	qq |    Preference.Value > 0                 |;
    
    my $dbh = lib::sql::DBH->get();
    my $sth = $dbh->prepare_cached ($sql);
    $sth->execute (periodicity());

    my $query = new lib::sql::Query
        sth => $sth,
	bless_into => 'flo::Record::Editor';
    
    my @res = $query->fetch_all();
    @res = map { $_->is_confirmed() ? $_ : () } @res;
    return wantarray ? @res : \@res;
}


1;


__END__
