# -------------------------------------------------------------------------------------
# MKDoc::HTTP::Response
# -------------------------------------------------------------------------------------
# Author : Jean-Michel Hiver <jhiver@mkdoc.com>.
# Copyright : (c) Jean-Michel Hiver, 2003.
#
# A package which formats HTTP responses.
#
#   my $response = new MKDoc::HTTP::Response();
#   $response->Status ('302 Found');
#   $response->Location ('http://example.com');
#   print $response->get();
# -------------------------------------------------------------------------------------
package MKDoc::HTTP::Response;
use strict;
use warnings;
our $AUTOLOAD;


##
# $class->new ($string or @args);
# -------------------------------
# Instanciates an MKDoc::HTTP::Response object from either a $string
# as returned by $self->get(), or @args which is a hash.
#
#    my $response = new MKDoc::HTTP::Response
#        Status       => '200 OK',
#        Set-Cookie   => 'EvilCookie',
#        Content-Type => 'text/plain',
#        BODY         => 'Hello, World!';
#
#    my $clone = new MKDoc::HTTP::Response ($response->get());
##
sub new
{
    my $class = shift;
    @_ == 1 and return $class->_new_from_string (@_);
    return $class->_new_from_args (@_);
}


sub _new_from_string
{
    my $class = shift;
    my $lines = shift;
    my @lines = split /\n/, $lines;
    my $self  = bless {}, $class;
    while (my $line = shift (@lines))
    {
	chomp ($line);
	chomp ($line);
	last unless ($line);
	
	my ($key, $value) = $line =~ /^(.*?)\:\s*(.*?)\s*$/;
	defined $key || next;
	$self->$key ($value);
    }
    
    $self->Body (join "\n", @lines);
    return $self;
}


sub _new_from_args
{
    my $class = shift;
    my $args  = shift;
    my $self  = bless {}, $class;
    $self->Body (delete $args->{'Body'});
    while (my ($key, $value) = each %{$args})
    {
	$self->$key ($value);
    }
    return $self;
}


##
# $self->HEAD();
# --------------
# Returns the HEAD of the HTTP query.
#
#     $ENV{REQUEST_METHOD} =~ /HEAD/ and print $response->head();
##
sub HEAD
{
    my $self = shift;

    my @res  = ();
    my $status = $self->Status() || '200 OK';
    push @res, "Status: $status";
    foreach my $key (sort $self->header_keys())
    {
	my $val = $self->{$key};
	my @val = ref $val ? @{$val} : $val;
	next unless ($val);
	foreach (@val)
	{
	    push @res, "$key: $_";
	}
    }
    
    my $res = join "\n", @res;
    $res .= "\n\n";
}


sub header_keys
{
    my $self = shift;
    return map { ($_ !~ /^(?:Status|Body)$/) ? $_ : () } keys %{$self};
}


##
# $self->GET();
# -------------
# Returns the HEAD plus the Body of the HTTP query.
##
sub GET
{
    my $self = shift;
    return $self->HEAD() . $self->Body();
}


##
# $self->Status();
# ----------------
# Setter / Getter for the response status code.
##
sub Status
{
    my $self = shift;
    $self->{Status} = shift if (@_);
    return $self->{Status} || '200 OK';
}


##
# $self->Body();
# --------------
# Setter / Getter for the message body.
##
sub Body
{
    my $self = shift;
    $self->{Body} = shift if (@_);
    return $self->{Body} || '';
}


sub clear
{
    my $self = shift;
    for (keys %{$self}) { delete $self->{$_} };
}


##
# $self->Xxx();
# -------------
# Setter / Getter for any other header.
##
sub AUTOLOAD
{
    my $self = shift;
    my ($pkg, $meth) = $AUTOLOAD =~ /(.*)::(.*)/;
    
    if ($meth =~ /^delete_/)
    {
	$meth =~ s/delete_//g;
	$meth =~ s/_/-/g;
	return delete $self->{$meth};
    }
    elsif ($meth =~ /^[A-Z]/)
    {
	$meth =~ s/_/-/g;
	$self->{$meth} = shift if (@_);
	return $self->{$meth};
    }
    else
    {
	use Carp;
	confess qq |Can't locate object method "$meth" via package "$pkg"|;
    }
}


sub DESTROY
{
}


1;


__END__
