# -----------------------------------------------------------------------------
# MKDoc::Site::Check::Redirect
# -----------------------------------------------------------------------------
#       Author : Jean-Michel Hiver <jhiver@mkdoc.com>
#    Copyright : Copyright (c) 2002 MKDoc Holdings Ltd.  All rights reserved.
#
#   Description:
#
#       Checks for consistency on the 'Redirect' table
#
# -----------------------------------------------------------------------------
package MKDoc::Site::Check::Redirect;
use flo::Standard;
use strict;
use warnings;
use Carp;


##
# $class->check();
# ----------------
# Checks for inconsistencies on the Redirect table.
##
sub check
{
    my $self = shift;
    $self->_check_redirect_source_non_existent();
    $self->_check_redirect_target_existent();
    $self->_check_redirect_not_redundant();
}


##
# $self->_redirect_table();
# -------------------------
# Returns the 'Redirect' table object.
# Internal use.
##
sub _redirect_table
{
    confess $@ if (defined $@ and $@);
    return flo::Standard::table ('Redirect');
}


##
# $self->_document_table();
# -------------------------
# Returns the 'Document' table object.
# Internal use.
##
sub _document_table
{
    confess $@ if (defined $@ and $@);
    return flo::Standard::table ('Document');
}


##
# $self->_check_redirect_source_non_existent();
# ---------------------------------------------
# Checks that all the 'Old_Path' attributes of the
# existing redirect are different from the 'Full_Path'
# attributes of the existing documents.
#
# Depending on $fixMe, deletes or warns about redirects
# for which this condition is not met.
##
sub _check_redirect_source_non_existent
{
    my $self = shift;
    my $redirect_t = $self->_redirect_table();
    my $document_t = $self->_document_table();
    
    my @redirects  = $self->_all_redirects();
    foreach my $redirect (@redirects)
    {
	$document_t->get ( Full_Path => $redirect->old_path() ) and do {
	    $self->_delete_redirect (
		( join '', ( $redirect->old_path(),
			     ' => ',
			     $redirect->new_path(),
			     ' : ',
			     $redirect->old_path(),
			     ' is an existing document' ) ),
		$redirect,
	       );
	};
    }
}


##
# $self->_check_redirect_target_existent();
# -----------------------------------------
# Checks that all the 'New_Path' attributes of the
# existing redirect reference documents which actually
# exist.
#
# Depending on $fixMe, deletes or warns about redirects
# for which this condition is not met.
##
sub _check_redirect_target_existent
{
    my $self = shift;
    my $redirect_t = $self->_redirect_table();
    my $document_t = $self->_document_table();
    my @redirects  = $self->_all_redirects();
    foreach my $redirect (@redirects)
    {
	!$document_t->get ( Full_Path => $redirect->new_path() ) and do {
	    $self->_delete_redirect (
		"[" . $redirect->id() . "] New_Path => " . $redirect->new_path() . " is not a document",
		$redirect,
	       );
	};
    }
}


##
# $self->_check_redirect_not_redundant();
# ---------------------------------------
# Checks that there's not redundant link, i.e.
# /foo/     => /hello/
# /foo/bar/ => /hello/bar/ # redundant !!!
#
# Deletes them if $fixMe is set to TRUE,
# Skips the method otherwise.
##
sub _check_redirect_not_redundant
{
    my $self = shift;
    my $redirect_t = $self->_redirect_table();
    
    my @redirects  = $self->_all_redirects();
    foreach my $redirect (@redirects)
    {
	my $redirect_from = $redirect->old_path();
	my $redirect_to   = $redirect->new_path();
	next unless ($redirect_from =~ /\/.*\// and $redirect_to =~ /\/.*\//);
	
	my $parent_redirect_from = $self->_parent_path ($redirect_from);
	my $parent_redirect_to = $self->_parent_path ($redirect_to);
	my $parent_redirect = $redirect_t->get (
	    Old_Path => $parent_redirect_from,
	    New_Path => $parent_redirect_to,
	   );
	
	next unless ($parent_redirect);
	
	$self->_delete_redirect (
	    "[$redirect_from] is redundant with [$parent_redirect_from]",
	    $redirect,
	   );
    }
}


##
# $self->_all_redirects();
# ------------------------
# Returns all redirects in proper order for
# _check_redirect_not_redundant();
##
sub _all_redirects
{
    my $self = shift;
    my $redirect_t = $self->_redirect_table();
    my @redirects = sort {
	length ( $b->old_path() ) <=>
	length ( $a->new_path() )
    } $redirect_t->search()->fetch_all();
    return wantarray ? @redirects : \@redirects;
}


##
# $self->_parent_path ($path);
# ----------------------------
# Returns the parent path of a given path,
# i.e. /foo/bar/ => /foo/
##
sub _parent_path
{
    my $self = shift;
    my $path = shift;
    $path =~ s/^\///;
    $path =~ s/\/$//;
    my @path = split /\//, $path;
    pop (@path);
    $path = join '/', @path;
    $path = "/$path/";
    return $path;
}


##
# $self->_delete_redirect ($message, $redirect);
# ----------------------------------------------
# Prints $message and deletes the $redirect object.
##
sub _delete_redirect
{
    my $self = shift;
    my $msg = shift;
    my $redirect = shift;
    print "DELETE FROM Redirect WHERE ID=$redirect->{ID};\n";
    $self->_print_line ($msg);
}


##
# $self->_print_line ($line);
# ---------------------------
# Prints $line followed by a carriage return.
##
sub _print_line
{
    my $self = shift;
    my $line = shift;
    print STDERR $line . "\n";
}


sub _print_info { return shift->_print_line (@_) };


1;
