# -----------------------------------------------------------------------------
# MKDoc::Site::ConfigUI
# -----------------------------------------------------------------------------
# Author : Jean-Michel Hiver <jhiver@mkdoc.com>
# Copyright : Copyright (c) 2003 MKDoc Holdings Ltd.  All rights reserved.
#
# This is a module that displays an interactive menu which can be used
# to let a user configure an MKDoc site base parameters. At the moment
# it is pretty monolithic but it should be modularized to allow further
# extension.
# -----------------------------------------------------------------------------
package MKDoc::Site::ConfigUI;
use strict;
use warnings;
use Carp;


our $SAVE_TO = undef;


sub new
{
    my $class = shift->class;
    return bless { @_ }, $class;
}


sub is_ok
{
    my $self = shift;
    return $self->{is_ok};
}


##
# $class->configure ($file);
# --------------------------
# Starts the configuration state machine and writes
# the results in $file once it's complete.
##
sub configure
{
    my $class = shift;
    my $file  = shift || $class->_error ("No file to configure!");
 
    (-e $file) ?
    $class->print_screen ("file $file exists. Skipping.") :
    do {
	$SAVE_TO = $file;
	my $config_ui = $class->new ( $class->_new_attributes );
	$config_ui->run();
    };
}


##
# $class->_new_attributes;
# ------------------------
# Finds out what are the attributes which needs to be
# passed to the constructor by looking in @ARGV and
# in the environment variables.
##
sub _new_attributes
{
    my $class = shift;
    return (
	mkdoc_dir          => $class->argv_param ('mkdoc_dir') || $ENV{MKDOC_DIR} || '/var/www/mkdoc-1-6',
	site_dir           => $class->argv_param ('prefix') || $ENV{SITE_DIR} || '/var/www/sites/example',
	public_domain      => $class->argv_param ('public_domain') || $ENV{PUBLIC_DOMAIN} || 'http://www.example.com/',
	user_domain        => $class->argv_param ('user_domain') || $ENV{USER_DOMAIN} || 'http://users.example.com/',
	server_admin       => $class->argv_param ('server_admin') || $ENV{SERVER_ADMIN} || 'webmaster@example.com',
	database_name      => $class->argv_param ('database_name') || $ENV{DATABASE_NAME} || 'test',
	database_host      => $class->argv_param ('database_host') || $ENV{DATABASE_HOST} || undef,
	database_port      => $class->argv_param ('database_port') || $ENV{DATABASE_PORT} || undef,
	database_user      => $class->argv_param ('database_user') || $ENV{DATABASE_USER} || 'root',
	database_pass      => $class->argv_param ('database_pass') || $ENV{DATABASE_PASS} || undef,
	organization_name  => $class->argv_param ('organization_name')  || $ENV{ORGANIZATION_NAME} || 'Your Organization',
	organization_email => $class->argv_param ('organization_email') || $ENV{ORGANIZATION_EMAIL} || 'info@example.com',
	user_login         => $class->argv_param ('user_login') || $ENV{USER_LOGIN} || 'fred',
	user_first_name    => $class->argv_param ('user_first_name') || $ENV{USER_FIRST_NAME} || 'Fred',
	user_last_name     => $class->argv_param ('user_last_name') || $ENV{USER_LAST_NAME} || 'Flintstone',
	user_pass          => $class->argv_param ('user_pass') || $ENV{USER_PASS} || 'rocks',
	user_email         => $class->argv_param ('user_email') || $ENV{USER_EMAIL} || 'fred@example.com',
	use_imap           => $class->argv_param ('use_imap') || $ENV{USE_IMAP} || 'no',
	imap_host          => $class->argv_param ('imap_host') || $ENV{IMAP_HOST} || undef,
	imap_port          => $class->argv_param ('imap_port') || $ENV{IMAP_PORT} || undef,
	imap_user          => $class->argv_param ('imap_user') || $ENV{IMAP_USER} || undef,
	imap_pass          => $class->argv_param ('imap_pass') || $ENV{IMAP_PASS} || undef,
	email_from         => $class->argv_param ('email_from') || $ENV{EMAIL_FROM} || 'mail_from@example.com',
	email_bcc          => $class->argv_param ('email_bcc') || $ENV{EMAIL_BCC} || 'mail_bcc@example.com',
	state              => 'menu_main',
       );
}


sub _set_instance_vars
{
    my $self = shift;
    my $args = shift;
    $self->{$_} = $args->{$_} for (keys %{$args});
    return $self;
}


##
# $class->argv_param;
# -------------------
# Extracts a parameter from the @ARGV array, in the form
# key=value
##
sub argv_param
{
    my $class = shift->class;
    my $param_name = shift;
    my ($string)   = grep /^$param_name=/, @ARGV;
    return unless (defined $string and $string);
    
    ($string) = $string =~ /^$param_name=(.*)/;
    return $string;
}


sub class
{
    my $class = shift;
    return ref $class || $class;
}

##
# Instance Methods
# =============================================================================
##

##
# $self->run();
# --------------
# This is the main loop of the FSM.
##
sub run
{
    my $self = shift;
    while ($self->{state} ne 'exit')
    {
	my $method = $self->{state};
	$self->$method();
    }
}


##
# $self->save();
# --------------
# Saves the object's attributes into the file which is given
# by $self->save_filename() and changes the state to 'exit'
# which will stop the FSM.
##
sub save
{
    my $self = shift;
    for (keys %{$self})
    {
	my $uc_key = uc ($_);
	$self->{$uc_key} = $self->{$_};
    }
    #my $file = $self->save_filename;
    #my $state = delete $self->{state};
    #open FP, ">$file" || confess "Cannot write-open $file: $!";
    #print FP join "\n", map {
	#( defined $_ and defined $self->{$_} ) ?
	#uc ($_) . "\t" . $self->{$_} :
	#()
    #} keys %{$self};
    #close FP;
    
    $self->{is_ok} = 1;
    $self->{state} = 'exit';
}


##
# $self->save_filename();
# -----------------------
# Returns the path to which the config file needs to be saved,
# including the name of the file itself.
##
sub save_filename { return $SAVE_TO }



##
# Menu methods
# -----------------------------------------------------------------------------
# These methods display menu options and prompt the user for choices,
# changing the FSM's state to whatever is appropriate in compliance
# with the user actions.
##


##
# $self->menu_main();
# -------------------
# Displays the main menu and prompts the user for choices.
##
sub menu_main
{
    my $self = shift;

    $self->clear_screen();    
    $self->print_screen (
	"1 - Basic MKDoc configuration",
	"2 - Database connection",
	"3 - Organization details",
	"4 - First user details",
#	"5 - IMAP discussion board",
#	"6 - User accounts / Personalization",
	"",
	""
       );
    
    $self->print_screen (
	"S - Save Makefile.PL and exit",
	"X - eXit without without saving",
	"",
	"Input Choice: "
       );
    
    my $line = $self->get_stdin_line();
    
    if    ($line eq '1')                 { $self->{state} = 'menu_basic'           }
    elsif ($line eq '2')                 { $self->{state} = 'menu_database'        }
    elsif ($line eq '3')                 { $self->{state} = 'menu_organization'    }
    elsif ($line eq '4')                 { $self->{state} = 'menu_user'            }
#    elsif ($line eq '5')                 { $self->{state} = 'menu_imap'            }
#    elsif ($line eq '6')                 { $self->{state} = 'menu_email'           }
    elsif ($line eq 'x' or $line eq 'X') { $self->{state} = 'exit_without_install' }
    elsif ($line eq 's' or $line eq 'S') { $self->{state} = 'check_params'         }
}


##
# $self->menu_basic();
# --------------------
# Displays the menu to change basic configuration options
# and prompts the user for choices.
##
sub menu_basic
{
    my $self = shift;

    $self->clear_screen();
    $self->print_screen (
	"1 - MKDoc directory    : " . ($self->{mkdoc_dir}           || '(undefined)'),
	"2 - Site directory     : " . ($self->{site_dir}            || '(undefined)'),
	"3 - Public domain      : " . ($self->{public_domain}       || '(undefined)'),
	"4 - User domain        : " . ($self->{user_domain}         || '(undefined)'),
	"5 - Server admin       : " . ($self->{server_admin}        || '(undefined)'),
	"",
	""
       );
    $self->print_screen (
	"B - Go back to main menu",
	"D - Delete an option",
	"",
	"Input Choice: "
       );
    
    my $line = $self->get_stdin_line();
    
    if    ($line eq 'b' or $line eq 'B') { $self->{state} = 'menu_main'          }
    elsif ($line eq 'd' or $line eq 'D') { $self->{state} = 'delete_basic'       }
    elsif ($line eq '1')                 { $self->{state} = 'set_mkdoc_dir'      }
    elsif ($line eq '2')                 { $self->{state} = 'set_prefix'         }
    elsif ($line eq '3')                 { $self->{state} = 'set_public_domain'  }
    elsif ($line eq '4')                 { $self->{state} = 'set_user_domain'    }
    elsif ($line eq '5')                 { $self->{state} = 'set_server_admin'   }
}


##
# $self->menu_database();
# -----------------------
# Displays the menu to change database configuration options
# and prompts the user for choices.
##
sub menu_database
{
    my $self = shift;
    
    $self->clear_screen();
    $self->print_screen (
	"1 - Database name      : " . ($self->{database_name}       || '(undefined)'),
	"2 - Database host      : " . ($self->{database_host}       || '(undefined)'),
	"3 - Database port      : " . ($self->{database_port}       || '(undefined)'),
	"4 - Database user      : " . ($self->{database_user}       || '(undefined)'),
	"5 - Database pass      : " . ($self->{database_pass}       || '(undefined)'),
	"",
	""
       );
    
    $self->print_screen (
	"B - Go back to main menu",
	"D - Delete an option",
	"",
	"Input Choice: "
       );
    
    my $line = $self->get_stdin_line();

    if    ($line eq 'b' or $line eq 'B') { $self->{state} = 'menu_main'            }
    elsif ($line eq 'd' or $line eq 'D') { $self->{state} = 'delete_database'    }
    elsif ($line eq '1')                 { $self->{state} = 'set_database_name'  }
    elsif ($line eq '2')                 { $self->{state} = 'set_database_host'  }
    elsif ($line eq '3')                 { $self->{state} = 'set_database_port'  }
    elsif ($line eq '4')                 { $self->{state} = 'set_database_user'  }
    elsif ($line eq '5')                 { $self->{state} = 'set_database_pass'  }
}


##
# $self->menu_organization();
# ---------------------------
# Displays the menu to change organization configuration options
# and prompts the user for choices.
##
sub menu_organization
{
    my $self = shift;
    $self->{organization_name} ||= $ENV{ORGANIZATION_NAME} || 'Example';
    
    $self->clear_screen();
    $self->print_screen (
	"1 - Organization name  : " . ($self->{organization_name}   || '(undefined)'),
	"2 - Organization email : " . ($self->{organization_email}  || '(undefined)'),
	"",
	""
       );
    $self->print_screen (
	"B - Go back to main menu",
	"D - Delete an option",
	"",
	"Input Choice: "
       );
    
    my $line = $self->get_stdin_line();

    if    ($line eq 'b' or $line eq 'B') { $self->{state} = 'menu_main'                }
    elsif ($line eq 'd' or $line eq 'D') { $self->{state} = 'delete_organization'    }
    elsif ($line eq '1')                 { $self->{state} = 'set_organization_name'  }
    elsif ($line eq '2')                 { $self->{state} = 'set_organization_email' }
}


##
# $self->menu_user();
# -------------------
# Displays the menu to change user configuration options
# and prompts the user for choices.
##
sub menu_user
{
    my $self = shift;
    
    $self->clear_screen();
    $self->print_screen (
	"1 - User login         : " . ($self->{user_login}          || '(undefined)'),
	"2 - User first name    : " . ($self->{user_first_name}     || '(undefined)'),
	"3 - User last name     : " . ($self->{user_last_name}      || '(undefined)'),
	"4 - User password      : " . ($self->{user_pass}           || '(undefined)'),
	"5 - User email         : " . ($self->{user_email}          || '(undefined)'),
	"",
       );
    $self->print_screen (
	"B - Go back to main menu",
	"D - Delete an option",
	"",
	"Input Choice: "
       );
    
    my $line = $self->get_stdin_line();

    if    ($line eq 'b' or $line eq 'B') { $self->{state} = 'menu_main'           }
    elsif ($line eq 'd' or $line eq 'D') { $self->{state} = 'delete_user'         }
    elsif ($line eq '1')                 { $self->{state} = 'set_user_login'      }
    elsif ($line eq '2')                 { $self->{state} = 'set_user_first_name' }
    elsif ($line eq '3')                 { $self->{state} = 'set_user_last_name'  }
    elsif ($line eq '4')                 { $self->{state} = 'set_user_pass'       }
    elsif ($line eq '5')                 { $self->{state} = 'set_user_email'      }
}


##
# $self->menu_imap();
# -------------------
# Displays the menu to change imap configuration options
# and prompts the user for choices.
##
sub menu_imap
{
    my $self = shift;
    
    $self->clear_screen();
    $self->print_screen (
	"1 - Use IMAP forums    : " . ($self->{use_imap}            || '(undefined)'),
	"2 - IMAP host          : " . ($self->{imap_host}           || '(undefined)'),
	"3 - IMAP port          : " . ($self->{imap_port}           || '(undefined)'),
	"4 - IMAP login         : " . ($self->{imap_user}           || '(undefined)'),
	"5 - IMAP password      : " . ($self->{imap_pass}           || '(undefined)'),
	"",
	""
       );
    $self->print_screen (
	"B - Go back to main menu",
	"D - Delete an option",
	"",
	"Input Choice: "
       );
    
    my $line = $self->get_stdin_line();
    
    if    ($line eq 'b' or $line eq 'B') { $self->{state} = 'menu_main'     }
    elsif ($line eq 'd' or $line eq 'D') { $self->{state} = 'delete_imap'   }
    elsif ($line eq '1')                 { $self->{state} = 'set_use_imap'  }
    elsif ($line eq '2')                 { $self->{state} = 'set_imap_host' }
    elsif ($line eq '3')                 { $self->{state} = 'set_imap_port' }
    elsif ($line eq '4')                 { $self->{state} = 'set_imap_user' }
    elsif ($line eq '5')                 { $self->{state} = 'set_imap_pass' }
}


##
# $self->menu_email();
# --------------------
# Displays the menu to change email configuration options
# and prompts the user for choices.
##
sub menu_email
{
    my $self = shift;
    
    $self->clear_screen();
    $self->print_screen (
	"1 - Email From : " . ($self->{email_from} || '(undefined)'),
	"2 - Email BCC  : " . ($self->{email_bcc}  || '(undefined)'),
	"",
	""
       );
    $self->print_screen (
	"B - Go back to main menu",
	"D - Delete an option",
	"",
	"Input Choice: "
       );
    
    my $line = $self->get_stdin_line();
    
    if    ($line eq 'b' or $line eq 'B') { $self->{state} = 'menu_main'        }
    elsif ($line eq 'd' or $line eq 'D') { $self->{state} = 'delete_email'   }
    elsif ($line eq '1')                 { $self->{state} = 'set_email_from' }
    elsif ($line eq '2')                 { $self->{state} = 'set_email_bcc'  }
}


##
# Delete prompt methods
# -----------------------------------------------------------------------------
##


##
# $self->delete_basic();
# ----------------------
# Deletes a value from the menu_basic() options.
##
sub delete_basic
{
    my $self = shift;        
    my $message = <<EOF;

Please enter the item number to undefine:
EOF
    
    my $line = $self->message_prompt ('', $message);
    if ($line eq '1')    { $self->{mkdoc_dir} = undef }
    elsif ($line eq '2') {
	@ARGV = map { /^PREFIX=/ ? () : $_ } @ARGV;
	$self->{site_dir} = undef;
    }
    elsif ($line eq '3') { $self->{public_domain}      = undef }
    elsif ($line eq '4') { $self->{user_domain}        = undef }
    elsif ($line eq '5') { $self->{server_admin}       = undef }
    
    $self->{state} = 'menu_basic';
}


##
# $self->delete_basic();
# ----------------------
# Deletes a value from the menu_database() options.
##
sub delete_database
{
    my $self = shift;
        
    my $message = <<EOF;

Please enter the item number to undefine:
EOF
    
    my $line = $self->message_prompt ('', $message);
    if    ($line eq '1') { $self->{database_name} = undef }
    elsif ($line eq '2') { $self->{database_host} = undef }
    elsif ($line eq '3') { $self->{database_port} = undef }
    elsif ($line eq '4') { $self->{database_user} = undef }
    elsif ($line eq '5') { $self->{database_pass} = undef }
    $self->{state} = 'menu_database';
}


##
# $self->delete_organization();
# -----------------------------
# Deletes a value from the menu_organization() options.
##
sub delete_organization
{
    my $self = shift;
        
    my $message = <<EOF;

Please enter the item number to undefine:
EOF
    
    my $line = $self->message_prompt ('', $message);
    if    ($line eq '1') { $self->{organization_name}  = undef }
    elsif ($line eq '2') { $self->{organization_email} = undef }
    $self->{state} = 'menu_organization';
}


##
# $self->delete_user();
# ---------------------
# Deletes a value from the menu_user() options.
##
sub delete_user
{
    my $self = shift;
        
    my $message = <<EOF;

Please enter the item number to undefine:
EOF
    
    my $line = $self->message_prompt ('', $message);
    if    ($line eq '1') { $self->{user_login}         = undef }
    elsif ($line eq '2') { $self->{user_first_name}    = undef }
    elsif ($line eq '3') { $self->{user_last_name}     = undef }
    elsif ($line eq '4') { $self->{user_pass}          = undef }
    elsif ($line eq '5') { $self->{user_email}         = undef }
    $self->{state} = 'menu_user';
}


##
# $self->delete_imap();
# ---------------------
# Delete a value from the menu_imap() options.
##
sub delete_imap
{
    my $self = shift;
        
    my $message = <<EOF;

Please enter the item number to undefine:
EOF
    
    my $line = $self->message_prompt ('', $message);

    if    ($line eq '1') { $self->{use_imap}           = 'no'  }
    elsif ($line eq '2') { $self->{imap_host}          = undef }
    elsif ($line eq '3') { $self->{imap_port}          = undef }
    elsif ($line eq '4') { $self->{imap_user}          = undef }
    elsif ($line eq '5') { $self->{imap_pass}          = undef }
    $self->{state} = 'menu_imap';
}


##
# $self->delete_email();
# ----------------------
# Delete a value from the menu_email() options.
##
sub delete_email
{
    my $self = shift;
        
    my $message = <<EOF;

Please enter the item number to undefine:
EOF
    
    my $line = $self->message_prompt ('', $message);
    
    if    ($line eq '1') { $self->{email_from} = undef }
    elsif ($line eq '2') { $self->{email_bcc}  = undef }
    $self->{state} = 'menu_email';
}


##
# Set attribute prompt methods
# -----------------------------------------------------------------------------
##


##
# $self->set_prefix;
# ------------------
# Sets the prefix, i.e. the site directory
##
sub set_prefix
{
    my $self = shift;
    
    my $value = shift;
    $self->{prefix} ||= $ENV{PREFIX} || '';
    $value ||= do {
	my $message = <<EOF;

Please enter the directory where you want to install your site.
Site directory [$self->{site_dir}] :
EOF
	
	$self->message_prompt ($self->{prefix}, $message);
    };
    
    $value =~ s/\/*$//;
    $self->{site_dir} = $value;
    $self->{state} = 'menu_basic';
}


##
# $self->set_mkdoc_dir();
# -----------------------
# Sets mkdoc codebase directory
##
sub set_mkdoc_dir
{
    my $self = shift;
        
    my $value = shift;
    $self->{mkdoc_dir} ||= $ENV{MKDOC_DIR} || '';
    $value ||= do {
	my $message = <<EOF;

Please enter the directory where MKDoc has been installed.
MKDoc directory dir [$self->{mkdoc_dir}] :
EOF
	
	$self->message_prompt ($self->{mkdoc_dir}, $message);
    };
    
    $value =~ s/\/*$//;
    $self->{mkdoc_dir} = $value;
    $self->{state} = 'menu_basic';
}


##
# $self->set_public_domain();
# ---------------------------
# Sets the public domain address for the site which
# is being set up.
##
sub set_public_domain
{
    my $self = shift;
        
    $self->{public_domain} ||= $ENV{PUBLIC_DOMAIN} || 'http://www.example.com/';
    my $message = <<EOF;

Your MKDoc website is going to have a public address for anonymous
users. You need to specify this address, including its protocol
(http or https).

Public domain address [$self->{public_domain}] :
EOF

    $self->{public_domain} = $self->message_prompt ($self->{public_domain}, $message);
    $self->{public_domain} =~ /^https?:\/\// or do { $self->{public_domain} = 'http://' . $self->{public_domain} };
    $self->{public_domain} =~ s/\/$//;
    $self->{public_domain} .= '/';    
    $self->{state} = 'menu_basic';
}


##
# $self->set_user_domain();
# -------------------------
# Sets the user domain address for the site which
# is being set up.
##
sub set_user_domain
{
    my $self = shift;
        
    $self->{user_domain} ||= $ENV{USER_DOMAIN} || 'http://users.example.com/';
    my $message = <<EOF;
Your MKDoc website is going to have an address for registered users
and editors. You need to specify this address, including its protocol
(http or https).

User domain address [$self->{user_domain}] :
EOF
    
    $self->{user_domain} = $self->message_prompt ($self->{user_domain}, $message);
    $self->{user_domain} =~ /^https?:\/\// or do { $self->{user_domain} = 'http://' . $self->{user_domain} };
    $self->{user_domain} =~ s/\/$//;
    $self->{user_domain} .= '/';
    $self->{state} = 'menu_basic';
}


##
# $self->set_admin_domain();
# --------------------------
# Sets the admin domain address for the site which
# is being set up.
##
sub set_admin_domain
{
    my $self = shift;
    $self->{admin_domain} ||= $ENV{ADMIN_DOMAIN} || 'http://admin.example.com/';
    my $message = <<EOF;

Your MKDoc website is going to have a administration address. You need
to specify this address, including its protocol (http or https).

Admin domain address [$self->{admin_domain}] :
EOF
    
    $self->{admin_domain} = $self->message_prompt ($self->{admin_domain}, $message);
    $self->{state} = 'menu_basic';
}


##
# $self->set_server_admin();
# --------------------------
# Sets the administrator email address for the
# site which is being set up.
##
sub set_server_admin
{
    my $self = shift;
    $self->{server_admin} ||= $ENV{SERVER_ADMIN} || 'admin@example.com';
    my $message = <<EOF;

Your MKDoc website needs an email address for its administrator.

Administrator address [$self->{server_admin}] :
EOF

    $self->{server_admin} = $self->message_prompt ($self->{server_admin}, $message);
    $self->{state} = 'menu_basic';
}


##
# $self->set_database_name();
# ---------------------------
# Sets the MySQL database name for the
# site which is being set up.
##
sub set_database_name
{
    my $self = shift;
    $self->{database_name} ||= $ENV{DATABASE_NAME} || 'test';
    my $message = <<EOF;

Your MKDoc website needs to use a MySQL database. You must pick a
MySQL database name for MKDoc to use.

MySQL database name [$self->{database_name}] :
EOF

    $self->{database_name} = $self->message_prompt ($self->{database_name}, $message);
    $self->{state} = 'menu_database';
}


##
# $self->set_database_host();
# ---------------------------
# Sets the MySQL database host for the
# site which is being set up.
##
sub set_database_host
{
    my $self = shift;
    $self->{database_host} ||= $ENV{DATABASE_HOST} || '';
    my $message = <<EOF;

Your MKDoc website needs an optional hostname where the MySQL
database is hosted. If the MySQL database which you intend to
use is running on the same machine as the MKDoc code, just
press enter.

MySQL database host [$self->{database_host}] :
EOF

    $self->{database_host} = $self->message_prompt ($self->{database_host}, $message);
    $self->{state} = 'menu_database';
}


##
# $self->set_database_port();
# ---------------------------
# Sets the MySQL database port for the
# site which is being set up.
##
sub set_database_port
{
    my $self = shift;
    $self->{database_port} ||= $ENV{DATABASE_PORT} || '';
    my $message = <<EOF;

Your MKDoc website needs an optional port where the MySQL
database accepts incoming connections... If the MySQL database
which you intend to use is running on its standard port, just
press enter.

MySQL database port [$self->{database_port}] :
EOF

    $self->{database_port} = $self->message_prompt ($self->{database_port}, $message);
    $self->{state} = 'menu_database';
}


##
# $self->set_database_user();
# ---------------------------
# Sets the MySQL database user for the
# site which is being set up.
##
sub set_database_user
{
    my $self = shift;
    $self->{database_user} ||= $ENV{DATABASE_USER} || 'root';
    my $message = <<EOF;

Your MKDoc website needs a user to connect to the database with.
By default when MySQL is installed, it defines the user 'root'.

MySQL database user [$self->{database_user}] :
EOF

    $self->{database_user} = $self->message_prompt ($self->{database_user}, $message);
    $self->{state} = 'menu_database';
}


##
# $self->set_database_pass();
# ---------------------------
# Sets the MySQL database password for the
# site which is being set up.
##
sub set_database_pass
{
    my $self = shift;
    $self->{database_pass} ||= $ENV{DATABASE_PASS} || '';
    my $message = <<EOF;

Your MKDoc website might need a password to connect to MySQL with.
By default when MySQL is installed, there is no password. If you
are using a password (it's good practice to do so), please enter
it here.

MySQL database password [$self->{database_pass}] :
EOF

    $self->{database_pass} = $self->message_prompt ($self->{database_pass}, $message);
    $self->{state} = 'menu_database';
}


##
# $self->set_organization_name();
# -------------------------------
# Sets the organization name which will
# be used to initialize the database.
##
sub set_organization_name
{
    my $self = shift;
    my $message = <<EOF;

Please enter the name of the organization which the site that you
are installing belongs to.

Organization name [$self->{organization_name}] :
EOF

    $self->{organization_name} = $self->message_prompt ($self->{organization_name}, $message);
    $self->{state} = 'menu_organization';
}


##
# $self->set_organization_email();
# --------------------------------
# Sets the organization email which will
# be used to initialize the database.
##
sub set_organization_email
{
    my $self = shift;
    $self->{organization_email} ||= $ENV{ORGANIZATION_EMAIL} || 'info@example.com';
    my $message = <<EOF;

Please enter the email of the organization which the site that you
are installing belongs to.

Organization email [$self->{organization_email}] :
EOF

    $self->{organization_email} = $self->message_prompt ($self->{organization_email}, $message);
    $self->{state} = 'menu_organization';
}


##
# $self->set_user_login();
# ------------------------
# Sets the first user's login.
##
sub set_user_login
{
    my $self = shift;
    $self->{user_login} ||= $ENV{USER_LOGIN} || 'fred';
    my $message = <<EOF;

You need to have at least one user in your MKDoc system. Please enter
the login which you will use for that first user.

User login [$self->{user_login}] :
EOF

    $self->{user_login} = $self->message_prompt ($self->{user_login}, $message);
    $self->{state} = 'menu_user';
}


##
# $self->set_user_first_name();
# -----------------------------
# Sets the first user's first name.
##
sub set_user_first_name
{
    my $self = shift;
    $self->{user_first_name} ||= $ENV{USER_FIRST_NAME} || 'Fred';
    my $message = <<EOF;

Please enter the first name of that user.

User first name [$self->{user_first_name}] :
EOF

    $self->{user_first_name} = $self->message_prompt ($self->{user_first_name}, $message);
    $self->{state} = 'menu_user';
}


##
# $self->set_user_last_name();
# ----------------------------
# Sets the first user's last name.
##
sub set_user_last_name
{
    my $self = shift;
    $self->{user_last_name} ||= $ENV{USER_LAST_NAME} || 'Flintstone';
    my $message = <<EOF;

Please enter the last name of that user.

User last name [$self->{user_last_name}] :
EOF

    $self->{user_last_name} = $self->message_prompt ($self->{user_last_name}, $message);
    $self->{state} = 'menu_user';
}


##
# $self->set_user_pass();
# -----------------------
# Sets the first user's password.
##
sub set_user_pass
{
    my $self = shift;
    $self->{user_pass} ||= $ENV{USER_PASS} || 'rocks';
    my $message = <<EOF;

Please enter the password that this user will use.

User password [$self->{user_pass}] :
EOF

    $self->{user_pass} = $self->message_prompt ($self->{user_pass}, $message);
    $self->{state} = 'menu_user';
}


##
# $self->set_user_email();
# ------------------------
# Sets the first user's email.
##
sub set_user_email
{
    my $self = shift;
    $self->{user_email} ||= $ENV{USER_EMAIL} || 'fred@example.com';
    my $message = <<EOF;

Please enter the email that this user will use.

User email [$self->{user_email}] :
EOF

    $self->{user_email} = $self->message_prompt ($self->{user_email}, $message);
    $self->{state} = 'menu_user';
}


##
# $self->set_use_imap();
# ----------------------
# Toggles the 'use imap' field to 1 or 0.
##
sub set_use_imap
{
    my $self = shift;
    $self->{use_imap} = (defined $self->{use_imap} and $self->{use_imap} eq 'no') ? 'yes' : 'no';
    $self->{state} = 'menu_imap';
}


##
# $self->set_imap_host();
# -----------------------
# Sets the imap host that'll be used for
# forum discussions.
##
sub set_imap_host
{
    my $self = shift;
    $self->{imap_host} ||= $ENV{IMAP_HOST} || '';
    my $message = <<EOF;

Please enter the IMAP host to use for the discussion board. Please
note that for performance it is better to use an IMAP server which
is on the same machine or network.

IMAP Host [$self->{imap_host}] :
EOF

    $self->{imap_host} = $self->message_prompt ($self->{imap_host}, $message);
    $self->{state} = 'menu_imap';
}


##
# $self->set_imap_port();
# -----------------------
# Sets the imap port that'll be used for
# forum discussions. Leave undef for default.
##
sub set_imap_port
{
    my $self = shift;
    $self->{imap_port} ||= $ENV{IMAP_PORT} || '';
    my $message = <<EOF;

Please enter the IMAP port to use for the discussion board.

IMAP Port [$self->{imap_port}] :
EOF

    $self->{imap_port} = $self->message_prompt ($self->{imap_port}, $message);
    $self->{state} = 'menu_imap';
}


##
# $self->set_imap_user();
# -----------------------
# Sets the imap server login.
##
sub set_imap_user
{
    my $self = shift;
    $self->{imap_user} ||= $ENV{IMAP_USER} || '';
    my $message = <<EOF;

Please enter the username (login) that should be used to
connect to the IMAP server.

IMAP user [$self->{imap_user}] :
EOF

    $self->{imap_user} = $self->message_prompt ($self->{imap_user}, $message);
    $self->{state} = 'menu_imap';
}


##
# $self->set_imap_pass();
# -----------------------
# Sets the imap server password.
##
sub set_imap_pass
{
    my $self = shift;
    $self->{imap_pass} ||= $ENV{IMAP_PASS} || '';
    my $message = <<EOF;

Please enter the password that should be used to
connect to the IMAP server.

IMAP user [$self->{imap_pass}] :
EOF

    $self->{imap_pass} = $self->message_prompt ($self->{imap_pass}, $message);
    $self->{state} = 'menu_imap';
}


##
# $self->set_email_from();
# ------------------------
# Sets the 'email from' that is being used when sending
# out newsletters.
##
sub set_email_from
{
    my $self = shift;
    $self->{email_from} ||= $ENV{EMAIL_FROM} || '';
    my $message = <<EOF;

Please enter the email address from which outgoing emails
which are used for user subscription should appear to come
from.

Mail from [$self->{email_from}] :
EOF

    $self->{email_from} = $self->message_prompt ($self->{email_from}, $message);
    $self->{state} = 'menu_email';
}


##
# $self->set_email_bcc();
# -----------------------
# Sets the 'email bcc' that is being used when sending
# out newsletters.
##
sub set_email_bcc
{
    my $self = shift;
    $self->{email_bcc} ||= $ENV{EMAIL_BCC} || '';
    my $message = <<EOF;

Please enter the email address which should receive a copy
of the outgoing emails which are used for user subscription.

Mail bcc [$self->{email_bcc}] :
EOF
    
    $self->{email_bcc} = $self->message_prompt ($self->{email_bcc}, $message);
    $self->{state} = 'menu_email';
}


##
# Check methods
# -----------------------------------------------------------------------------
##


##
# $self->check_params();
# ----------------------
# Checks all of $self's attribute for some kind of consistency.
# If everything is OK then the state is switched to 'save'
# Otherwise print errors and switches the state to 'menu_main'
# so that the user can correct them.
##
sub check_params
{
    my $self = shift;
    print "Checking config... ";
    
    my $errors = $self->check_dirs()          +
                 $self->check_apache_params() +
		 $self->check_database()      +
		 $self->check_user()          +
		 $self->check_imap()          +
		 $self->check_email();
    
    if ($errors)
    {
	print "\n\nThere were $errors error(s). Please fix them and try again!";
	print "\nPress ENTER to go back to the main menu.";
	+ $self->get_stdin_line();
	$self->{state} = 'menu_main';
    }
    else
    {
	print " OK\n";
	$self->{state} = 'save';
    }
}


##
# $self->check_dirs();
# --------------------
sub check_dirs
{
    my $self = shift;
    return $self->check_mkdoc_dir() +
           $self->check_site_dir();
}


##
# $self->check_apache_params();
# -----------------------------
sub check_apache_params
{
    my $self = shift;
    return $self->check_public_domain() +
           $self->check_user_domain()   +
	   $self->check_server_admin();
}


##
# $self->check_database();
# ------------------------
sub check_database
{
    my $self = shift;
    return $self->check_database_name() +
           $self->check_database_user();
}


##
# $self->check_organization();
# ----------------------------
sub check_organization
{
    my $self = shift;
    return $self->check_organization_name() +
           $self->check_organization_email();
}


##
# $self->check_user();
# --------------------
sub check_user
{
    my $self = shift;
    return $self->check_user_login()      +
           $self->check_user_pass()       +
	   $self->check_user_email()      +
	   $self->check_user_first_name() +
	   $self->check_user_last_name();
}


##
# $self->check_imap();
# --------------------
sub check_imap
{
    my $self = shift;
    if ($self->{use_imap} eq 'yes')
    {
	return $self->check_imap_host() +
	       $self->check_imap_user() +
	       $self->check_imap_pass() +
	       $self->check_imap_port();
    }
    return 0;
}


##
# $self->check_email();
# ---------------------
sub check_email
{
    my $self = shift;
    return $self->check_email_from() +
           $self->check_email_bcc();
}


##
# $self->check_user_login();
# --------------------------
sub check_user_login
{
    my $self = shift;
    unless (defined $self->{user_login})
    {
	$self->print_screen ( "\n\tUser login is not defined \n\t. (this is mandatory)");
	return 1;
    }
    return 0;
}


##
# $self->check_user_last_name();
# ------------------------------
sub check_user_last_name
{
    my $self = shift;
    unless (defined $self->{user_last_name})
    {
	$self->print_screen ( "\n\tUser last name is not defined. (this is mandatory)");
	return 1;
    }
    return 0;
}


##
# $self->check_user_pass();
# -------------------------
sub check_user_pass
{
    my $self = shift;
    unless (defined $self->{user_pass})
    {
	$self->print_screen ( "\n\tUser password is not defined. (this is mandatory)");
	return 1;
    }
    return 0;
}


##
# $self->check_user_email();
# --------------------------
sub check_user_email
{
    my $self = shift;
    unless (defined $self->{user_email})
    {
	$self->print_screen ( "\n\tUser email is not defined. (this is mandatory)");
	return 1;
    }
    return 0;
}


##
# $self->check_user_first_name();
# -------------------------------
sub check_user_first_name
{
    my $self = shift;
    unless (defined $self->{user_first_name})
    {
	$self->print_screen ( "\n\tUser first name is not defined. (this is mandatory)");
	return 1;
    }
    return 0;
}


##
# $self->check_imap_host();
# -------------------------
sub check_imap_host
{
    my $self = shift;
    unless (defined $self->{imap_host})
    {
	$self->print_screen ( "\n\tImap hostname is not defined. (this is mandatory)");
	return 1;
    }
    return 0;
}


##
# $self->check_imap_user();
# -------------------------
sub check_imap_user
{
    my $self = shift;
    unless (defined $self->{imap_user})
    {
	$self->print_screen ( "\n\tImap user is not defined. (this is mandatory)");
	return 1;
    }
    return 0;
}


##
# $self->check_imap_pass();
# -------------------------
sub check_imap_pass
{
    my $self = shift;
    unless (defined $self->{imap_pass})
    {
	$self->print_screen ( "\n\tImap password is not defined. (this is mandatory)");
	return 1;
    }
    return 0;
}


##
# $self->check_imap_port();
# -------------------------
sub check_imap_port
{
    my $self = shift;
    return 0;
}


##
# $self->check_public_domain();
# -----------------------------
sub check_public_domain
{
    my $self = shift;
    unless (defined $self->{public_domain})
    {
	$self->print_screen ( "\n\tPublic domain address is not defined. (this is mandatory)");
	return 1;
    }
    return 0;
}



##
# $self->check_user_domain();
# ---------------------------
sub check_user_domain
{
    my $self = shift;
    unless (defined $self->{user_domain})
    {
	$self->print_screen ( "\n\tUser domain address is not defined. (this is mandatory)");
	return 1;
    }
    return 0;
}


##
# $self->check_server_admin();
# ----------------------------    
sub check_server_admin
{
    my $self = shift;
    unless (defined $self->{server_admin})
    {
	$self->print_screen ( "\n\tServer Admin email is not defined. (this is mandatory)");
	return 1;
    }
    return 0;
}


##
# $self->check_server_dir();
# --------------------------    
sub check_mkdoc_dir
{
    my $self = shift;
    my $error = 0;
    if (defined $self->{mkdoc_dir})
    {
	return $self->check_mkdoc_dir_not_exists() +
	       $self->check_mkdoc_dir_not_a_dir();
    }
    else
    {
	$self->print_screen ( "\n\tMKDoc directory is not defined. (this is mandatory)");
	return 1;
    }
}


##
# $self->check_mkdoc_dir_not_exists();
# -----------------------------------
sub check_mkdoc_dir_not_exists
{
    my $self = shift;
    return 0 if (-e $self->{mkdoc_dir});
    $self->print_screen ( "\n\tMKDoc directory $self->{mkdoc_dir} does not exist. (this is mandatory)");
    return 1;
}


##
# $self->check_mkdoc_dir_not_a_dir();
# -----------------------------------
sub check_mkdoc_dir_not_a_dir
{
    my $self = shift;
    return 0 unless (-e $self->{mkdoc_dir});
    return 0 if (-d $self->{mkdoc_dir});
    $self->print_screen ( "\n\tMKDoc directory $self->{mkdoc_dir} does not seem to be a directory. (this is mandatory)");
    return 1;
}


##
# $self->check_email_from();
# --------------------------
sub check_email_from
{
    my $self = shift;
    unless (defined $self->{email_from})
    {
	$self->print_screen ( "\n\tSubscription email from address is not defined. (this is mandatory)");
	return 1;
    }
    return 0;
}


##
# $self->check_email_bcc();
# -------------------------
sub check_email_bcc
{
    my $self = shift;
    unless (defined $self->{email_bcc})
    {
	$self->print_screen ( "\n\tSubscription email to address is not defined. (this is mandatory)");
	return 1;
    }
    return 0;
}


##
# $self->check_site_dir();
# ------------------------
sub check_site_dir
{
    my $self = shift;
    unless (defined $self->{site_dir})
    {
	$self->print_screen ( "\n\tSite directory address is not defined. (this is mandatory)");
	return 1;
    }
    return 0;
}


##
# $self->check_database_name();
# -----------------------------
sub check_database_name
{
    my $self = shift;
    unless (defined $self->{database_name})
    {
	$self->print_screen ( "\n\tDatabase name is not defined. (this is mandatory)");
	return 1;
    }
    return 0;
}


##
# $self->check_database_user();
# -----------------------------
sub check_database_user
{
    my $self = shift;
    unless (defined $self->{database_user})
    {
	$self->print_screen ( "\n\tDatabase user is not defined. (this is mandatory)");
	return 1;
    }
    return 0;
}


##
# $self->check_organization_name();
# ---------------------------------
sub check_organization_name
{
    my $self = shift;   
    unless (defined $self->{organization_name})
    {
	$self->print_screen ( "\n\tOrganization name is not defined. (this is mandatory)");
	return 1;
    }
    return 0;
}


##
# $self->check_organization_email();
# ----------------------------------
sub check_organization_email
{
    my $self = shift;   
    unless (defined $self->{organization_email})
    {
	$self->print_screen ( "\n\tOrganization email is not defined. (this is mandatory)");
	return 1;
    }
    return 0;
}


##
# Miscelleanous methods
# -----------------------------------------------------------------------------
##

##
# $self->get_stdin_line();
# ------------------------
# Gets one line from standard input and returns it
# without any carriage returns.
##
sub get_stdin_line
{
    my $self = shift;
    my $line = <STDIN>;
    chomp ($line);
    chomp ($line);
    return $line;
}


##
# $self->clear_screen();
# ----------------------
# Sweep sweep sweep.
# TODO: Rewrite this properly
##
sub clear_screen
{
    my $self = shift;
    system ('clear');
}


##
# $self->print_screen (@);
# ----------------------
# Prints an array of lines, inserts a carriage return between each line.
##
sub print_screen
{
    my $self = shift;
    print join "\n", @_;
}


##
# $self->exit_without_install();
# ------------------------------
# Sets the state to 'exit', which ends the FSM without going
# through the 'save' state.
##
sub exit_without_install
{
    my $self = shift;
    my $message = <<EOF;

This will exit without installing the new site.
Are you sure [NO] :
EOF
    
    my $answer = $self->message_prompt ('NO', $message);
    $self->{state} = ($answer =~ /^y(es)?$/i) ? 'exit' : 'menu_main';
}


##
# $self->message_prompt ($default_value, $message);
# -------------------------------------------------
# Displays $message to the user, using $default_value
# as the default user's answer.
##
sub message_prompt
{
    my $self = shift;
    my $default = shift;
    my $message = shift;
    chomp ($message);
    chomp ($message);
    $self->print_screen ($message);
    my $line = $self->get_stdin_line();
    return $line || $default;
}


1;
