# -------------------------------------------------------------------------------------
# MKDoc::Util::TemplateFinder
# -------------------------------------------------------------------------------------
# Author : Jean-Michel Hiver <jhiver@mkdoc.com>.
# Copyright : (c) MKDoc Holdings Ltd, 2003
#
# Finds all the template directories that can be used for a certain path.
#
#     # returns all the document templates
#     return MKDoc::Util::TemplateFinder ("document");
#
# -------------------------------------------------------------------------------------
package MKDoc::Util::TemplateFinder;
use strict;
use warnings;
use Petal;


sub sillyness
{
    @Petal::BASE_DIR;
}


##
# list_structure ($what, 'default');
# ----------------------------------
# Returns a list of hash reference, each containing the following data:
#
#    {
#        value    => 'directory_name'
#        label    => 'Directory Name'
#        selected => 'selected' | undef
#    }
##
sub list_structure
{
    my $what    = shift;
    my $default = shift || 'default';
    
    my @res = map {
	my $template = $_;
	my $name = $_;
	$name =~ s/_/ /g;
	{
	    value    => $template,
	    label    => $name,
	    selected => ($template eq $default) ? 'selected' : 0,
	};
    } MKDoc::Util::TemplateFinder::list ($what);

    return (wantarray) ? @res : \@res;
}


##
# list_structure_selected (@list_structure);
# ------------------------------------------
# Returns the 'selected' item out of a list structure.
##
sub list_structure_selected
{
    my @res = grep { $_->{selected} ? $_ : () } @_;
    return shift (@res);
}


##
# list_structure_unselected (@list_structure);
# --------------------------------------------
# Returns all the items which are not 'selected' our of
# a list structure.
##
sub list_structure_unselected
{
    my @res = grep { $_->{selected} ? () : $_ } @_;
    return wantarray ? @res : \@res;
}


##
# list ($what);
# -------------
# Returns a list of directory templates which are available for $what,
# relative to @PETAL::BASE_DIR variable (see perldoc Petal).
#
#    MKDoc::TemplateFinder::list ('document'); 
##
sub list
{
    my $what = shift;
    my %res = 
        map { $_ => 1 }
        map { _available_templates_fordir ("$_/$what") }
	@Petal::BASE_DIR;
   
    my @res = sort keys %res;
    return (wantarray) ? @res : \@res;
}


sub _available_templates_fordir
{
    my $dir = shift;   
    return wantarray ? () : [] unless (-e $dir and -d $dir);
    
    my @res = ();
    opendir DD, $dir or die "Cannot read-open directory $dir";
    foreach my $file (readdir (DD))
    {
	($file eq '.' or $file eq '..') and next; # no . and .. directories
	($file eq 'CVS') and next;                # we love CVS dirs, but these are no templates
	($file =~ /^\./) and next;                # no hidden files
	(-d $dir . '/' . $file) or next;          # templates are directories
	push @res, $file;
    }
    
    return (wantarray) ? @res : \@res;
}


1;


__END__
