# -------------------------------------------------------------------------------------
# flo::Standard
# -------------------------------------------------------------------------------------
#
#     Author   : Jean-Michel Hiver
#    Copyright : (c) 2001, MKDoc Holdings Ltd
#
#      Unauthorized modification, use, reuse, distribution or redistribution
#      of this module is stricly forbidden.
#
#   Description:
#
#      A module that provides methods to easily access most commonly used objects
#      Adding a 'use flo::Standard' statement imports the following methods:
#
#        config(), cgi(), path_info(), raw_path_info(),
#        table($), template($$)
#
# -------------------------------------------------------------------------------------
package flo::Standard;
use MKDoc::Config;
use lib::sql::Table;
use MKDoc::CGI;
use Exporter;
use Carp;
use strict;
use utf8;
use 5.008_000;

our @ISA       = qw /Exporter/;
our @EXPORT_OK = qw /general_config cgi path_info raw_path_info table template current_document/;
our @TMPL_EXTS = qw /.html .txt/;


sub current_user
{
    my $timestamp = flo::Standard::cgi()->param ('logout');
    $timestamp and time < $timestamp and return;
    
    not defined $::MKD_USER and defined $ENV{REMOTE_USER} and do {
        $::MKD_USER = flo::Standard::table ('Editor')->get (Login => $ENV{REMOTE_USER});
    };
    
    return $::MKD_USER;
}


sub current_document
{
    $::MKD_CURRENT_DOCUMENT ||= do {
        my $path_info  = _current_document_path_info();
        my $document_t = flo::Standard::table ('Document');
        $document_t->get ( Full_Path => $path_info );
    };

    return $::MKD_CURRENT_DOCUMENT;
}


sub _current_document_path_info
{
    my $path_info = path_info();
    $path_info =~ s/^\///;
    $path_info =~ s/\/$//;
    my @path_info = split /\//, $path_info;
    
    pop (@path_info) if (scalar @path_info and $path_info[$#path_info] =~ /\./);
    return '/' if (scalar @path_info == 0);
    
    $path_info = join '/', @path_info;
    $path_info = "/$path_info/";
    return $path_info;
}


# miscelleanous stuff -------------------------------------------------------------------------------------------

##
# config();
# ---------
#   Returns a MKDoc::Config object that uses the global configuration file
##
sub general_config() { return new MKDoc::Config ("GLOBAL") }


##
# cgi();
# ------
#   Returns the CGI object that was instanciated for that HTTP query
##
sub cgi()
{
    return MKDoc::CGI->get;
}


##
# path_info();
# ------------
#   Returns a string which should match the Full_Path of the current document,
#   provided that this document exists
#
#   Examples of strings returned:
#     /         # good - root document
#     /foo/bar/ # good
#     /foo/bar  # bad  - missing trailing slash
#     /Foo/Bar  # bad  - mkdoc URIs are always lowercased
##
sub path_info()
{
    my $path_info = lc (raw_path_info());
    $path_info =~ s/^\/*//;
    $path_info =~ s/\/*$//;
    return ($path_info) ? '/' . $path_info .'/' : '/';
}


##
# table ($table_name);
# --------------------
#   Returns the table database object for the table which name is
#   $table_name
##
sub table($)
{
    my $table_name = shift;
    my $table = lib::sql::Table->table ($table_name);
    return $table;
}


sub raw_path_info()
{
    return $ENV{PATH_INFO};
}


1;
