# -------------------------------------------------------------------------------------
# flo::User::Preference
# -------------------------------------------------------------------------------------
#
#       Author : Steve Purkis <spurkis@mkdoc.com>
#    Copyright : (c) MKDoc Holdings Ltd, 2002.
#
#       Unauthorized modification, use, reuse, distribution or redistribution
#       of this module is stricly forbidden.
#
#    Description:
#
#       This class contains hooks for working with a user's preferences.
#
#    Notes:
#
#       @GENERALS is a bad name.
#
#       audience_options(), language_options(), general_options() should
#       be defined by correspoding Record::* class.  _map_options() really
#       returns what should be a SelectionList object.
#
#       Caching audiences, languages and generals really doesn't belong here.
#       This should be done at the persistency layer.
#
#    Synopsis:
#
#       use flo::User::Preference qw/ LIKE DONT_MIND HATE /;
#
#       # load this user's prefs:
#       my $prefs = new flo::User::Preferences( User => $user );
#
#       my @langs = $prefs->language_names;
#       $prefs->set_language_preference($langs[0], LIKES);
#
#       my @gens = $prefs->general_names;
#       my $val  = $prefs->general_preference($gens[4]);
#
#       # prefs that aren't set default to DONT_MIND
#       my @auds = $prefs->audience_names;
#       $prefs->audience_preference($auds[2]) == DONT_MIND;
#
# -------------------------------------------------------------------------------------

package flo::User::Preferences;

use flo::Record::Preference qw/ LIKE DONT_MIND HATE YES NO /;
use flo::Record::Preference::Language;
use flo::Record::Preference::Audience;
use flo::Record::Audience;

use flo::Record::Document;

use MKDoc;
use MKDoc::Config;

use Exporter;
use strict;
use 5.008_000;


##
# INHERITS FROM
# =============
#
# Exporter
##

use base qw/ Exporter /;


##
# CLASS VARIABLES
# ===============
##

## Version (from CVS revision string).
our $VERSION = (split(/ /, '$Revision: 1.11.2.8 $'))[1];


## Boolean switch controlling debug printing to STDERR.
our $DEBUG = 0;


## Things that are ok to Export.
our @EXPORT_OK = qw/ LIKE DONT_MIND HATE YES NO /;


## All general 'objects' available (from MKDoc::Config).
#our @GENERALS;


## All language 'objects' available (from MKDoc::Config).
#our @LANGUAGES;


## All audience objects available (from flo::Record::Audience).
#our @AUDIENCES;

our $RADIO_FORM = undef;


##
# INSTANCE VARIABLES
# ==================
#
# A User::Preferences contains a user and their preference objects:
#
#  user:               flo::Record::Editor object
#  general_prefs:      hash of flo::Record::Preference objects
#  language_prefs:     hash of flo::Record::Preference::Language objects
#  audience_prefs:     hash of flo::Record::Preference::Audience objects
#
# It also contains a cache of objects available to have preferences
# about (instance variables as they vary from site to site):
#
#  _generals_:         list of general 'objects' available (from MKDoc::Config)
#  _languages_:        list of languge 'objects' available (from MKDoc::Config)
#  _audiences_:        list of audience objects available (from flo::Record::Audience)
#
# Of course, you'll always use methods to access these fields, won't you?
# That's good.  It might break if you don't.
##


##
# CONSTRUCTOR
# ===========
##

##
# $pref = new flo::User::Preferences();
# -------------------------------------
#   Instantiates and returns a new object.
##
sub new
{
    my $class = shift;
    $class    = ref $class || $class;
    my $self  = bless {}, $class;
    $self->_initialize(@_);
    return $self;
}


##
# $class->_initialize(@args)
# --------------------------
#   Initializes this object, called by new().  This is empty, and
#   here to be overrided if necessary.
##
sub _initialize
{
    my $self = shift || return;
    return;
}



##
# CLASS METHODS
# =============
##


##
# @opts = $class->general_options( [$current_value] );
# ----------------------------------------------------
#
#   Gets all the general preference options available.
#   If $current_value is passed, this option is returned B<first>
#   with a 'selected' attribute set (for HTML browser compatability).
#
#   Returns a list of options containing elements of the form:
#
#       {
#        name     => $name,
#        label    => $label,
#        selected => 'selected', # if $current_value == $name
#       }
#
#   or undef on error.
##
sub general_options
{
    my $class    = shift;
    my $selected = shift;
    local $RADIO_FORM = shift;
    
    # this will have to get options out of MKDoc::Config, but
    # for the moment, just return 'yes' and 'no':
    my $opts     = {
		    &YES => { is_yes => 1 },
		    &NO  => {
			     is_no   => 1,
			     default => 1,
			    },
		   };
    return $class->_map_options($opts, $selected);
}


##
# @opts = $obj->language_options( [$current_value] );
# ---------------------------------------------------
#   Gets all the language preference options available.
#   See audience_options() for more details.
##
sub language_options ($)
{
    my $class = shift;
    return $class->audience_options(@_);
}


##
# @opts = $obj->audience_options( [$current_value] );
# ----------------------------------------------------
#
#   Gets all the audience preference options available.
#   If $current_value is passed, this option is returned B<first>
#   with a 'selected' attribute set (for HTML browser compatability).
#
#   Returns a list of options containing elements of the form:
#
#       {
#        name     => $name,
#        label    => $label,
#        selected => 'selected', # if $current_value == $name
#       }
#
#   or undef on error.
##
sub audience_options ($)
{
    my $class    = shift;
    my $selected = shift;
    local $RADIO_FORM = shift;
    my $opts     = {
		    &DONT_MIND => {
				   is_dont_mind => 1,
				   default      => 1,
				  },
		    &LIKE      => { is_like      => 1 },
		    &HATE      => { is_hate      => 1 },
		   };
    return $class->_map_options($opts, $selected);
}


##
# @opts = $class->_map_options( $opts [, $selected] );
# ----------------------------------------------------
#    Map options onto a list of {name => $x,label => $y, selected => $z}
#    Returns list or ref to list if scalar context.
##
sub _map_options ($$$)
{
    my $class    = shift;
    my $opts     = shift;
    my $selected = shift;

    my @options;
    foreach my $name (keys %$opts) {
	my $elem = $opts->{$name};
	$elem->{name} = $name;
	$elem->{selected} = 'selected' if (defined($selected) and $selected eq $name);
	$elem->{checked}  = 'checked'  if (defined($selected) and $selected eq $name);
	if ($RADIO_FORM)
	{
	    push @options, $elem;
	}
	else
	{
	    ($elem->{selected} or (! defined($selected) and $elem->{default}))
	    ? unshift @options, $elem
	    : push    @options, $elem;
	}
    }
    
    return wantarray ? @options : \@options;
}



##
# INSTANCE METHODS
# ================
##


##
# $obj->user;
# -----------
#   Returns the flo::Record::Editor object associated with this object.
##
sub user
{
    my $self = shift;
    return $self->{user};
}


##
# $obj = $obj->set_user ($user);
# ------------------------------
#   Set 'user' attribute to the flo::Record::Editor object, and load
#   this user's preferences.  Returns this object.
##
sub set_user
{
    my $self      = shift;
    $self->{user} = shift;
    return $self->_load_user_prefs;
}



##
# $val = $obj->general_preference($name);
# ---------------------------------------
#   Returns the value of the general preference $name'd, DONT_MIND if
#   not set, or undef on error.
##
sub general_preference ($$)
{
    my $self = shift;
    my $name = shift;

    # deactivate this stuff for the moment JM 02.06.2003
    # return unless ($self->general_name_exists($name));
    
    # TODO: error checks (user not set)
    my $pref = $self->_general_prefs->{$name};
    
    return $pref ? $pref->value : DONT_MIND;
}


##
# $obj = $obj->set_general_preference($name, $val);
# -------------------------------------------------
#   Sets the $name'd general preference to the string $val.
#   Returns this object, or undef on error.
##
sub set_general_preference ($$$)
{
    my $self = shift;
    my $name = shift;
    my $val  = shift;

    return unless ($self->general_name_exists($name));
    # TODO: error checks (user not set)
    my $pref = $self->_general_prefs->{$name} || $self->_create_general_pref($name);
    $pref->set_value($val)
         ->save || return;

    return $self;
}


##
# $pref = $obj->_create_general_pref($name);
# ------------------------------------------
#   Returns new $name'd general preference, or undef on error.
##
sub _create_general_pref ($$)
{
    my $self = shift;
    my $name = shift;
    
    # TODO: error checks (user not set)
    unless ($self->_general_prefs->{$name})
    {
	my $pref = new flo::Record::Preference
	  ->set_name($name)
	  ->set_user($self->user);
	# don't save it here
	$self->_general_prefs->{$name} = $pref;
    }

    return $self->_general_prefs->{$name};
}


##
# @prefs = $obj->generals;
# --------------------------
#   Returns a list of all general preferences available to be set
#   ordered by name, or undef on error.  As no object exists for
#   this yet, each element contains a hash of the form:
#       (name => $name, label => $value)
##
sub generals($)
{
    my $self = shift;
    my @generals = (
        { 'name' => 'newsletter-daily',   'label' => 'newsletter-daily' },
        { 'name' => 'newsletter-monthly', 'label' => 'newsletter-monthly' },
        { 'name' => 'newsletter-weekly',  'label' => 'newsletter-weekly' }
    );
    
    return wantarray ? @generals : \@generals; 
}


##
# @generals = $obj->general_names();
# ------------------------------------
#   Returns a list of all general preference names available,
#   or undef on error.
##
sub general_names($)
{
    my $obj = shift;
    my @names = map { $_->{name} } $obj->generals;
    return wantarray ? @names : \@names;
}


##
# $bool = $obj->general_name_exists($name);
# -------------------------------------------
#   Returns true $name'd general preference exists, false if not.
##
sub general_name_exists($)
{
    my $obj = shift;
    my $name  = shift;
    return grep({$_ eq $name} $obj->general_names) ? 1 : 0;
}



##
# $val = $obj->language_preference($name);
# ----------------------------------------
#   Returns the value of the language preference $name'd, or undef on error.
##
sub language_preference ($$)
{
    my $self = shift;
    my $name = shift;

    return unless ($self->language_name_exists($name));
    # TODO: error checks (user not set)
    my $pref = $self->_language_prefs->{$name};

    return $pref ? $pref->value : DONT_MIND;
}


##
# $obj = $obj->set_language_preference($name, $val);
# --------------------------------------------------
#   Sets the $name'd language preference to $val, one of: LIKE,
#   DONT_MIND, HATE as exported.
#   Returns this object, or undef on error.
##
sub set_language_preference ($$$)
{
    my $self = shift;
    my $name = shift;
    my $val  = shift;

    return unless ($self->language_name_exists($name));
    # TODO: error checks (user not set) (val not LIKE DONT_MIND HATE)
    my $pref = $self->_language_prefs->{$name} || $self->_create_language_pref($name);
    $pref->set_value($val)
         ->save || return;

    return $self;
}


##
# $pref = $obj->_create_language_pref($name);
# -------------------------------------------
#   Returns new $name'd language preference, or undef on error.
##
sub _create_language_pref ($$)
{
    my $self = shift;
    my $name = shift;

    # TODO: error checks (user not set)
    unless ($self->_language_prefs->{$name})
    {
	my $pref = new flo::Record::Preference::Language
	  ->set_name($name)
	  ->set_user($self->user);
	# don't save it here
	$self->_language_prefs->{$name} = $pref;
    }

    return $self->_language_prefs->{$name};
}


##
# @langs = $obj->languages;
# ---------------------------
#   Returns a list of all languages available to be set
#   ordered by name, or undef on error.  As no object exists for
#   this yet, each element contains a hash of the form:
#       (name => $name, label => $value)
##
sub languages($)
{
    my $self = shift;
    my @LANGUAGES;

    unless (exists $self->{_languages_})
    {
	my $phash = new MKDoc::Config->parsefile_hashref (MKDoc::Config->LANGUAGE_LIST);
	my @used_langs = flo::Record::Document->used_langs;
	@LANGUAGES = map { { name => $_, label => $phash->{$_} } } sort @used_langs;
	$self->{_languages_} = [ @LANGUAGES ];
    }

    return wantarray ? @{$self->{_languages_}} : $self->{_languages_};
}


##
# @languages = $obj->language_names();
# --------------------------------------
#   Returns a list of all language preference names available,
#   or undef on error.
##
sub language_names($)
{
    my $obj = shift;
    my @names = map { $_->{name} } $obj->languages;
    return wantarray ? @names : \@names;
}


##
# $bool = $obj->language_name_exists($name);
# ------------------------------------------
#   Returns true $name'd language preference exists, false if not.
##
sub language_name_exists($)
{
    my $obj  = shift;
    my $name = shift;
    return grep({$_ eq $name} $obj->language_names) ? 1 : 0;
}



##
# $val = $obj->audience_preference($name);
# ----------------------------------------
#   Returns the value of the audience preference $name'd, or undef on error.
##
sub audience_preference ($$)
{
    my $self = shift;
    my $name = shift;

    return unless ($self->audience_name_exists($name));
    # TODO: error checks (user not set)
    my $pref = $self->_audience_prefs->{$name};

    return $pref ? $pref->value : DONT_MIND;
}


##
# $obj = $obj->set_audience_preference($name, $val);
# --------------------------------------------------
#   Sets the $name'd audience preference to the string $val.
#   Returns this object, or undef on error.
##
sub set_audience_preference ($$$)
{
    my $self = shift;
    my $name = shift;
    my $val  = shift;

    return unless ($self->audience_name_exists($name));
    # TODO: error checks (user not set) (val not LIKE DONT_MIND HATE)

    my $pref = $self->_audience_prefs->{$name} || $self->_create_audience_pref($name);
    $pref->set_value($val)
         ->save || return;

    return $self;
}


##
# $pref = $obj->_create_audience_pref($name);
# -------------------------------------------
#   Returns new $name'd audience preference, or undef on error.
##
sub _create_audience_pref ($$)
{
    my $self     = shift;
    my $name     = shift;
    my $audience = $self->audience($name);

    # TODO: error checks (user not set)
    unless ($self->_audience_prefs->{$name})
    {
	my $pref = new flo::Record::Preference::Audience
	  ->set_audience ($audience)
	  ->set_user     ($self->user);
	# don't save it here
	$self->_audience_prefs->{$name} = $pref;
    }

    return $self->_audience_prefs->{$name};
}


##
# $audience = $obj->audience($name);
# ----------------------------------
#   Returns named flo::Record::Audience from this object's '_audiences_',
#   or undef on error.  This is just a shortcut JMH invented.
##
sub audience($$)
{
    my $self      = shift;
    my $name      = shift;
    my %audiences = map {$_->name => $_} $self->audiences;
    return $audiences{$name};
}


##
# $audience = $obj->audience_by_id($id);
# --------------------------------------
#   Returns named flo::Record::Audience from this object's '_audiences_',
#   or undef on error.  This is just a shortcut JMH invented.
##
sub audience_by_id($$)
{
    my $self      = shift;
    my $id        = shift;
    my %audiences = map {$_->id => $_} $self->audiences;
    return $audiences{$id};
}


##
# @auds = $obj->audiences;
# ------------------------
#   Returns a list of all flo::Record::Audience objects available
#   to be set, ordered by name, or undef on error.
##
sub audiences($)
{
    my $self = shift;
    my @AUDIENCES;

    unless (exists $self->{_audiences_})
    {
	my @auds   = load flo::Record::Audience( All => 1 );
	@AUDIENCES = sort { $a->name cmp $b->name } @auds;
	$self->{_audiences_} = [@AUDIENCES];
    }

    return wantarray ? @{$self->{_audiences_}} : $self->{_audiences_};
}


##
# @audiences = $obj->audience_names();
# ------------------------------------
#   Returns a list of all audience preference names available,
#   or undef on error.
##
sub audience_names($)
{
    my $obj   = shift;
    my @names = map { $_->name } $obj->audiences;
    return wantarray ? @names : \@names;
}


##
# $bool = $obj->audience_name_exists($name);
# ------------------------------------------
#   Returns true $name'd audience preference exists, false if not.
##
sub audience_name_exists($)
{
    my $obj  = shift;
    my $name = shift;
    return grep({$_ eq $name} $obj->audience_names) ? 1 : 0;
}



##
# $str = $obj->to_string;
# -----------------------
#   Returns a string representation of this object.
##
sub to_string ($)
{
    my $self = shift;
    return ref($self) . '=[' . join (' ', map {"$_: $self->{$_}"} keys %$self) . ']';
}




##
# PRIVATE METHODS
# ===============
##


##
# \%prefs = $self->_general_prefs;
# --------------------------------
#   Returns the value of the 'general_prefs' attribute.
##
sub _general_prefs
{
    my $self = shift;
    return $self->{general_prefs};
}


##
# $obj = $obj->_set_general_prefs( \%prefs );
# -------------------------------------------
#   Set 'general_prefs' attribute to \%prefs hashref.  Returns this object.
##
sub _set_general_prefs
{
    my $self               = shift;
    $self->{general_prefs} = shift;
}


##
# \%prefs = $self->_language_prefs;
# ---------------------------------
#   Returns the value of the 'language_prefs' attribute.
##
sub _language_prefs
{
    my $self = shift;
    return $self->{language_prefs};
}


##
# $obj = $obj->_set_language_prefs( \%prefs );
# --------------------------------------------
#   Set 'language_prefs' attribute to \%prefs hashref.  Returns this object.
##
sub _set_language_prefs
{
    my $self                = shift;
    $self->{language_prefs} = shift;
}


##
# \%prefs = $self->_audience_prefs;
# ---------------------------------
#   Returns the value of the 'audience_prefs' attribute.
##
sub _audience_prefs
{
    my $self = shift;
    return $self->{audience_prefs};
}


##
# $obj = $obj->_set_audience_prefs( \%prefs );
# --------------------------------------------
#   Set 'audience_prefs' attribute to \%prefs hashref.  Returns this object.
##
sub _set_audience_prefs
{
    my $self                = shift;
    $self->{audience_prefs} = shift;
}


##
# $obj = $obj->_load_user_prefs();
# ---------------------------
#   Load this object's user's preferences.
#   Returns this object, or undef on error.
##
sub _load_user_prefs($)
{
    my $self = shift;
    $self->_load_general_prefs  || return;
    $self->_load_language_prefs || return;
    $self->_load_audience_prefs || return;
    return $self;
}



##
# $obj = $obj->_load_general_prefs();
# -----------------------------------
#   Load this object's user's general preferences.
#   Returns this object, or undef on error.
##
sub _load_general_prefs($)
{
    my $self  = shift;
    my @prefs = load flo::Record::Preference(User => $self->user);
    my %hash  = map { $_->name, $_ } @prefs;
    $self->_set_general_prefs(\%hash);
}



##
# $obj = $obj->_load_language_prefs();
# -----------------------------------
#   Load this object's user's language preferences.
#   Returns this object, or undef on error.
##
sub _load_language_prefs($)
{
    my $self = shift;
    my @prefs = load flo::Record::Preference::Language(User => $self->user);
    my %hash  = map { $_->name, $_ } @prefs;
    $self->_set_language_prefs(\%hash);
}



##
# $obj = $obj->_load_audience_prefs();
# ------------------------------------
#   Load this object's user's audience preferences.
#   Returns this object, or undef on error.
##
sub _load_audience_prefs($)
{
    my $self  = shift;
    my @prefs = load flo::Record::Preference::Audience(User => $self->user);
    my %hash  = map { $self->audience_by_id($_->audience_id)->name => $_ } @prefs;
    $self->_set_audience_prefs(\%hash);
}


##
# $self->is_selected ('pref_name');
# ---------------------------------
# Returns TRUE if the GENERAL preference 'pref_name' is TRUE,
# FALSE otherwise.
##
sub is_selected
{
    my $self = shift;
    my $name = shift;
    my $preference_t = flo::Standard::table ('Preference');
    my $preference = $preference_t->get (Name => $name, Editor_ID => $self->user()->id()) || return;
    $preference->{Value} ? return 'selected' : return;
}


sub is_checked
{
    my $self = shift;
    $self->is_selected (@_) ? return 'checked' : return;
}


sub is_checked_not
{
    my $self = shift;
    $self->is_selected (@_) ? return : return 'checked';
}


1;


__END__


