# -------------------------------------------------------------------------------------
# flo::plugin::Discussion::Message
# -------------------------------------------------------------------------------------
#
#       Author : Jean-Michel Hiver <jhiver@mkdoc.com>
#    Copyright : (c) MKDoc Holdings Ltd, 2002
#
#      Unauthorized modification, use, reuse, distribution or redistribution
#      of this module is stricly forbidden
#
#    Description:
#
# -------------------------------------------------------------------------------------
package flo::plugin::Discussion::Message;
use Encode qw /decode/;
use MKDoc::Config;
use Mail::IMAPClient;
use flo::Standard qw /general_config cgi path_info raw_path_info table template current_document/;
use warnings;
use strict;
use 5.008_000;
use Carp;
use Petal;

use base qw /flo::Object/;

$::MKD_MESSAGES = {};
$::MKD_MESSAGES_HEADERS_TEMP = {};


##
# $class->new;
# ------------
#   Creates a new flo::plugin::Discussion::Message object.
#   Registers the object in $::MKD_MESSAGES hashref
##
sub new
{
    my $class = shift;
    $class = ref $class || $class;
    
    if (@_ == 1) { unshift (@_, 'uid') }
    my $self = bless { @_ }, $class;
    
    $::MKD_MESSAGES->{$self} = $self;
    $self->_fetch_header;
    return $self;
}


##
# $class->get ($uid); 
# -------------------
#   Returns the message matching $uid if it is somewhere in the tree,
#   undef otherwise.
##
sub get
{
    my $self = shift;
    my $uid = shift;
    return $self if (defined $self->{uid} and $self->{uid} == $uid);
    if (defined $self->{children})
    {
	foreach (@{$self->{children}})
	{
	    my $res = $_->get ($uid);
	    return $res if (defined $res);
	}
    }
    return;
}


##
# $self->uid;
# -----------
#   Returns the identifier of the specified message
##
sub uid
{
    my $self = shift;
    return unless (defined $self->{uid} and $self->{uid});
    return $self->{uid};
}


##
# $self->message_id;
# ------------------
#   Returns the Message-ID field of the specified message
##
sub message_id
{
    my $self = shift;
    return unless (defined $self->{uid} and $self->{uid});
    $self->_fetch_header;
    foreach my $header_key (keys %{$self->{headers}})
    {
	next unless ($header_key =~ /^message-id$/i);
	my $res = $self->{headers}->{$header_key}->[0];
	Encode::_utf8_on ($res);
	return $res;
    }
    return;
}


##
# $self->references;
# ------------------
#   Returns the references field of the specified message
##
sub references
{
    my $self = shift;
    return unless (defined $self->{uid} and $self->{uid});
    $self->_fetch_header;
    foreach my $header_key (keys %{$self->{headers}})
    {
	next unless ($header_key =~ /^references$/i);
	my $res = $self->{headers}->{$header_key}->[0];
	Encode::_utf8_on ($res);
	return $res;
    }
    return;
}


##
# $self->date;
# ------------
#   Returns the date field of the specified message
##
sub date
{
    my $self = shift;
    return unless (defined $self->{uid} and $self->{uid});
    $self->_fetch_header;
    foreach my $header_key (keys %{$self->{headers}})
    {
	next unless ($header_key =~ /^date$/i);
	my $res = $self->{headers}->{$header_key}->[0];
	Encode::_utf8_on ($res);
	return $res;
    }
    return;
}


##
# $self->date_w3c;
# ----------------
#   Returns the date field of the specified message in
#   W3C DTF.
##
sub date_w3c
{
    my $self = shift;
    my $res  = '';
    eval {
	use Date::Manip;
	my $date = &ParseDate ($self->date);
	my @date = &UnixDate ($date, qw /%Y %m %d %H %M %S/);
	$res = "$date[0]-$date[1]-$date[2]" . 'T' . "$date[3]:$date[4]:$date[5]Z";
    };
    if ($@ and $@)
    {
	warn $@;
	return '';
    }
    else
    {
	return $res;
    }
}


##
# $self->subject;
# ---------------
#   Returns the subject field of the specified message
##
sub subject
{
    my $self = shift;
    return unless (defined $self->{uid} and $self->{uid});
    $self->_fetch_header;
    foreach my $header_key (keys %{$self->{headers}})
    {
	next unless ($header_key =~ /^subject$/i);
	my $res = $self->{headers}->{$header_key}->[0];
	$res = decode ('MIME-Header', $res);
	Encode::_utf8_on ($res);
	return $res;
    }
    return;
}


##
# $self->to;
# ----------
#   Returns the to field of the specified message
##
sub to
{
    my $self = shift;
    return unless (defined $self->{uid} and $self->{uid});
    $self->_fetch_header;
    foreach my $header_key (keys %{$self->{headers}})
    {
	next unless ($header_key =~ /^to$/i);
	my $res = $self->{headers}->{$header_key}->[0];
	$res = decode ('MIME-Header', $res);
	Encode::_utf8_on ($res);
	return $res;
    }
    return;
}


##
# $self->from;
# ------------
#   Returns the from field of the specified message
##
sub from
{
    my $self = shift;
    return unless (defined $self->{uid} and $self->{uid});
    $self->_fetch_header;
    foreach my $header_key (keys %{$self->{headers}})
    {
	next unless ($header_key =~ /^from$/i);
	my $res = $self->{headers}->{$header_key}->[0];
	$res = decode ('MIME-Header', $res);
	Encode::_utf8_on ($res);
	return $res;
    }
    return;
}


##
# $self->lang;
# ------------
#   Returns the lang field of the specified message
##
sub lang
{
    my $self = shift;
    return unless (defined $self->{uid} and $self->{uid});
    $self->_fetch_header;
    foreach my $header_key (keys %{$self->{headers}})
    {
	next unless ($header_key =~ /^content-language$/i);
	my $res = $self->{headers}->{$header_key}->[0];
	Encode::_utf8_on ($res);
	return $res;
    }
    
    return 'en';
}



sub language { return shift()->lang (@_) }


##
# $self->lang_label;
# ------------------
#   Returns the current language label for that document,
#   i.e. 'British English'
##
sub lang_label
{
    my $self   = shift;
    my $lang   = $self->Lang;
    my $hash   = MKDoc::Config->parsefile_hashref (MKDoc::Config->LANGUAGE_LIST);
    return $hash->{$lang};
}


##
# $self->align;
# -------------
#   Returns the alignment of this object depending on the language,
#   i.e. 'left' or 'right'
##
sub align
{
    my $self   = shift;
    return ($self->direction eq 'ltr') ? 'left' : 'right';
}


##
# $self->align_opposite;
# ----------------------
#   Returns 'left' when align() returns 'right', returns 'right' otherwise
##
sub align_opposite
{
    my $self = shift;
    return ($self->direction eq 'ltr') ? 'right' : 'left';
}


##
# $self->direction;
# -----------------
#   Returns the direction depending on the language, i.e. 'left' or 'right'
##
sub direction
{
    my $self = shift;
    my $hash = MKDoc::Config->parsefile_hashref (MKDoc::Config->LANGUAGE_LIST_RTL);
    my $lang = $self->language();
    return exists $hash->{$lang} ? 'rtl' : 'ltr';
}


##
# $self->name;
# ------------
#   Returns the 'name' attribute of the specified message
##
sub name
{
    my $self = shift;
    return unless (defined $self->{uid} and $self->{uid});
    $self->_fetch_header;
    my $from = $self->from;
    $from =~ s/\<.*//;
    $from =~ s/^\s+//;
    $from =~ s/\s+$//;
    $from =~ s/\"//g;
    $from =~ s/=\?.*?\?=//;
    Encode::_utf8_on ($from);
    return $from;
}


##
# $self->body;
# ------------
#   Returns the 'body' attribute of the specified message
##
sub body
{
    my $self = shift;
    return unless (defined $self->{uid} and $self->{uid});
    $self->_fetch_body;
    my $res = $self->{body};
    Encode::_utf8_on ($res);
    return $res;
}


##
# $self->body_hyperlinked;
# ------------------------
#   Returns the body with addresses being hyperlinked
##
sub body_as_xhtml
{
    my $self = shift;
    my $res = MKDoc::Util::Text2HTML::text2xml ($self->body);
    $res =~ s/([a-z]+\:\/\/\S+)/<a href="$1">$1<\/a>/gi;
    return $res;
}


##
# $self->uri;
# -----------
#   Returns the uri of this message
##
sub uri
{
    my $self = shift;
    return unless (defined $self->{uid} and $self->{uid});
    my $cgix = cgi()->new;
    my $path_info = $cgix->path_info;
    my $hint = MKDoc::Config->IMAP_VIEW_URI_HINT;
    if ($path_info =~ /\,/) { $path_info =~ s/\,.*$/\,$hint/ }
    else                    { $path_info .= ",$hint"         }
    
    $cgix->delete ($_) for ($cgix->param);
    $cgix->path_info ($path_info);
    $cgix->param ('uid', $self->uid);
    my $res = $cgix->self_url;
    $res =~ s/\%2C/,/;
    return $res;
}


##
# $self->parent;
# --------------
#   Returns the parent message of the current message,
#   or undef if none
##
sub parent
{
    my $self = shift;
    my $parent = $::MKD_MESSAGES->{$self->{parent}};
    return $parent;
}


##
# $self->children;
# ----------------
#   Returns the children messages of the current message,
#   or an empty list if none
##
sub children
{
    my $self = shift;
    return (wantarray) ? @{$self->{children}} : $self->{children};
}


##
# $self->prev;
# ------------
#   Returns the previous sibling of the current message, or undef
#   if none
##
sub prev
{
    my $self = shift;
    my $parent = $self->parent;
    my @children = $parent->children;
    for (my $i=0; $i < @children; $i++)
    {
	next if ($i == 0);
	next unless ($children[$i]->uid);
	return $children[--$i] if ($children[$i]->uid eq $self->uid);
    }
    return;
}


##
# $self->next;
# ------------
#   Returns the next sibling of the current message, or undef
#   if none
##
sub next
{
    my $self = shift;
    my $parent = $self->parent;
    my @children = $parent->children;
    
    for (my $i=0; $i < @children; $i++)
    {
	last if ($i == $#children);
	next unless (defined $children[$i]->uid);
	return $children[++$i] if ($children[$i]->uid eq $self->uid);
    }
    return;
}


##
# $self->_fetch_body;
# -------------------
#   Fetches the current message's body if necessary
##
sub _fetch_body
{
    my $self = shift;
    return unless (defined $self->{uid} and $self->{uid});
    my $imap = new flo::plugin::Discussion::IMAP();
    $self->{body} ||= $imap->message_body ($self->{uid});
}


##
# $self->_fetch_header;
# ---------------------
#   Fetches the current message's headers, if necessary
##
sub _fetch_header
{
    my $self = shift;
    return unless (defined $self->{uid} and $self->{uid});
    my $uid  = $self->{uid};
    $self->{headers} ||= $::MKD_MESSAGES_HEADERS_TEMP->{$uid} || do {
	my $imap = new flo::plugin::Discussion::IMAP();
	$imap->message_header ($uid);
    };
}


1;


__END__
