# -------------------------------------------------------------------------------------
# flo::plugin::Redirect
# -------------------------------------------------------------------------------------
#
#       Author : Jean-Michel Hiver (jhiver@mkdoc.com).
#    Copyright : (c) MKDoc Holdings Ltd, 2001
# 
#      Unauthorized modification, use, reuse, distribution or redistribution
#      of this module is stricly forbidden
#
#    Description:
#
#      A plugin that is used to redirect HTTP requests that ask for a document
#      which has been moved to another location
#
# -------------------------------------------------------------------------------------
package flo::plugin::Redirect;
use flo::RedirectManager;
use flo::Standard;
use strict;
use CGI::Util;

use vars qw /$cgi $rawPathInfo/;


##
# $class->main;
# -------------
#   Redirects the HTTP query, if needed
##
sub main
{
    my $class = shift;
    local $cgi = flo::Standard::cgi();
    local $rawPathInfo = flo::Standard::raw_path_info() || "";
    
    # if the path is '/', return immediately
    return if ($rawPathInfo eq '/');
    
    # if the current url ends by index.html, we should redirect
    return $class->_redirect_index_html() ||
           $class->_redirect_add_slash()  ||
	   $class->_redirect_normalize()  ||
	   $class->_redirect_translate()  ||
	   $class->_redirect_add_slash_and_translate() ||
	   undef;
}


##
# $class->_redirect_index_html;
# -----------------------------
# if $rawPathInfo is (something)/index.html,
# redirects to (something)/
##
sub _redirect_index_html
{
    my $class = shift;
    if ($rawPathInfo =~ /\/index\.html?$/i)
    {
	$rawPathInfo =~ s/\/index.html?$/\//i;
	$cgi->path_info ($rawPathInfo);
	$class->_send_redirect ($cgi->self_url);
	return 'TERMINATE';
    }
    return;
}


##
# $class->_redirect_add_slash;
# ----------------------------
# if the current url has no trailing slash, then
# we should perform a redirection, but only if such
# a document exists
##
sub _redirect_add_slash
{
    my $class = shift;
    
    # if the current url has no trailing slash, then
    # we should perform a redirection, but only if such
    # a document exists
    if ($rawPathInfo !~ /\/$/)
    {
	my $old_path = quotemeta ($rawPathInfo);
	my $new_path = $rawPathInfo . '/';
	my $document_table = flo::Standard::table ('Document');
	if ($document_table->get ( Full_Path => $new_path ))
	{
	    $cgi->path_info ($new_path);
	    $class->_send_redirect ($cgi->self_url);
	    return 'TERMINATE';
	}
	return;
    }
    return;
}


##
# $class->_redirect_normalize();
# ------------------------------
# If $rawPathInfo doesn't look like a valid MKDoc URI,
# normalizes it and performs a redirect.
##
sub _redirect_normalize
{
    my $class = shift;
    
    # let's normalize the path info
    my $normal_path_info = CGI::Util::unescape ($rawPathInfo);
    $normal_path_info = join '/', map {
	s/[^a-z0-9-.]/-/g;
	s/^-*//;
	s/-*$//;
	s/-+/-/g;
	$_;
    }
    split /\//, lc ($normal_path_info);
    
    # the line above would remove the trailing slash if there
    # was one...
    
    # perldoc -f split
    # split   Splits a string into a list of strings and returns
    # that list.  By default, empty leading fields are
    # preserved, and empty trailing ones are deleted.
    
    # therefore we need to add it if necessary.
    $normal_path_info .= '/' if ($rawPathInfo =~ /\/$/);
    
    if ($rawPathInfo ne $normal_path_info)
    {
	$cgi->path_info ($normal_path_info);
	$class->_send_redirect ($cgi->self_url);
	return 'TERMINATE';
    }
    
    return;
}


##
# $class->_redirect_translate();
# ------------------------------
# Attempts to find a suitable redirect for $rawPathInfo.
# Returns 'TERMINATE' if a redirect has been sent or undef
# otherwise.
##
sub _redirect_translate
{
    my $class = shift;
    my $new_path_info = flo::RedirectManager->translate_path ($rawPathInfo) || return;
    $cgi->path_info ($new_path_info);
    $class->_send_redirect ($cgi->self_url);
    return 'TERMINATE';
}


##
# $class->_send_redirect ($redirect);
# -----------------------------------
# Sends a redirect to the client
##
sub _send_redirect
{
    my $class = shift;
    my $redirect = shift;
    print $cgi->redirect ($redirect);
}


##
# $class->_redirect_add_slash_and_translate;
# ------------------------------------------
# if the current url has no trailing slash, then
# we should perform a redirection, but only if such
# a document exists
##
sub _redirect_add_slash_and_translate
{
    my $class = shift;
    
    # if the current url has no trailing slash, then
    # we should perform a redirection, but only if such
    # a document is in the redirect table
    if ($rawPathInfo !~ /\/$/)
    {
	my $old_path = quotemeta ($rawPathInfo);
	my $new_path = $rawPathInfo . '/';
	my $redirect_table = flo::Standard::table ('Redirect');
	my $redirect = $redirect_table->get ( Old_Path => $new_path ) || return;
	
	$cgi->path_info ($redirect->{New_Path});
	$class->_send_redirect ($cgi->self_url);
	return 'TERMINATE';
    }
    return;
}


1;
