#!/usr/bin/perl
use strict;
use diagnostics;

use English;
use Pod::Usage;
use Getopt::Long;
use File::Basename;
use File::Spec::Functions qw(catfile rel2abs);

use FindBin qw($Bin);
use lib "$Bin";
use mkdoc::find;

use constant DELIMITER =>
{
  BEGIN => '[[',
  END => ']]',
  FILEEXTENSION => '\.[^\.]+$'
};

use constant FOLDER =>
{
  IGNORE => 'CVS'
};

use constant FILE =>
{
  TEMPLATE => 'xx\.html',
#  RESOURCE => '..\.txt',
  RESOURCE => '..(?:\-..)?\.txt', # add (optional) language variant support. format: xx-xx.txt
  IGNORE => 'xx\.txt',
};

use constant LANGUAGE =>
{
  RIGHTTOLEFT => '^ar|ar-sa|ar-iq|ar-eg|ar-ly|ar-dz|ar-ma|ar-tn|ar-om|ar-ye|ar-sy|ar-jo|ar-lb|ar-kw|ar-ae|ar-bh|ar-qa|he|fa|ur|yi$'
};

sub getoption
{
  my %optiondefault;

  %optiondefault = (
    help => 0,
    resourcepath => '',
    templatepath => '',
    quiet => 0
  );

  GetOptions
  (
    \%optiondefault,
    'help|h',
    'resourcepath|r=s',
    'templatepath|t=s',
    'quiet|q'
  )
    or pod2usage(2);

  pod2usage(1)
    if ($optiondefault{help});

  pod2usage("ERROR: No resource path given.\n       ^")
    if (!($optiondefault{resourcepath}) && (-t STDIN));

  pod2usage("ERROR: No template path given.\n       ^")
    if (!($optiondefault{templatepath}) && (-t STDIN));

  return %optiondefault;
}

sub readfile
{
  my ($filename) = @_;

  # expand tildes (~), and relative paths to absolute paths
  $filename = rel2abs(glob($filename));

  open(FH, '<' . $filename)
    or die "i can't read $filename: $!\n";

  my $content = join('', <FH>);

  close(FH)
    or die "i can't close $filename: $!\n";

  return ($content);
}

sub writefile
{
  my ($filename, @content) = @_;

  # STDOUT
  if ($filename eq '')
  {
    print @content;
    return 1;
  }

  # expand tildes (~), and relative paths to absolute paths
  $filename = rel2abs(glob($filename));

#  if (-e $filename)
#  {
#    print "Warning: $filename exists. Overwrite [y|N] ?";
#    my $in = <STDIN>;
#
#    chomp($in);
#
#    return 0 if (lc($in) ne 'y');
#  }

  open(FH, '>' . $filename)
    or die "i can't write $filename: $!\n";

  print FH @content;

  close(FH)
    or die "i can't close $filename: $!\n";

  return 1;
}

#in: resource file
#out: hash of [[]] [[]]
sub readresource
{
  my ($resource) = @_;

  my $BEGIN = DELIMITER->{BEGIN};
  my $END = DELIMITER->{END};

  my ($convert, %capture);

  $convert = readfile($resource);

  while ($convert =~
    /
    (
    \Q$BEGIN\E # catch the start delimiter
    .*?        # grab the contents, and don't be greedy
    \Q$END\E   # catch the end delimiter
    )

    (.*?)      # allow anything between key and translation

    \Q$BEGIN\E # catch the start delimiter
    (.*?)      # grab the contents
    \Q$END\E   # catch the end delimiter
    /gsmx
  )
  {
    my ($key, $whitespace, $translation) = ($1, $2, $3);
    
    $capture{$key} = $translation;
    warn join ("\n",
      'Warning: Content found outside translation delimiters ([[ ]]).',
      "Resource: $resource",
      "Key: $key",
      "Translation: $translation",
      "Non-whitespace: $whitespace\n\n")
      if ($whitespace =~ /\S+/gsm);
  }

  return (%capture);
}

# in: resourcepath, templatepath
#out: language templates
sub dolanguage
{
  my ($resourcepath, $templatepath, $quiet) = @_;

  my %findlang = (
    findignoredirectory => FOLDER->{IGNORE},
    findfile => 1,
    findname => FILE->{RESOURCE},
    findignorename => FILE->{IGNORE},
    findstart => $resourcepath,
    findrecurse => 0
  );

  foreach my $resource (findfile(%findlang))
  {
    my ($rname, $rpath, $rsuffix) = fileparse($resource, DELIMITER->{FILEEXTENSION});

    print "Creating templates for $rname" if (!$quiet);
    my %rhash = readresource($resource);

    my %findtemplate = (
      findignoredirectory => FOLDER->{IGNORE},
      findfile => 1,
      findname => FILE->{TEMPLATE},
      findstart => $templatepath
    );

    foreach my $template (findfile(%findtemplate))
    {
      my $tcontent = readfile($template);
      my ($tname, $tpath, $tsuffix) = fileparse($template, DELIMITER->{FILEEXTENSION});
      my $langtemplate = catfile(dirname($template), $rname . $tsuffix);
      my $TEMPLATE = FILE->{TEMPLATE};

      print '.' if (!$quiet);

      # replace keys
      while (my ($key, $value) = each %rhash)
      {
        $tcontent =~ s|\Q$key\E|$value|gsm;
      }

      # lang-xx syntax changed to xx for classes by Chris 26 June 2002
      # we need to match class="foo xx" and class="xx foo"
      #$tcontent =~ s|lang-xx|lang-$lang|ig;
      $tcontent =~ s|xx"|$rname"|gsmi;
      $tcontent =~ s|"xx|"$rname|gsmi;
      $tcontent =~ s|lang="xx"|lang="$rname"|gsmi;

      $tcontent =~ s|<% include file="(.*?)/$TEMPLATE" %>|<% include file="$1/$rname$tsuffix" %>|gsmi;

#      $tcontent =~ s|dir="ltr"|dir="rtl"|gi
#        if ($rname =~ LANGUAGE->{RIGHTTOLEFT});
      if ($rname =~ LANGUAGE->{RIGHTTOLEFT})
      {
        $tcontent =~ s|dir="xtx"|dir="rtl"|gsmi;
      }
      else
      {
        $tcontent =~ s|dir="xtx"|dir="ltr"|gsmi;
      }

      writefile($langtemplate, $tcontent);
    }
    print "OK\n" if (!$quiet);
  }
}

sub main
{
  my (%option);

  %option = getoption();
  dolanguage($option{resourcepath}, $option{templatepath}, $option{quiet});
}

main();

__END__

=pod

=head1 NAME

mklang.pl - Create language templates from resource files.

=head1 SYNOPSIS

mklang.pl --resourcepath=<path> --templatepath=<path> [options]

=head1 Example

mklang.pl -r ~/mkdoc-site/templates/resource -t ~/mkdoc-site/templates

Try --help for more help.

=head1 OPTIONS

    -h   --help                  print this message
    -r   --resourcepath=<path>   path to resource files
    -t   --templatepath=<path>   path to template files
    -q   --quiet                 suppress non-critical error messages

=head1 DESCRIPTION

For each language file found in the resource path, recurse the template
path turning the generic templates into supported language templates.

=over 8

=item -h --help

Print a brief help message and exits.

=item -r --resourcepath=<path>

Specify the path to MKDoc's resource files.

=item -t --templatepath=<path>

Specify the path to MKDoc's generic template files.

=item -q --quiet

Suppress non-critical messages.

=back

=cut

