# -----------------------------------------------------------------------------
# MKDoc::Site::Deploy::DB::Driver
# -----------------------------------------------------------------------------
#
#        Author : Steve Purkis <spurkis@mkdoc.com>
#     Copyright : (c) MKDoc Holdings Ltd, 2001
#
# This file is part of MKDoc. 
# 
# MKDoc is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# MKDoc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with MKDoc; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#  Description :
#
#      Generates Perl database drivers for MKDoc Sites.  Only MySQL is
#      supported ATM.
#
#     Synopsis :
#
#       use MKDoc::Site::Deploy::DB::Driver;
#
#       my $dbd         = new MKDoc::Site::Deploy::DB::Driver;
#       my $driver_code = $dbd->generate_driver() || die "error: $@\n";
#
# -----------------------------------------------------------------------------

package MKDoc::Site::Deploy::DB::Driver;

use strict;
use warnings;
use Carp;

use base qw/ MKDoc::Site::Deploy /;


##
# CONSTRUCTOR
# ===========
#
# $pref = new MKDoc::Site::Deploy::DB::Driver ( %args )
# -----------------------------------------------------
#   Inherited from MKDoc::Site::Deploy.  Default args:
#       verbose => true
##
sub _initialize
{
    my $self = shift || return;
    my %args = @_;

    # set defaults:
    $args{verbose} = 1 unless exists $args{verbose};

    return $self->SUPER::_initialize( @_ );
}


##
# Class methods
# =============
##


##
# $class->setup();
# ----------------
#   Grabs the config file and writes the driver.pl file
#   JM 2002.12.13
##
sub setup
{
    my $class = shift;
    my $config = new MKDoc::Site::Config;
    my $self = $class->new();
    
    open FP, ">su/driver.pl" or croak "Cannot write-open su/driver.pl";
    print FP $self->generate_driver ($config);
    close FP;
}


##
# Instance Methods
# =============================================================================
#
# Accessors & Instance Variables
# -----------------------------------------------------------------------------
# Inherited from MKDoc::Site::Deploy.
##


##
# Setup related
# -----------------------------------------------------------------------------
##


##
# $code = $obj->generate_driver( \%config )
# -----------------------------------------
# Creates a Perl database driver with the given config options.
# See get_args_from_config() for required options.
#
# Returns the driver code as text, or undef on error.
##
sub generate_driver
{
    my $self   = shift;
    my $config = shift;

    my $driver_args = $self->get_args_from_config($config)       or return;
    my $driver_code = $self->generate_driver_code(%$driver_args) or return;

    return $driver_code;
}


##
# \%args = $obj->get_args_from_config(\%config)
# -----------------------------------------------
# Maps driver arguments for DBH->spawn() from the hash of config
# variables given.  Makes sure neccessary arguments (*) are present:
#
#   *   DATABASE_NAME
#       DATABASE_HOST
#       DATABASE_PORT
#   *   DATABASE_USER
#       DATABASE_PASS
#
# Returns %args ref, or undef on error.
##
sub get_args_from_config
{
    my $self   = shift;
    my $config = shift;

    my %map = ('DATABASE_NAME' => 'database',
	       'DATABASE_HOST' => 'host',
	       'DATABASE_PORT' => 'port',
	       'DATABASE_USER' => 'user',
	       'DATABASE_PASS' => 'password');
    
    # don't use map(): not all config params will be defined
    
    my %args;
    foreach my $db_arg (keys %map)
    {
	my $val = $config->{$db_arg};
	if (not defined $val)
	{
	    next unless ($db_arg =~ /(?:NAME)|(?:USER)/);
	    return $self->_error("$db_arg not set in config");
	}
	$args{$map{$db_arg}} = $val;
    }
    
    return \%args;
}


##
# $text = $obj->generate_driver_code(%dbh_options)
# ------------------------------------------------
# Creates a Perl database driver with the given options for
# lib::sql::DBH->spawn().
#
# Returns the driver as text, or undef on error.
##
sub generate_driver_code
{
    my $self        = shift;
    my %options     = @_;
    my $dbh_options = "\n";

    $dbh_options   .= "\t$_ => '$options{$_}',\n" for (keys %options);
    
    return <<END_OF_DRIVER;
#!/usr/bin/perl

# -----------------------------------------------------------------------------
# driver.pl
# -----------------------------------------------------------------------------
#    Description: Automatically generated MKDoc Site database driver.
#    Note       : ANY CHANGES TO THIS FILE WILL BE LOST!
# -----------------------------------------------------------------------------

use lib::sql::Category;
use lib::sql::Table;
use lib::sql::type::Char;
use lib::sql::type::DateTime;
use lib::sql::type::Int;
use lib::sql::type::Text;
use lib::sql::type::LongText;
use lib::sql::DBH;

lib::sql::DBH->spawn($dbh_options);

lib::sql::Table->driver('MySQL');

1;
END_OF_DRIVER
}



1;

__END__
