# -------------------------------------------------------------------------------------
# flo::editor::Price
# -------------------------------------------------------------------------------------
# Author : Jean-Michel Hiver.
# Copyright : (c) MKDoc Holdings Ltd, 2003
#
# This file is part of MKDoc. 
# 
# MKDoc is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# MKDoc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with MKDoc; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#
# This component is used to put a price on an MKDoc page. A price consists of a
# reference ID, a description, and an amount field.
# -------------------------------------------------------------------------------------
package flo::editor::Price;
use strict;
use warnings;
use MKDoc::Ouch;
use MKDoc::ECommerce::Item;
use LWP::Simple;

use base qw /flo::Component/;


sub in_stock
{
    my $self = shift;
    my $uri  = $ENV{MKD__ECOMMERCE_STOCK_URI} || return 0;

    my $cgix = flo::Standard::cgi()->new();
    for ($cgix->param()) { $cgix->delete ($_) };
    $cgix->path_info (undef);
    $cgix->param ('reference', $self->ref_id());
    my $query_string = $cgix->query_string();

    my $qty  = get ("$uri?$query_string") || return 0;
    chomp ($qty);

    no warnings;
    return 0 + $qty;
}


##
# $self->_initialize();
# ---------------------
# Initializes this object's attributes from the arguments
# returned from the cgi_args() method.
##
sub _initialize
{
    my $self = shift;
    my $args = $self->cgi_args() || return;
    $self->set_ref_id       ($args->{ref_id}       || '');
    $self->set_description  ($args->{description} || '');
    $self->set_amount       ($args->{amount}      || '');
}


##
# $self->amount();
# ----------------
# Returns the amount associated with this price object.
##
sub amount
{
    my $self = shift;
    return $self->{amount};
}


##
# $self->set_amount ($amount);
# ----------------------------
# Sets the amount associated with this price object.
##
sub set_amount
{
    my $self = shift;
    $self->{amount} = shift;
    $self->{amount} =~ s/^\D+//;
    $self->{amount} =~ s/\D+$//;
    $self->{amount} =~ s/^0+//;
}


##
# $self->ref_id();
# ---------------
# Returns the product reference code associated with this price object.
##
sub ref_id
{
    my $self = shift;
    return $self->{ref_id};
}


##
# $self->set_ref_id ($ref_id);
# ----------------------------
# Sets the product reference code associated with this price object.
##
sub set_ref_id
{
    my $self = shift;
    $self->{ref_id} = shift;
    $self->{ref_id} =~ s/^\s+//;
    $self->{ref_id} =~ s/\s+$//;
}


##
# $self->description();
# ---------------------
# Returns the distinctive description associated with this price object.
##
sub description
{
    my $self = shift;
    return $self->{description};
}


##
# $self->set_description ($description);
# --------------------------------------
# Sets the distinctive description associated with this price object.
##
sub set_description
{
    my $self = shift;
    $self->{description} = shift;
    $self->{description} =~ s/^\s+//;
    $self->{description} =~ s/\s+$//;
}


##
# $self->validate();
# ------------------
# Validates this component. Returns TRUE if the component's data
# integrity is OK, FALSE otherwise.
##
sub validate
{
    my $self = shift;
    
    # set up the callback for errors
    local $MKDoc::Ouch::CALLBACK;
    $MKDoc::Ouch::CALLBACK = sub { $self->add_error (@_) };
    
    return $self->_validate_ref_id() &
           $self->_validate_price();
}


##
# $self->_validate_ref_id();
# --------------------------
# Validates the reference ID to see if it exists.
##
sub _validate_ref_id
{
    my $self = shift;
    $self->ref_id() or do {
	new MKDoc::Ouch 'component/price/ref_id_undef';
	return 0;
    };
    
    return 1;
}


##
# $self->_validate_price();
# -------------------------
# Validates the reference ID to see if it exists.
##
sub _validate_price
{
    my $self = shift;
    $self->amount() or do {
	new MKDoc::Ouch 'component/price/amount_undef';
	return 0;
    };
    
    $self->amount() =~ /^\d+\.\d\d$/ or do {
	new MKDoc::Ouch 'component/price/amount_invalid';
	return 0;
    };
    
    return 1;
}


sub as_item
{
    my $self = shift;
    my $desc = $self->parent->title();
    if ($self->description()) { $desc .= " (" . $self->description . ")" }

    return new MKDoc::ECommerce::Item (
       reference   => $self->ref_id,
       description => $desc,
       unit_price  => $self->amount(),
       quantity    => 1,
    );
}


1;


__END__
