# -------------------------------------------------------------------------------------
# flo::plugin::Discussion::List
# -------------------------------------------------------------------------------------
#
#       Author : Jean-Michel Hiver
#    Copyright : (c) MKDoc Holdings Ltd, 2002
#
# This file is part of MKDoc. 
# 
# MKDoc is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# MKDoc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with MKDoc; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#
#    Description:
#
# -------------------------------------------------------------------------------------
package flo::plugin::Discussion::List;
use MKDoc::Config;
use flo::ComponentPlugin;
use flo::HTTP::Header;
use flo::plugin::Discussion::IMAP;
use flo::plugin::Discussion::Message;
use flo::Standard qw /cgi/;
use strict;
use warnings;
use 5.008_000;
use Carp;

use base qw /flo::ComponentPlugin/;


sub type { 'discussion' }


sub _activate_passthru
{
    my $self = shift;
    return $self->SUPER::activate (@_);
}


sub activate
{
    my $self = shift;
    $self->SUPER::activate() || return;
    my $cgi = flo::Standard::cgi();
    my $path_info = $cgi->path_info;
    $path_info =~ /\,/ and return;
    
    my $component = $self->component;
    $component->isa ('flo::editor::Discussion') || return;

    return $self->_initialize_after_activate;
}


sub cgi_do { 'list' }


sub _initialize_after_activate
{
    my $self = shift;
    $self->{template} = 'discussion/error';
    
    # attempts to initialize the IMAP connection
    # by instanciating the IMAP singleton
    my $imap = undef;
    eval { $imap = new flo::plugin::Discussion::IMAP };
    if (defined $@ and $@)
    {
	warn "$@";
	$self->{error} = 1;
	$self->{cannot_initialize} = 1;
	return;
    }
    
    # if the mbox does not exist, let's return as well
    my $mbox = $self->component->mbox;
    $imap->set_mbox ($mbox);
    unless ($imap->mbox_exists)
    {
	warn "mbox '$mbox' does not exist";
	$self->{error} = 1;
	$self->{mbox_exists} = 1;
	return;
    }
    
    # everything is fine, let's do proper initialization
    $self->{uids}   = $imap->uid_most_recent_first;
    $self->{slices} = $imap->slicing_structure ($self->slice_thickness, $self->slice_number, $self->{uids});
    
    $self->{top_thread} = do {
	my $slice = $self->current_slice;
	my $imap = new flo::plugin::Discussion::IMAP;
	my $thread = $imap->slice_threaded ($slice);
	
	# we want the messages most recent first...
	$thread->{children} = [reverse @{$thread->{children}}]
	    if (defined $thread->{children});
	
	$thread;
    };
    
    $self->{template} = 'discussion/list';
}


##
# $self->slice_thickness;
# -----------------------
#   Returns the thickness of each page, i.e. the number
#   of message subjects which are displayed on each page
##
sub slice_thickness
{
    my $class = shift;
    $class = ref $class || $class;
    
    my $config = { new MKDoc::Config ('flo.plugin.Discussion.List')->hash };
    my $thickness = $config->{'slice_thickness'} or
        confess $class . " 'slice_thickness' config is not defined";
    return $thickness;
}


##
# $self->slice_number;
# --------------------
#   Returns current page that we're viewing
##
sub slice_number
{
    my $cgi = cgi();
    my $page = $cgi->param ('page') || 1;
    return $page;
}


##
# $self->current_slice;
# ---------------------
#   Returns the current slice that we're onto
##
sub current_slice
{
    my $self = shift;
    return unless (defined $self->{slices});
    foreach (@{$self->{slices}})
    {
	return $_ if ($_->{current});
    }
    
    return $self->{slices}->[0];
}


##
# $self->top_thread;
# ------------------
#   Returns the top thread, which holds all the messages for the
#   current page in a tree-shaped structure
##
sub top_thread
{
    my $self = shift;
    return $self->{top_thread};
}


##
# $self->has_many_slices;
# -----------------------
#   Returns TRUE if this mailbox is split into multiple pages,
#   FALSE otherwise
##
sub has_many_slices
{
    my $self = shift;
    return @{$self->{slices}} > 1;
}


##
# $self->template_path;
# ---------------------
#   Returns the directory path which is supposed to contain the language
#   template files. This method is there just to be subclassed.
##
sub template_path
{
    my $self = shift;
    return $self->{template};  
}


##
# $self->can_post;
# ----------------
#   Returns TRUE if the current user can post to the message board,
#   FALSE otherwise
##
sub can_post
{
    my $self = shift;
    $::MKD_USER;
}


##
# $self->run;
# -----------
#   Displays a list of messages
##
sub run
{
    my $self = shift;
    $self->render_http (
	__input__  => 'XML',
	__output__ => 'XHTML',
	self       => $self,
	object     => $self->component,
       );
    return 'TERMINATE';
}


1;
