# -------------------------------------------------------------------------------------
# flo::Record::Audience
# -------------------------------------------------------------------------------------
#
#       Author : Jean-Michel Hiver,
#                Steve Purkis <spurkis@mkdoc.com>
#    Copyright : (c) MKDoc Holdings Ltd, 2002.
#
# This file is part of MKDoc. 
# 
# MKDoc is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# MKDoc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with MKDoc; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#
#    Description:
#
#       Every record from the 'Audience' table will be blessed into this class
#       so that we can add extra method to access to in the template engine.
#
#    Notes:
#
#       'Value' should be changed to 'Name' (Value came from the HTML input
#       field params, which is not really suitable at this level).
#
#    Synopsis:
#
#       use flo::Record::Audience;
#
#       # create + save new audience:
#       my $audience = new flo::Record::Audience
#       $audience->set('name', 'label')
#                ->save;
#
#       # load existing audience:
#       my $audience  = load flo::Record::Audience( ID  => $id );
#       my @audiences = load flo::Record::Audience( All => 1 );
#
#       # delete existing audience:
#       $audience->delete;
#
# -------------------------------------------------------------------------------------
package flo::Record::Audience;
use flo::Standard qw/ table /;
use strict;
use 5.008_000;


##
# CLASS VARIABLES
# ===============
##

## Version (from CVS revision string).
our $VERSION = (split(/ /, '$Revision: 1.7.2.4 $'))[1];


## Boolean switch controlling debug printing to STDERR.
our $DEBUG = 0;


## SQL Table name of this class
use constant TABLE_NAME => 'Audience';



##
# INSTANCE VARIABLES
# ==================
#
# An Audience object contains these fields from the DB:
#
#  ID:          Integer object key
#  Value:       255 char Value  (change to 'Name'? -spurkis)
#  Label:       255 char Label
#
# Of course, you'll always use methods to access these fields, won't you?
# That's good.  It might break if you don't.
##


##
# CONSTRUCTOR
# ===========
##

##
# $pref = new flo::Record::Audience()
# -------------------------------------
#   Instantiates and returns a new object.
##
sub new
{
    my $class = shift;
    $class    = ref $class || $class;
    my $self  = bless { @_ }, $class;
    $self->_initialize(@_);
    return $self;
}


##
# $class->_initialize(@args)
# --------------------------
#   Initializes this object, called by new().  This is empty, and
#   here to be overrided if necessary.
##
sub _initialize
{
    my $self = shift || return;
    return;
}


##
# @objs = load flo::Record::Audience( %args );
# --------------------------------------------
#   Loads all Audience objects matching given search criteria.
#   Accepts one or more of:
#
#     *  ID           load object by unique id
#        Value        load by value
#        Name         load by value
#        Label        load by label
#     *  All          load all objects
#
#   Name is a synonym for Value (as 'Value' doesn't make sense here).
#   Additional or undefined fields are ignored.
#
#   Returns one object in scalar context, a list of Audience objects,
#   or undef on failure.
##
sub load
{
    my $class = shift || return;
    my $args  = $class->_filter_search_args(@_) || return;

    warn "  +  loading audiences matching: {" .
      join (' ', map {"$_: $args->{$_}"} keys %$args) . "}\n" if ($DEBUG);

    my $table = $class->table_handle;
    my @prefs = $table->search( %$args )->fetch_all;

    return wantarray
      ? @prefs
      : shift @prefs;
}



##
# CLASS METHODS
# =============
##


##
# $args = $class->_filter_search_args(%args);
# -------------------------------------------
#   Filters search arguments, as defined by $class->load().
#   Returns \%args, or undef on error.
##
sub _filter_search_args
{
    my $class = shift || return;
    my %args  = @_;

    # Name == Value  (strange, I know)
    foreach my $key (grep /^Name/, keys %args)
    {
	my $new_key = $key;
	$new_key    =~ s/Name/Value/;
	$args{$new_key} = delete $args{$key};
    }

    if ($args{All})
    {
	# no search criteria if they want all objects
	return {};
    }
    else
    {
	# don't pass db fields that don't exist to lib::sql::Table
	for (keys %args)
	{
	    delete($args{$_}) unless ((/(ID)|(Value)|(Label)/)
				      and defined($args{$_}));
	}
	return unless (keys %args);
    }

    return keys(%args) ? \%args : undef;
}


##
# $table = $class->table_handle;
# ------------------------------
#   See $TABLE_NAME.
#   Returns the lib::sql::Table handle of this object.
##
sub table_handle
{
    my $class = shift;
    my $table = table( $class->TABLE_NAME );
    unless ($table)
    {
	warn "error opening " . $class->TABLE_NAME . " table: $!";
	return;
    }
    return $table;
}




##
# INSTANCE METHODS
# ================
##


##
# $id = $self->id;
# ----------------
#   Returns the value of the 'ID' attribute.
##
sub id
{
    my $self = shift;
    return $self->{ID};
}


##
# $obj = $obj->set_id ($id);
# --------------------------
#   Set the 'ID' attribute to $id.  Returns this object.
##
sub set_id
{
    my $self    = shift;
    $self->{ID} = shift;
    return $self;
}


##
# $name = $self->name;
# --------------------
#   Returns the value of the 'Value' attribute (which should be called 'Name').
##
sub name
{
    my $self = shift;
    return $self->{Value};
}


##
# $obj = $obj->set_name ($name);
# ------------------------------
#   Set the 'Value' attribute (which should be called 'Name') to $name.
#   Returns this object.
##
sub set_name
{
    my $self       = shift;
    $self->{Value} = shift;
    return $self;
}


##
# $label = $self->label;
# --------------------
#   Returns the label of the 'Label' attribute.
##
sub label
{
    my $self = shift;
    return $self->{Label};
}


##
# $obj = $obj->set_label ($label);
# --------------------------------
#   Set the 'Label' attribute to $label.  Returns this object.
##
sub set_label
{
    my $self       = shift;
    $self->{Label} = shift;
    return $self;
}


##
# ($name, $label) = $obj->get;
# ----------------------------
#   Returns the value of the 'Value' and 'Label' attributes.
##
sub get
{
    my $self = shift;
    return ($self->name, $self->label);
}


##
# $obj = $obj->set ($name, $label);
# ---------------------------------
#   Sets the 'Value' and 'Label' attributes.  Returns this object.
##
sub set
{
    my $self = shift;
    $self->set_name(shift)
         ->set_label(shift);
    return $self;
}




##
# $bool = $obj->save;
# -------------------
#   Saves this object into the database (inserts or updates as appropriate).
#   Returns 1 on success, or undef on error.
##
sub save ($)
{
    my $self = shift || return;

    warn "  +  saving audience " . $self->to_string . " to db\n" if ($DEBUG);

    my $table = $self->table_handle;

    # to insert or to update... ?
    if ($self->load( ID => $self->id))
    {
	warn "\t  i  updating existing audience\n" if ($DEBUG);
	my $rs = $table->update($self, { 'ID' => $self->id });
	unless ($rs and ($rs != '0E0'))
	{
	    warn "  !  error updating existing audience " . $self->to_string . " in db!\n";
	    return;
	}
    }
    else
    {
	warn "\t  i  inserting new audience\n" if ($DEBUG);
	my $id = $table->insert( $self );
	if ($id and ($id != '0E0'))
	{
	    $self->set_id($id);
	}
	else
	{
	    warn "  !  error inserting new audience " . $self->to_string . " into db!\n";
	    return;
	}
    }

    return 1;
}



##
# $bool = $obj->delete;
# ---------------------
#   Deletes this audience from the database.
#
#   Note: does not update object (potential problem?)  But does
#   delete ID, in case you try & save() it later...
#
#   Returns 1 on success, or undef on error.
##
sub delete ($)
{
    my $self = shift || return;

    warn "  +  deleting audience " . $self->to_string ." from db\n" if ($DEBUG);

    my $table = $self->table_handle;

    # try delete...
    my $rs = $table->delete( ID => $self->id );

    if ($rs eq '0E0' or (! defined $rs))
    {
	warn "  !  error deleting audience " . $self->to_string ." from db!" unless ($rs eq '0E0');
	return;
    }

    # delete ID just in case...
    $self->set_id(undef);

    return 1;
}



##
# $str = $obj->to_string;
# -----------------------
#   Returns a string representation of this object.
##
sub to_string ()
{
    my $self = shift || return;
    return ref($self) . '=[' . join (' ', map {"$_: $self->{$_}"} keys %$self) . ']';
}



##
# $value = $self->value;
# --------------------
#   Returns the value of the 'Value' attribute.
#   Deprecated:  'Value' doesn't make sense. -spurkis
##
sub value
{
    my $self = shift;
    warn ref($self) . "->value() is deprecated. use name()\n";
    return $self->name;
}


##
# $obj = $obj->set_value ($value);
# --------------------------------
#   Set the 'Value' attribute to $value.  Returns this object.
#   Deprecated:  'Value' doesn't make sense. -spurkis
##
sub set_value
{
    my $self = shift;
    warn ref($self) . "->set_value() is deprecated. use set_name().\n";
    return $self->set_name(@_);
}



1;


__END__

