# -------------------------------------------------------------------------------------
# flo::Record::Preference::Language
# -------------------------------------------------------------------------------------
#
#       Author : Steve Purkis <spurkis@mkdoc.com>
#    Copyright : (c) MKDoc Holdings Ltd, 2002.
#
# This file is part of MKDoc. 
# 
# MKDoc is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# MKDoc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with MKDoc; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#
#    Description:
#
#       Every record from the 'Preference_Language' table will be blessed into
#       this class.  This class adds language checking to Preference objects
#       using the language information found in MKDoc::Config.
#
#    Synopsis:
#
#       use flo::Record::Preference::Language qw/LIKE DONT_MIND HATE/;
#
#       # create + save new language pref:
#       my $pref = new flo::Record::Preference::Language;
#       $pref->set_editor($editor)
#            ->set('en', LIKE)
#            ->save;
#
#       # load existing language pref:
#       my $pref   = load flo::Record::Preference::Language( ID => $id );
#       my $editor = $pref->editor;
#
#       # load editor's language prefs:
#       my @prefs  = load flo::Record::Preference::Language( Editor => $editor );
#
#       # delete existing language pref:
#       $pref->delete;
#
# -------------------------------------------------------------------------------------

package flo::Record::Preference::Language;

use flo::Record::Preference qw/ LIKE DONT_MIND HATE /;
use flo::Standard qw/ table /;

use MKDoc::Config;

use Exporter;
use strict;
use 5.008_000;


##
# INHERITS FROM
# =============
#
# Exporter, flo::Record::Preference
##

use base qw/ Exporter flo::Record::Preference /;


##
# CLASS VARIABLES
# ===============
##

## Version (from CVS revision string).
our $VERSION = (split(/ /, '$Revision: 1.2.2.4 $'))[1];


## Boolean switch controlling debug printing to STDERR.
our $DEBUG = 0;


## Things that are ok to Export.
our @EXPORT_OK = qw/ LIKE DONT_MIND HATE /;


## SQL Table name of this class
use constant TABLE_NAME => 'Preference_Language';



##
# INSTANCE VARIABLES
# ==================
#
# A Preference contains these fields from the DB:
#
#  ID:          Integer object key
#  Editor_ID:   Integer flo::Record::Editor object key
#  Language_ID: 25 char Language ISO code [MKDoc::Config('languages')]
#  Value:       Integer Value [LIKES | DONT_MIND | HATES]
#
# Of course, you'll always use methods to access these fields, won't you?
# That's good.  It might break if you don't.
##


##
# @objs = load flo::Record::Preference::Language( %args );
# --------------------------------------------------------
#   Loads all Language Preference objects matching given search criteria.
#   Accepts one or more of:
#
#     *  ID           load object by unique id
#     *  Editor       load by editor obj
#        Editor_ID    load by editor id
#        Language_ID  load by language id
#        Name         load by language id (preserve API)
#        Value        load by value
#
#   Passing an Editor object will clobber the value of Editor_ID.
#   Additional or undefined fields are ignored.
#
#   Returns one object in scalar context, a list of Preference objects,
#   or undef on failure.
##

# inherit load()



##
# CLASS METHODS
# =============
##


##
# $args = $class->_filter_search_args(%args);
# -------------------------------------------
#   Filters search arguments, as defined by $class->load().
#   Returns \%args, or undef on error.
##
sub _filter_search_args
{
    my $class = shift || return;
    my $args  = $class->SUPER::_filter_search_args(@_) || return;

    # convert Name into Language_ID (Name doesn't exist in this table)
    $args->{Language_ID} = delete $args->{Name} if (exists $args->{Name});

    return keys(%$args) ? $args : undef;
}




##
# INSTANCE METHODS
# ================
##


##
# $id = $self->language_id;
# -------------------------
#   Returns the value of the 'Language_ID' attribute.
##
sub language_id
{
    my $self = shift;
    return $self->{Language_ID};
}


##
# $obj = $obj->set_language_id ($id);
# -----------------------------------
#   Set the 'Language_ID' attribute to $id.  Returns this object.
##
sub set_language_id
{
    my $self             = shift;
    $self->{Language_ID} = shift;
    return $self;
}


##
# $name = $self->name;
# --------------------
#   Synonym for $self->language_id to preserve API.
##
sub name {
    my $self = shift;
    return $self->language_id(@_);
}


##
# $obj = $obj->set_name ($name);
# ------------------------------
#   Synonym for $obj->set_language_id to preserve API.
##
sub set_name
{
    my $self = shift;
    return $self->set_language_id(@_);
}


##
# $lang = $self->language_label;
# ------------------------------
#   Returns the language label associated with the 'Language_ID' attribute
#   in MKDoc::Config.
##
sub language_label {
    my $self = shift;
    my $cfg  = new MKDoc::Config ('languages');
    return $cfg->get($self->language_id);
}


##
# $self->direction;
# -----------------
# Returns the direction depending on the language, i.e. 'left' or 'right'
##
sub direction
{
    my $self = shift;
    my $hash = MKDoc::Config->parsefile_hashref (MKDoc::Config->LANGUAGE_LIST_RTL);
    my $lang = $self->language_id();
    return exists $hash->{$lang} ? 'rtl' : 'ltr';
}




1;


__END__


