# -------------------------------------------------------------------------------------
# flo::editor::Headlines
# -------------------------------------------------------------------------------------
# Author : Jean-Michel Hiver, Sam Tregar
# Copyright : (c) MKDoc Holdings Ltd, 2003
#
# This file is part of MKDoc. 
# 
# MKDoc is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# MKDoc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with MKDoc; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# -------------------------------------------------------------------------------------
package flo::editor::Headlines;
use flo::Record::Editor;
use Text::Unidecode;
use MKDoc::Config;
use flo::Editor;
use flo::Standard;
use MKDoc::CGI;
use MKDoc::Util::Text2HTML;
use flo::RedirectManager;
use strict;
use utf8;

use constant DEFAULT_BASE_PATH     => '/';
use constant DEFAULT_MAX_HEADLINES => 7;


use base qw /flo::Component
	     flo::editor::Mixin::compute_name
	     flo::editor::Mixin::normalize_name/;


sub preferred_extension { 'html' };


sub _initialize
{
    my $self = shift;
    my $args = $self->cgi_args();
    $self->{title}         = $args->{title}           || '';
    $self->{from_path}     = $args->{'from_path'}     || '';
    $self->{max_headlines} = $args->{'max_headlines'} || '';
    $self->{leaf_only}     = $args->{'leaf_only'}     || '';
    $self->{mode}          = $args->{'mode'}          || 'newest';
}

# getter for mode
sub mode {
    my $self = shift;
    croak("Too many args for getter!") if @_;
    return $self->{mode};
}

# true when mode == newest
sub mode_is_newest { shift->{mode} eq 'newest' }

# true when mode == upcoming
sub mode_is_upcoming { shift->{mode} eq 'upcoming' }

# setter for mode, must be newest or upcoming
sub set_mode {
    my ($self, $value) = @_;
    croak("Wrong number of args for setter!") if @_ != 2;
    croak("Mode must be 'newest' or 'upcoming' not '$value'.") 
      unless $value eq 'newest' or $value eq 'upcoming';
    $self->{mode} = $value;
}

sub leaf_only
{
    my $self = shift;
    return ($self->{leaf_only}) ? 'checked' : undef;
}


sub validate
{
    my $self = shift;
    
    # set up the callback for errors
    local $MKDoc::Ouch::CALLBACK;
    $MKDoc::Ouch::CALLBACK = sub { $self->add_error (@_) };
    
    return $self->validate_title() &
           $self->validate_from_path() &
	   $self->validate_max_headlines() &
           $self->validate_mode;
}

# check mode value
sub validate_mode {
    my $self = shift;
    unless (defined $self->{mode} and 
            ($self->{mode} eq 'newest' or $self->{mode} eq 'upcoming')) {
	new MKDoc::Ouch 'component/headlines/mode_invalid';
	return 0;
    }
    return 1;
}

sub validate_title
{
    my $self = shift;
    defined $self->{title} and $self->{title} !~ /^\s*$/ or do {
	new MKDoc::Ouch 'component/headlines/title_empty';
	return 0;
    };
    
    return 1;
}


sub validate_from_path
{
    return 1;
}


sub validate_max_headlines
{
    my $self = shift;
    defined $self->{max_headlines} or do {
	new MKDoc::Ouch 'component/headlines/max_headlines_empty';
	return 0;
    };
    
    my $max_headlines = $self->{max_headlines};
    $max_headlines =~ /^[0-9]+$/ or do {
	new MKDoc::Ouch 'component/headlines/max_headlines_malformed';
	return 0;
    };
    
    $self->{max_headlines} =~ s/^0+//;
    $self->{max_headlines} ||= 0;
    return 1;
}


##
# $self->title;
# -------------
#   Returns the title of that component
##
sub title
{
    my $self = shift;
    return $self->{title};
}


##
# $self->set_title ($new_title);
# ------------------------------
#   $new_title - new title to set
##
sub set_title
{
    my $self = shift;
    $self->{title} = shift;
    $self->{title} = join ' ', split /(?:\n|\r)/, $self->{title};
    $self->{title} =~ s/^\s+//;
    $self->{title} =~ s/\s+$//;
}


##
# $self->from_path;
# -----------------
#   Returns the from_path of that component
##
sub from_path
{
    my $self = shift;
    my $path = $self->{from_path};
    
    my $doc_t = flo::Standard::table ('Document');
    $doc_t->get ($path) or do { $path = flo::RedirectManager->translate_path ($path) || $path };
    $self->{from_path} = $path;    
    
    return $path;
}


##
# $self->set_from_path ($new_from_path);
# --------------------------------------
#   $new_from_path - new from_path to set
##
sub set_from_path
{
    my $self = shift;
    $self->{from_path} = shift;
    $self->{from_path} = join ' ', split /(?:\n|\r)/, $self->{from_path};
    $self->{from_path} =~ s/^\s+//;
    $self->{from_path} =~ s/\s+$//;
}


sub from_path_search_value
{
    my $self = shift;
    my $path = $self->from_path() || flo::Standard::current_document()->path();

    if ($path =~ /\*/)
    {
        $path =~ s/^\s+//;
        $path =~ s/\s+$//;

        my @path = ($path =~ /\s/) ? (split /\s+/, $path) : ($path);
        @path = map {
            my $path = lc ($_);
            $path =~ s/[^a-z-0-9\*\/\\-]//g;

            # turn all the backslashes into forward slashes
            $path =~ s/\\/\//g;

            # remove the trailing slash if any (we'll re-add one later)
            $path =~ s/\/$//;

            # if there is no starting slash, prefix with current document's path
            $path = flo::Standard::current_document()->path() . $_ unless ($path =~ /^\//);

            # remove starting slash
            $path =~ s/^\///g;

            # turn foo/*/blah/blurb into a MySQL regex
            $path =~ s/\*/[a-z0-9-]+/g;

            # re-add the slashes
            $path = "/$path/";
        } @path;

        my $path = join '|', @path;
        return "^($path)\$";
    }
    else
    {
        return "^" . quotemeta ($path) . ".+";
    }
}


sub max_headlines
{
    my $self = shift;
    return $self->{max_default_headlines}      || # backwards
           $self->{max_personalized_headlines} || # compatibility
           $self->{max_headlines};
}


sub set_max_headlines
{
    my $self = shift;
    $self->{max_headlines} = shift || '';
    $self->{max_headlines} = join ' ', split /(?:\n|\r)/, $self->{max_default_headlines} || '';
    $self->{max_headlines} =~ s/^(?:\s|\+)+//;
    $self->{max_headlines} =~ s/\s+$//;
    $self->{max_headlines} = DEFAULT_MAX_HEADLINES
        unless ($self->{max_headlines} =~ /^\+?\d+$/);
}


sub max_personalized_headlines
{
    my $self = shift;
    return $self->max_headlines();
}


sub max_default_headlines
{
    my $self = shift;
    return $self->max_headlines();
}


##
# $self->default_headlines;
# -------------------------
# Returns the default headlines, with no personalization as
# the user is not logged in.
##
sub default_headlines
{
    my $self = shift;
    my $mode = $self->{mode};

    # switch on mode
    if ($mode eq 'newest') {
        return $self->_default_headlines_newest();
    } else {
        return $self->_default_headlines_upcoming();
    }
}

sub _default_headlines_upcoming {
    my $self = shift;
    my $limit = $self->max_default_headlines();

    # as far as I can tell the lib::sql system can't handle a simple
    # join, so this has to be done in straight SQL.  Sigh.
    my $sql = "SELECT Document.ID as ID, 
                      Document_TimeRange.ID as Document_TimeRange_ID
               FROM Document, Document_TimeRange
               WHERE Document.ID = Document_TimeRange.Document_ID 
                 AND Document_TimeRange.FromDate >= NOW()
                 AND Full_Path REGEXP ?
               ORDER BY Document_TimeRange.FromDate ASC";
    
    my $dbh = lib::sql::DBH->get;
    my $sth = $dbh->prepare_cached($sql);
    $sth->execute($self->from_path_search_value());
    my $query = new lib::sql::Query (sth => $sth, bless_into => 'flo::Record::Document');
    my @res = $query->fetch_all;
    
    my $doc_t = flo::Standard::table ('Document');

    # get document objects for results
    my @documents = map { $doc_t->get ( $_->{ID} ) } @res;
    
    # get timerange objects from documents
    my @timerange_ids = map { $_->{Document_TimeRange_ID} } @res;
    my @timeranges;
    foreach my $doc (@documents) {
        my $timerange_id = shift @timerange_ids;
        # find this timerange in the list of components for this doc
        my ($timerange) = 
          grep { $_->isa('flo::editor::TimeRange') and
                 $_->Document_TimeRange_ID == $timerange_id }
            $doc->components;
        push(@timeranges, $timerange);
    }

    # combine documents and their timeranges into a single results
    # stream, weeding out missing timeranges.  This can happen when
    # the Document_TimeRange table gets out of sync with the Document
    # table
    @res = grep { defined $_->{timerange} }
           map  { { document  => $documents[$_],
                    timerange => $timeranges[$_] } } (0 .. $#documents);
                   
    # limit to documents which are showable
    @res = grep { $_->{document}->is_showable() } @res;
    
    # limit to leaves
    if ($self->leaf_only()) {
        @res = map { my @children = $_->{document}->children_showable(); @children ? () : $_ } @res;
    }
 
    # limit to max number of headlines
    @res = splice @res, 0, $self->max_default_headlines();

    return wantarray ? @res : \@res;
}


sub _default_headlines_newest {
    my $self = shift;

    # performs the query and put results in the stash
    my $document_t = flo::Standard::table ('Document');
    
    use lib::sql::Condition;
    my $cond = new lib::sql::Condition;
    my $doc  = flo::Standard::current_document();
    $cond->add ('Full_Path', 'REGEXP', $self->from_path_search_value());
    
    my $query = $document_t->select ( {
	cols  => '*',
	where => $cond,
	sort  => [ qw /Date_Created/ ],
	desc  => 1,
    } );

    my @res = $query->fetch_all();
    
    # limit to documents which are showable
    @res = map { $_->is_showable() ? $_ : () } @res;
   
    if ($self->leaf_only())
    {
       @res = map { my @children = $_->children_showable(); @children ? () : $_ } @res;
    }
 
    # limit to max number of headlines
    @res = splice @res, 0, $self->max_default_headlines();
    
    \@res;
}


##
# $self->user_logged_in;
# ----------------------
#   Returns TRUE if a user is logged in, FALSE otherwise.
##
sub is_user_logged_in
{
    return flo::Standard::current_user();
}


##
# $self->personalized_headlines;
# ------------------------------
#   Returns the personalized headlines.
#   If the user is not logged in, returns an empty list.
##
sub personalized_headlines
{
    my $self = shift;
    
    my $user = flo::Standard::current_user() || return [];
    
    my $mode = $self->{mode};

    # concoct SQL needed for upcoming or newest mode
    my ($extra_from, $extra_where, $extra_select, $order_by, $group_by);
    if ($mode eq 'upcoming') {
        $order_by     = "Document_TimeRange.FromDate ASC";
        $group_by     = "Document_TimeRange.ID";

        $extra_from   = ", Document_TimeRange";
        $extra_where  = "AND Document.ID = Document_TimeRange.Document_ID";
        $extra_select = ", Document_TimeRange.ID as Document_TimeRange_ID";
    } else {
        $order_by     = "Date_Created DESC";
        $group_by     = "Document.ID";
        ($extra_from, $extra_where, $extra_select) = ("") x 3;
    }


    $self->{_personalized_headlines} ||= do {
	# This is horrible, but I really don't see how to get around it
	# until we get some kind of RSS search engine
	my $sql  = <<SQL;
SELECT Document.ID AS ID, SUM(Preference_Audience.Value) AS Pref_Score 
       $extra_select
FROM   Document, Document_Audience, Audience, Preference_Audience, Editor, Preference_Language
       $extra_from
WHERE
        -- join the tables together
        (
                Preference_Language.Language_ID = Document.Lang         AND
                Preference_Language.Editor_ID = Editor.ID               AND
                Preference_Audience.Audience_ID = Audience.ID           AND
                Preference_Audience.Editor_ID = Editor.ID               AND
                Document_Audience.Audience_ID = Audience.ID             AND
                Document_Audience.Document_ID = Document.ID
        )
AND
        -- limit to the specified path
        Full_Path REGEXP ?
AND
        -- limit to the current editor
        Editor.Login = ?
AND
        -- remove languages which are not wanted
        Preference_Language.Value = 1

$extra_where

GROUP BY $group_by
HAVING Pref_Score > 0
ORDER BY $order_by
SQL
	# $sql .= "\nLIMIT 0, " . $self->max_personalized_headlines;
	my $dbh = lib::sql::DBH->get;
	my $sth = $dbh->prepare_cached ($sql);
	$sth->execute ($self->from_path_search_value(), $user->login);
	my $query = new lib::sql::Query (sth => $sth, bless_into => 'flo::Record::Document');
	my @res = $query->fetch_all;

        # get documents from ID list
        my $doc_t = flo::Standard::table ('Document');
        my @documents = map { $doc_t->get( $_->{ID} ) } @res;

        # deal with timerange data for upcoming events list
        if ($mode eq 'upcoming') {
            # get timerange objects from documents
            my @timerange_ids = map { $_->{Document_TimeRange_ID} } @res;
            my @timeranges;
            foreach my $doc (@documents) {
                my $timerange_id = shift @timerange_ids;
                # find this timerange in the list of components for this doc
                my ($timerange) = 
                  grep { $_->isa('flo::editor::TimeRange') and
                           $_->Document_TimeRange_ID == $timerange_id }
                    $doc->components;
                push(@timeranges, $timerange);
            }
            
            # combine documents and their timeranges into a single
            # results stream, weeding out missing timeranges.  This
            # can happen when the Document_TimeRange table gets out of
            # sync with the Document table
            @res = grep { defined $_->{timerange} }
                   map  { { document  => $documents[$_],
                            timerange => $timeranges[$_] } } (0..$#documents);

        } else {
            @res = @documents;
        }

        # limit to documents which are showable
        @res = grep { ($mode eq 'newest' ? $_->is_showable() : 
                                           $_->{document}->is_showable) } 
          @res;
        
        if ($self->leaf_only()) {
            @res = grep { my @children = ($mode eq 'newest' ? 
                                         $_->children_showable() : 
                                         $_->{document}->children_showable());
                          @children } @res;
        }
        
        # limit to max number of headlines
        @res = splice @res, 0, $self->max_personalized_headlines();

	\@res;
    };
    
    my $res = $self->{_personalized_headlines};
    return wantarray ? @{$res} : $res;
}


1;
