# -------------------------------------------------------------------------------------
# flo::editor::Poll
# -------------------------------------------------------------------------------------
# Author : Jean-Michel Hiver.
# Copyright : (c) MKDoc Holdings Ltd, 2003
#
# This file is part of MKDoc. 
# 
# MKDoc is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# MKDoc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with MKDoc; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# -------------------------------------------------------------------------------------
package flo::editor::Poll;
use MKDoc::Config;
use Time::Local;
use flo::Standard;
use flo::Editor;
use MKDoc::Util::Text2HTML;
use strict;

use base qw /flo::Component
	     flo::editor::Mixin::compute_name
	     flo::editor::Mixin::hr_name
	     /;


sub preferred_extension { 'html' }


sub id          { return shift->{id}         }
sub question    { return shift->{question}   }
sub answer1     { return shift->{answer1}    }
sub answer2     { return shift->{answer2}    }
sub answer3     { return shift->{answer3}    }
sub answer4     { return shift->{answer4}    }
sub answer5     { return shift->{answer5}    }
sub answer6     { return shift->{answer6}    }
sub answer7     { return shift->{answer7}    }
sub answer8     { return shift->{answer8}    }
sub answer9     { return shift->{answer9}    }
sub date_start  { return shift->{date_start} }
sub date_stop   { return shift->{date_stop}  }
sub hour_start  { return shift->{hour_start} }
sub hour_stop   { return shift->{hour_stop}  }


sub _initialize
{
    my $self = shift;
    my $args = $self->cgi_args() || return;
    
    $self->{id}         = $args->{id}         || join '', map { chr ( ord ('A') + int (rand (26)) ) } 1..10;
    $self->{question}   = $args->{question}   || '';
    $self->{answer1}    = $args->{answer1}    || '';
    $self->{answer2}    = $args->{answer2}    || '';
    $self->{answer3}    = $args->{answer3}    || '';
    $self->{answer4}    = $args->{answer4}    || '';
    $self->{answer5}    = $args->{answer5}    || '';
    $self->{answer6}    = $args->{answer6}    || '';
    $self->{answer7}    = $args->{answer7}    || '';
    $self->{answer8}    = $args->{answer8}    || '';
    $self->{answer9}    = $args->{answer9}    || '';
    $self->{date_start} = $args->{date_start} || flo::Standard::current_document()->now_day_iso();
    $self->{date_stop}  = $args->{date_stop}  || flo::Standard::current_document()->now_day_iso();
    $self->{hour_start} = $args->{hour_start} || '00';
    $self->{hour_stop}  = $args->{hour_stop}  || '00';
}


sub validate
{
    my $self = shift;
    
    # set up the callback for errors
    local $MKDoc::Ouch::CALLBACK;
    $MKDoc::Ouch::CALLBACK = sub { $self->add_error (@_) };

    return $self->validate_id()         &
           $self->validate_question()   &
	   $self->validate_answers()    &
	   $self->validate_date_start() &
	   $self->validate_hour_start() &
	   $self->validate_date_stop()  &
	   $self->validate_hour_stop()  &&
	   $self->validate_date_interval();
}


sub validate_id
{
    my $self = shift;
    defined $self->{id} and $self->{id} !~ /^\s*$/ or do {
	new MKDoc::Ouch 'component/poll/id_empty';
	return 0;
    };
    
    return 1;
}


sub validate_question
{
    my $self = shift;
    defined $self->{question} and $self->{question} !~ /^\s*$/ or do {
	new MKDoc::Ouch 'component/poll/question_empty';
	return 0;
    };
    
    return 1;
}


sub validate_answers
{
    my $self  = shift;
    my $count = 0;
    for (1..9) {
	$self->{"answer$_"} && $count++;
    };
    
    $count > 1 or do {
	new MKDoc::Ouch 'component/poll/answers_incomplete';
	return 0;
    };
    
    return 1;
}


sub validate_date_start
{
    my $self = shift;
    my $date = $self->{date_start} or do {
	new MKDoc::Ouch 'component/poll/date_start_empty';
	return 0;
    };

    $date =~ /^\d\d\d\d-\d\d-\d\d$/ or do {
	new MKDoc::Ouch 'component/poll/date_start_malformed';
	return 0;
    };
    
    my ($year, $month, $day) = split /-/, $date;
    $year  =~ s/^0+//;
    $month =~ s/^0+//;
    $day   =~ s/^0+//;
    
    $year  = 0 if ($year eq '');
    $day   = 0 if ($day eq '');
    
    $month == 0 and do {
	new MKDoc::Ouch 'component/poll/date_start_invalid';
	return 0;
    };
    
    $month--;
    $year -= 1900;
    eval { timelocal ( 0, 0, 0, $day, $month, $year) };
    
    $@ and do {
	new MKDoc::Ouch 'component/poll/date_start_invalid';
	return 0;
    };
    
    return 1;
}


sub validate_date_stop
{
    my $self = shift;
    my $date = $self->{date_stop} or do {
	new MKDoc::Ouch 'component/poll/date_stop_empty';
	return 0;
    };

    $date =~ /^\d\d\d\d-\d\d-\d\d$/ or do {
	new MKDoc::Ouch 'component/poll/date_stop_malformed';
	return 0;
    };
    
    my ($year, $month, $day) = split /-/, $date;
    $year  =~ s/^0+//;
    $month =~ s/^0+//;
    $day   =~ s/^0+//;
    
    $year  = 0 if ($year eq '');
    $day   = 0 if ($day eq '');
    
    $month == 0 and do {
	new MKDoc::Ouch 'component/poll/date_stop_invalid';
	return 0;
    };
    
    $month--;
    $year -= 1900;
    eval { timelocal ( 0, 0, 0, $day, $month, $year) };
    
    $@ and do {
	new MKDoc::Ouch 'component/poll/date_stop_invalid';
	return 0;
    };
    
    return 1;
}


sub validate_hour_start
{
    my $self = shift;
    defined $self->{hour_start} or do {
	new MKDoc::Ouch 'component/poll/hour_start_empty';
	return 0;
    };
    
    my $hour = $self->{hour_start};
    $hour =~ /^\d\d$/ or do {
	new MKDoc::Ouch 'component/poll/hour_start_malformed';
	return 0;
    };
    
    $hour =~ s/^0//;
    $hour < 24 or do {
	new MKDoc::Ouch 'component/poll/hour_start_invalid';
	return 0;
    };
    
    return 1;
}


sub validate_hour_stop
{
    my $self = shift;
    defined $self->{hour_stop} or do {
	new MKDoc::Ouch 'component/poll/hour_stop_empty';
	return 0;
    };
    
    my $hour = $self->{hour_stop};
    $hour =~ /^\d\d$/ or do {
	new MKDoc::Ouch 'component/poll/hour_stop_malformed';
	return 0;
    };
    
    $hour =~ s/^0//;
    $hour < 24 or do {
	new MKDoc::Ouch 'component/poll/hour_stop_invalid';
	return 0;
    };
    
    return 1;
}


sub validate_date_interval
{
    my $self  = shift;
    my $start = $self->start_localtime();
    my $stop  = $self->stop_localtime();
    $start < $stop or do {
	new MKDoc::Ouch 'component/poll/interval_invalid';
	return 0;
    };
    
    return 1;
}


# $self->{date_year_start}  = $start_year;
# $self->{date_month_start} = $start_month;
# $self->{date_day_start}   = $start_day;
# $self->{date_hour_start}  = $start_hour;
# $self->{date_year_stop}   = $stop_year;
# $self->{date_month_stop}  = $stop_month;
# $self->{date_day_stop}    = $stop_day;
# $self->{date_hour_stop}   = $stop_hour;


##
# $self->start_localtime;
# -----------------------
#   Compute localtime from which to activate the poll
##
sub start_localtime
{
    my $self = shift;
    my $date = $self->date_start();
    my ($year, $month, $day) = split /-/, $date;
    return timelocal ( 0, 0, $self->hour_start(), $day, $month - 1, $year - 1900 );
}


##
# $self->stop_localtime;
# ----------------------
#   Compute localtime from which to activate the poll
##
sub stop_localtime
{
    my $self = shift;
    my $date = $self->date_stop();
    my ($year, $month, $day) = split /-/, $date;
    return timelocal ( 0, 0, $self->hour_stop(), $day, $month - 1, $year - 1900 );
}


##
# $self->is_started;
# ------------------
#   Returns TRUE if the poll is started, FALSE otherwise
##
sub is_started
{
    my $self = shift;
    return time > $self->start_localtime and time <= $self->stop_localtime;
}


##
# $self->is_stopped;
# ------------------
#   Returns TRUE is the poll is stopped, FALSE otherwise
##
sub is_stopped
{
    my $self = shift;
    return time > $self->stop_localtime;
}


sub count_votes
{
    my $self = shift;
    my $poll_results_table = flo::Standard::table ('Poll_Results');
    my $count = 0;
    my $query = $poll_results_table->select ( where => { Poll_ID => $self->{id} } );
    
    while (my $vote = $query->next) { $count++ }
    return $count;
}


sub results
{
    my $self = shift;
    my $poll_results_table = flo::Standard::table ('Poll_Results');
    
    # now we need to compute the results for that poll;
    # query the poll tables to get the results for that vote, and
    # aggregate these results
    my $votes = {};
    my $count = 0;
    my $query = $poll_results_table->select ( where => { Poll_ID => $self->{id} } );
    
    while (my $vote = $query->next)
    {
	$count++;
	my $answer = $vote->{Answer};
	if (defined $votes->{$answer}) { $votes->{$answer}++   }
	else                           { $votes->{$answer} = 1 }
    }
    
    # if count is equals to zero, then scores should be quite easy to
    # compute. We need to return before we screw up with a division
    # by zero error
    my @poll_results = ();
    if ($count == 0)
    {
	foreach my $answer_name (qw /answer1 answer2 answer3 answer4 answer5
				     answer6 answer7 answer8 answer9/)
	{
	    my $answer = $self->{$answer_name};
	    next unless (defined $answer and $answer ne '');
	    push @poll_results, {
		answer      => $answer,
		votes       => 0,
		score       => 0,
		score_array => [],
	    };
	}
    }
    else
    {
	foreach my $answer_name (qw /answer1 answer2 answer3 answer4 answer5
				     answer6 answer7 answer8 answer9/)
	{
	    my $answer = $self->{$answer_name};
	    next unless (defined $answer and $answer ne '');
	    my $votes = $votes->{$answer_name};
	    $votes = 0 unless (defined $votes);
	    
	    my $score = 100 * $votes / $count;
	    unless ($score == int ($score))
	    {
		my $score_low = int ($score);
		my $score_high = $score_low + 1;
		if (abs ($score - $score_low) < abs ($score - $score_high))
		{
		    $score = $score_low;
		}
		else
		{
		    $score = $score_high;
		}
	    }
	    
	    push @poll_results, {
		answer => $answer,
		votes  => $votes,
		score  => $score,
		score_array => [ map { {} } 1 .. $score ],
	    }
	}
    }
    return wantarray ? @poll_results : \@poll_results;
}


##
# $self->view_results_uri;
# ------------------------
#   Computes the URI to see the results for that poll
##
sub view_results_uri
{
    my $self = shift;
    return $self->uri();
}


##
# $self->parse_xml;
# -----------------
#   Parses the XML to reconstruct the component
##
sub parse_xml
{
    my $self = shift;
    my $data = shift || return;

    if ($data =~ /^\<perl/ or $data =~ /^\<hash/) { return $self->SUPER::parse_xml ($data, @_) }
    
    # backwards compatibility... bleurgh    
    ($self->{id})       = $data =~ /\<id\>(.*?)\<\/id\>/;
    ($self->{question}) = $data =~ /\<question\>(.*?)\<\/question\>/;
    ( $self->{answer1}, $self->{answer2}, $self->{answer3}, $self->{answer4}, 
      $self->{answer5}, $self->{answer6}, $self->{answer7}, $self->{answer8}, 
      $self->{answer9} ) = $data =~ /\<answer\>(.*?)\<\/answer\>/gsim;
    
    $self->{question} = html2text ($self->{question});
    $self->{answer1}  = html2text ($self->{answer1});
    $self->{answer2}  = html2text ($self->{answer2});
    $self->{answer3}  = html2text ($self->{answer3});
    $self->{answer4}  = html2text ($self->{answer4});
    $self->{answer5}  = html2text ($self->{answer5});
    $self->{answer6}  = html2text ($self->{answer6});
    $self->{answer7}  = html2text ($self->{answer7});
    $self->{answer8}  = html2text ($self->{answer8});
    $self->{answer9}  = html2text ($self->{answer9});

    my ($start_localtime) = $data =~ /\<startdate\>(.*?)\<\/startdate\>/ism;
    my ($stop_localtime)  = $data =~ /\<stopdate\>(.*?)\<\/stopdate\>/ism;
    
    my ($sec, $min, $hour, $mday, $mon, $year, $wday, $yday, $isdst) = localtime ($start_localtime);
    $mon  += 1;
    $year += 1900;
    $hour  = "0$hour" if (length ($hour) < 2);
    $mday  = "0$mday" if (length ($mday) < 2);
    $mon   = "0$mon"  if (length ($mon)  < 2);
    
    $self->{date_start} = "$year-$mon-$mday";
    $self->{hour_start} = $hour;
    
    ($sec, $min, $hour, $mday, $mon, $year, $wday, $yday, $isdst) = localtime ($stop_localtime);
    $mon  += 1;
    $year += 1900;
    $hour  = "0$hour" if (length ($hour) < 2);
    $mday  = "0$mday" if (length ($mday) < 2);
    $mon   = "0$mon"  if (length ($mon)  < 2);
    
    $self->{date_stop} = "$year-$mon-$mday";
    $self->{hour_stop} = $hour;
}


1;
