# -------------------------------------------------------------------------------------
# flo::editor::TimeRange
# -------------------------------------------------------------------------------------
# Author : Sam Tregar
# Copyright : (c) MKDoc Holdings Ltd, 2004
#
# This file is part of MKDoc. 
# 
# MKDoc is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# MKDoc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with MKDoc; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# -------------------------------------------------------------------------------------
package flo::editor::TimeRange;

=head1 NAME

flo::editor::TimeRange - an editor for time-ranges

=head1 DESCRIPTION

The module provides an editor for time-range components.  A time-range
component has a title, a start date and an end date.  Documents with
time-ranges will appear in the upcoming-events list in headlines and
newsletters.

=head1 INTERFACE

=over 4

=cut

use flo::Standard;
use Carp qw(croak);
use strict;
use warnings;
use utf8;

use flo::Record::Editor;
use Text::Unidecode;
use MKDoc::Config;
use flo::Editor;
use flo::Standard;
use MKDoc::CGI;
use MKDoc::Util::Text2HTML;
use flo::RedirectManager;
use DateTime;
use DateTime::TimeZone;

use base qw /flo::Component
	     flo::editor::Mixin::compute_name
	     flo::editor::Mixin::normalize_name/;

use constant DATETIME_PARTS => qw(month day year hour minute tz);
use constant DATE_PARTS     => qw(month day year);

sub preferred_extension { 'html' };

sub _initialize {
    my $self = shift;
    my $args = $self->cgi_args();

    # setup simple fields
    $self->{Document_TimeRange_ID} = $args->{Document_TimeRange_ID} || undef;
    $self->{title}                 = $args->{title}                 || '';

    # setup date fields, defaulting date parts to now and time to 00:00
    my $now = DateTime->now(time_zone => "local");
    foreach my $prefix (qw(from to)) {
        foreach my $part (DATE_PARTS) {
            $self->{"${prefix}_$part"} = $args->{"${prefix}_$part"} || $now->$part;
        }
        $self->{"${prefix}_hour"}   = $args->{"${prefix}_hour"}  || "00";
        $self->{"${prefix}_minute"} = $args->{"${prefix}_minute"}|| "00";
        $self->{"${prefix}_tz"}     = $args->{"${prefix}_tz"}    || $now->time_zone->name;
    }
}

=item C<< $self->validate() >>

Check the data entered for validity.  Makes sure everything has a
value and that the dates aren't invalid.  Returns 1 if everything is
ok, 0 otherwise.  When errors are found MKDoc::Ouch objects are
created with codes to describe the problem (ex:
component/timerange/title_empty).

=cut

sub validate {
    my $self = shift;
    my $problems = 0;
    
    # set up the callback for errors
    local $MKDoc::Ouch::CALLBACK;
    $MKDoc::Ouch::CALLBACK = sub { $self->add_error (@_) };

    # look at dates individually
    foreach my $prefix (qw(from to)) {
        $problems += not $self->validate_date($prefix);
    }

    # compare the dates and make sure from is before to.  Only do this
    # if everything else is ok since it won't work on incomplete dates
    if (not $problems) {
        my $cmp = $self->from_datetime <=> $self->to_datetime;
        if ($cmp == 0) {
            new MKDoc::Ouch "component/timerange/dates_equal";
            $problems++;
        } elsif ($cmp == 1) {
            new MKDoc::Ouch "component/timerange/dates_reversed";
            $problems++;
        }
    }                           

    # validate title, must have something aside from space
    if (not defined $self->{title} or $self->{title} =~ /^\s*$/) {
	new MKDoc::Ouch 'component/timerange/title_empty';
        $problems++;
    }    

    return not $problems;
}

sub validate_date {
    my ($self, $prefix) = @_;
    my $problems = 0;

    # these must be non-zero
    foreach my $part (qw(month day year)) {
        next if $self->{"${prefix}_$part"};
        new MKDoc::Ouch "component/timerange/${prefix}_${part}_empty";
        $problems++;
    }

    # these just need to be defined
    foreach my $part (qw(hour minute tz)) {
        next if defined $self->{"${prefix}_$part"};
        new MKDoc::Ouch "component/timerange/${prefix}_${part}_empty";
        $problems++;
    }

    # make sure the date is valid if everything else was ok
    if (not $problems and not $self->_gen_datetime($prefix)) {
        new MKDoc::Ouch "component/timerange/${prefix}_invalid";
        $problems++;
    }

    return not $problems;
}

=item C<< $title = $self->title >>

Returns the title attribute

=cut

sub title { 
    my $self = shift;
    croak("Too many args for getter!") if @_;
    $self->{title};
}

=item C<< $self->set_title($new_title); >>

Sets the title attribute

=cut

sub set_title {
    my $self = shift;
    $self->{title} = shift;
    $self->{title} = join ' ', split /(?:\n|\r)/, $self->{title};
    $self->{title} =~ s/^\s+//;
    $self->{title} =~ s/\s+$//;
}

=item C<< $id = $self->Document_TimeRange_ID >>

Returns the ID for the row in Document_TimeRange corresponding to this
object or undef if no row exists.

=cut

sub Document_TimeRange_ID { 
    my $self = shift;
    croak("Too many args for getter!") if @_;
    $self->{Document_TimeRange_ID};
}

=item C<< $self->set_Document_TimeRange_ID($new_ID); >>

Sets the ID for the row in Document_TimeRange corresponding to this
object.

=cut

sub set_Document_TimeRange_ID {
    my ($self, $value) = @_;
    croak("Wrong number of args for setter!") if @_ != 2;
    $self->{Document_TimeRange_ID} = $value;
}

=item C<< $month = $self->from_month >>

=item C<< $self->set_from_month($month) >>

=item C<< $day = $self->from_day >>

=item C<< $self->set_from_day($day) >>

=item C<< $year = $self->from_year >>

=item C<< $self->set_from_year($year) >>

=item C<< $hour = $self->from_hour >>

=item C<< $self->set_from_hour($hour) >>

=item C<< $minute = $self->from_minute >>

=item C<< $self->set_from_minute($minute) >>

=item C<< $tz = $self->from_tz >>

=item C<< $self->set_from_tz($tz) >>

Get/set parts of the from date.  Months are numbered from 1 to 12,
hours are 0 to 23 and timezones are offsets from UTC in +/-HH:MM format
(ex. +05:00, -13:50).

=item C<< $month = $self->to_month >>

=item C<< $self->set_to_month($month) >>

=item C<< $day = $self->to_day >>

=item C<< $self->set_to_day($day) >>

=item C<< $year = $self->to_year >>

=item C<< $self->set_to_year($year) >>

=item C<< $hour = $self->to_hour >>

=item C<< $self->set_to_hour($hour) >>

=item C<< $minute = $self->to_minute >>

=item C<< $self->set_to_minute($minute) >>

=item C<< $tz = $self->to_tz >>

=item C<< $self->set_to_tz($tz) >>

Get/set parts of the to date.  Months are numbered from 1 to 12, hours
are 0 to 23 and timezones are offsets from UTC in +/-HH:MM format
(ex. +05:00, -13:50).

=item C<< $date = $self->from_datetime >>

Returns a DateTime object representing the from_date.  Returns undef
if the date is invalid.

=item C<< $date = $self->to_datetime >>

Returns a DateTime object representing the to_date.  Returns undef if
the date is invalid.

=cut

# auto-generate accessors and mutators for date parts
BEGIN {
    no strict 'refs';
    foreach my $prefix qw(from to) {
        foreach my $part (DATETIME_PARTS) {
            my $name = "${prefix}_$part";
            *{$name} = sub {
                my $self = shift;
                croak("Too many args for getter!") if @_;
                return $self->{$name};
            };
            *{"set_$name"} = sub { 
                my ($self, $value) = @_;
                croak("Wrong number of args for setter!") if @_ != 2;
                $self->{$name} = $value;
            }
        }
    }
}

# from_datetime and to_datetime share a common worker, _gen_datetime
sub from_datetime {
    return shift->_gen_datetime("from");
}

sub to_datetime {
    return shift->_gen_datetime("to");
}

sub _gen_datetime {
    my ($self, $which) = @_;

    # catch error for bad dates and just return undef
    my $dt;
    eval {
        $dt = DateTime->new( (map { ($_ => $self->{"${which}_$_"}) }
                              (qw(year month day hour minute))),
                             second => 0,
                             time_zone => $self->{"${which}_tz"},
                           );
    };

    return $dt;
}

=item C<< $self->month_select($which) >>

Returns an array of hashes for use in building the month selector in
the time-range editor.  Pass "from" or "to" as the sole parameter
depending on which selector is being built.  Keys are 'label', 'value'
and 'is_selected'.

=cut

sub month_select {
    my $self = shift;
    croak("Wrong number of params for month_select().") unless @_ == 1;
    my $which = shift;
    croak("Bad value for which: '$which'.  Should be 'from' or 'to'.")
      unless $which eq 'from' or $which eq 'to';
    my $x = 0;
    return [ map { $x++; 
                   { label       => $_,
                     value       => $x, 
                     is_selected => (($self->{"${which}_month"} || 0) == $x) ? 
                                    "selected" : undef
                   } } 
             (qw(January February March April May June July August September October November December)) ];
}

=item C<< $self->day_select($which) >>

Returns an array of hashes for use in building the day selector in
the time-range editor.  Pass "from" or "to" as the sole parameter
depending on which selector is being built.  Keys are 'label', 'value'
and 'is_selected'.

=cut

sub day_select {
    my $self = shift;
    croak("Wrong number of params for day_select().") unless @_ == 1;
    my $which = shift;
    croak("Bad value for which: '$which'.  Should be 'from' or 'to'.")
      unless $which eq 'from' or $which eq 'to';
    return [ map { { label       => $_,
                     value       => $_, 
                     is_selected => (($self->{"${which}_day"} || 0) == $_) ? 
                                    "selected" : undef
                   } } 
             (1 .. 31) ];
}

=item C<< $self->year_select($which) >>

Returns an array of hashes for use in building the year selector in
the time-range editor.  Pass "from" or "to" as the sole parameter
depending on which selector is being built.  Keys are 'label', 'value'
and 'is_selected'.

=cut
 
sub year_select {
    my $self = shift;
    croak("Wrong number of params for year_select().") unless @_ == 1;
    my $which = shift;
    croak("Bad value for which: '$which'.  Should be 'from' or 'to'.")
      unless $which eq 'from' or $which eq 'to';
    my $year = (localtime())[5] + 1900;
    $year = $self->{"${which}_year"} if $self->{"${which}_year"} and 
                                        $year > ($self->{"${which}_year"});
    return [ map { { label       => $_,
                     value       => $_, 
                     is_selected => (($self->{"${which}_year"} || 0) == $_) ? 
                                    "selected" : undef
                   } } 
             ($year .. $year + 25) ];
}

=item C<< $self->hour_select($which) >>

Returns an array of hashes for use in building the hour selector in
the time-range editor.  Pass "from" or "to" as the sole parameter
depending on which selector is being built.  Keys are 'label', 'value'
and 'is_selected'.

=cut
 
sub hour_select {
    my $self = shift;
    croak("Wrong number of params for hour_select().") unless @_ == 1;
    my $which = shift;
    croak("Bad value for which: '$which'.  Should be 'from' or 'to'.")
      unless $which eq 'from' or $which eq 'to';
    return [ map { { label       => sprintf("%02d", $_),
                     value       => $_, 
                     is_selected => (($self->{"${which}_hour"} || 0) == $_) ? 
                                    "selected" : undef
                   } } 
             (0 .. 23) ];
}

=item C<< $self->minute_select($which) >>

Returns an array of hashes for use in building the minute selector in
the time-range editor.  Pass "from" or "to" as the sole parameter
depending on which selector is being built.  Keys are 'label', 'value'
and 'is_selected'.

=cut
 
sub minute_select {
    my $self = shift;
    croak("Wrong number of params for minute_select().") unless @_ == 1;
    my $which = shift;
    croak("Bad value for which: '$which'.  Should be 'from' or 'to'.")
      unless $which eq 'from' or $which eq 'to';
    return [ map { { label       => sprintf("%02d", $_),
                     value       => $_, 
                     is_selected => (($self->{"${which}_minute"} || 0) == $_) ? 
                                    "selected" : undef
                   } } 
             (0 .. 59) ];
}

=item C<< $self->tz_select($which) >>

Returns an array of hashes for use in building the tz selector in
the time-range editor.  Pass "from" or "to" as the sole parameter
depending on which selector is being built.  Keys are 'label', 'value'
and 'is_selected'.

=cut

# I wish there was a way to do this that used documented
# functionality.  Maybe we'll switch to DateTime::TimeZone.  Also, the
# list of timezones is much too small.
 
sub tz_select {
    my $self = shift;
    croak("Wrong number of params for tz_select().") unless @_ == 1;
    my $which = shift;
    croak("Bad value for which: '$which'.  Should be 'from' or 'to'.")
      unless $which eq 'from' or $which eq 'to';
    my $value = $self->{"${which}_tz"};

    # return all available names with current selected
    return [ map { { label       => $_,
                     value       => $_,
                     is_selected => $_ eq $value ? 1 : undef,
                 } } DateTime::TimeZone->all_names ];
}

=item C<< $self->generate_xml() >>

This class overrides generate_xml() in order to update the
Document_TimeRange table when the user submits a change.

=cut

# it might be better to have an explicit on_save() hook available,
# since it's possible that something else might want to use
# generate_xml() someday.  At the moment it appears to be used only to
# save changes to the database making it acceptable for this use.

sub generate_xml {
    my $self = shift;
    $self->write_index();
    return $self->SUPER::generate_xml(@_);
}

=item C<< $self->write_index() >>

Called by generate_xml() to write to the index table,
Document_TimeRange.  Returns 1 on success.

=cut

sub write_index {
    my $self = shift;
    my $dbh = lib::sql::DBH->get();
    my $id = $self->Document_TimeRange_ID;

    # normalize the dates to UTC to make searches easier and format
    # them for MySQL
    my ($from_date, $to_date) =
      map { my $date = $self->_gen_datetime($_);
            $date->set_time_zone("UTC");
            $date->strftime('%Y-%m-%d %H:%M:00') }
        (qw(from to));

    # insert a new record and get back the new ID
    $dbh->do('INSERT INTO Document_TimeRange 
                  (Document_ID, FromDate, ToDate) VALUES (?, ?, ?)',
             undef, $self->parent_id, $from_date, $to_date);
    $self->set_Document_TimeRange_ID($dbh->{mysql_insertid});

    return 1;
}

=back

=cut

1;
