# -------------------------------------------------------------------------------------
# flo::plugin::Admin::Sort
# -------------------------------------------------------------------------------------
# Author : Jean-Michel Hiver
# Copyright : (c) MKDoc Holdings Ltd, 2003
#
# This file is part of MKDoc. 
# 
# MKDoc is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# MKDoc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with MKDoc; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#
# Lets an editor manually sort the child documents of the current document.
# -------------------------------------------------------------------------------------
package flo::plugin::Admin::Sort;
use flo::Standard;
use strict;
use warnings;
use 5.008_000;
use Carp;

use base qw /flo::Plugin/;

sub template_path { 'admin/sort' }


##
# $self->activate;
# ----------------
# Returns TRUE if this plugin can be activated, FALSE
# otherwise.
##
sub activate
{
    my $self = shift;
    return unless ($self->SUPER::activate (@_));
    my $current_document = flo::Standard::current_document();
    return $current_document->Can_Admin;
}


sub http_get
{
    my $self = shift;
    $self->render_http (
	self       => $self,
	__input__  => 'XML',
	__output__ => 'XHTML',
	param_list => [ $self->cgi()->param() ],
       );
    
    return 'TERMINATE';
}


sub http_post
{
    my $self = shift;
    my $document_t = flo::Standard::table ('Document');
    my $document = flo::Standard::current_document();
    
    if ($self->sort_by() eq 'title')
    {
	$document->set_sort_method (flo::Record::Document::SORT_METHOD_TITLE());
	$document_t->modify ($document);
    }
    elsif ($self->sort_by() eq 'date_created')
    {
	$document->set_sort_method (flo::Record::Document::SORT_METHOD_CREATED());
	$document_t->modify ($document);
    }
    elsif ($self->sort_by() eq 'date_last_modified')
    {
	$document->set_sort_method (flo::Record::Document::SORT_METHOD_LAST_MODIFIED());
	$document_t->modify ($document);
    }
    else
    {
	$document->set_sort_method (flo::Record::Document::SORT_METHOD_POSITION());
	$document_t->modify ($document);
	
	my $dbh = lib::sql::DBH->get();
	my $sth = $dbh->prepare_cached ('UPDATE Document SET Sibling_Position=? WHERE ID=?');
	my @children = $self->children();
	for (my $i=0; $i < @children; $i++)
	{
	    my $child = $children[$i];
	    my $position = $i + 1;
	    my $id = $child->id();
	    $sth->execute ($position, $child->id());
	}
    }
    
    $self->redirect();
    return 'TERMINATE';
}


sub uri_by_title
{
    my $self = shift;
    my $child = shift;
    my @children = $self->children();
    my $cgix = flo::Standard::cgi()->new();
    for ($cgix->param) { $cgix->delete ($_) }
    
    my $pos = 0;
    for (@children) { $cgix->param ($_->uri_name() => ++$pos) }
    $cgix->param ('sort_by', 'title');
    return $cgix->self_url;
}


sub uri_by_date_created
{
    my $self = shift;
    my $child = shift;
    my @children = $self->children();
    my $cgix = flo::Standard::cgi()->new();
    for ($cgix->param) { $cgix->delete ($_) }
    
    my $pos = 0;
    for (@children) { $cgix->param ($_->uri_name() => ++$pos) }
    $cgix->param ('sort_by', 'date_created');
    return $cgix->self_url;
}


sub uri_by_date_last_modified
{
    my $self = shift;
    my $child = shift;
    my @children = $self->children();
    my $cgix = flo::Standard::cgi()->new();
    for ($cgix->param) { $cgix->delete ($_) }
    
    my $pos = 0;
    for (@children) { $cgix->param ($_->uri_name() => ++$pos) }
    $cgix->param ('sort_by', 'date_last_modified');
    return $cgix->self_url;
}


sub uri_by_position
{
    my $self = shift;
    my $child = shift;
    my @children = $self->children();
    my $cgix = flo::Standard::cgi()->new();
    for ($cgix->param) { $cgix->delete ($_) }
    
    my $pos = 0;
    for (@children) { $cgix->param ($_->uri_name() => ++$pos) }
    $cgix->param ('sort_by', 'position');
    return $cgix->self_url;
}


sub move_up_uri
{
    my $self     = shift;
    my $child    = shift;
    my @children = $self->children();

    my $cgix = flo::Standard::cgi()->new();
    my $position = 0;
    foreach (@children)
    {
	my $index = $position;
	$position++;
	$_->equals ($child) and do {
	    return unless ($index);
	    $cgix->param ( $children[$index]->name(),   $position - 1);
	    $cgix->param ( $children[$index-1]->name(), $position);
	    return $cgix->self_url;
	};
    }
    
    return;
}


sub move_down_uri
{
    my $self     = shift;
    my $child    = shift;
    my @children = $self->children();

    my $cgix = flo::Standard::cgi()->new();
    my $position = 0;
    foreach (@children)
    {
	my $index = $position;
	$position++;
	$_->equals ($child) and do {
	    return if ($position >= scalar @children);
	    $cgix->param ( $children[$index]->name(),   $position + 1);
	    $cgix->param ( $children[$index+1]->name(), $position);
	    return $cgix->self_url;
	};
    }
    
    return;
}


sub children
{
    my $self = shift;
    my $doc  = flo::Standard::current_document();
    $self->{'.children'} ||= do {
	my @res  = sort { $self->_sort_from_cgi ($a, $b) } $doc->children();
	\@res;
    };
    
    return wantarray ? @{$self->{'.children'}} : $self->{'.children'};
}


sub _sort_from_cgi
{
    my $self = shift;
    my ($a, $b) = @_;
    $_ = $self->sort_by();
    /title/              and return $self->_sort_from_cgi_by_title ($a, $b);
    /date_created/       and return $self->_sort_from_cgi_by_date_created ($a, $b);
    /date_last_modified/ and return $self->_sort_from_cgi_by_date_last_modified ($a, $b);
    
    return $self->_sort_from_cgi_by_position ($a, $b);
}


sub _sort_from_cgi_by_position
{
    my $self  = shift;
    my $a     = shift;
    my $b     = shift;
    my $cgi   = $self->cgi();
    my $a_num = $cgi->param ($a->name()) || $a->position() || '100000';
    my $b_num = $cgi->param ($b->name()) || $b->position() || '100000';
    return $a_num <=> $b_num;
}


sub _sort_from_cgi_by_date_created
{
    my $self = shift;
    my $a    = shift;
    my $b    = shift;
    my $a_dc = $a->date_created();
    my $b_dc = $b->date_created();
    return $b_dc cmp $a_dc;
}


sub _sort_from_cgi_by_date_last_modified
{
    my $self  = shift;
    my $a     = shift;
    my $b     = shift;
    my $a_dlm = $a->date_last_modified();
    my $b_dlm = $b->date_last_modified();
    return $b_dlm cmp $a_dlm;
}


sub _sort_from_cgi_by_title
{
    my $self = shift;
    my $a    = shift;
    my $b    = shift;
    my $a_t  = $a->title();
    my $b_t  = $b->title();
    return uc ($a_t) cmp uc ($b_t);
}


sub sort_by
{
    my $self = shift;
    my $cgi  = flo::Standard::cgi();
    return $cgi->param ('sort_by') || do {
	my $doc = flo::Standard::current_document();
	my $m   = $doc->sort_method();
	my $res = undef;
	$res = 'title'              if ($m == flo::Record::Document::SORT_METHOD_TITLE());
	$res = 'date_created'       if ($m == flo::Record::Document::SORT_METHOD_CREATED());
	$res = 'date_last_modified' if ($m == flo::Record::Document::SORT_METHOD_LAST_MODIFIED());
	$res = 'position'           if ($m == flo::Record::Document::SORT_METHOD_POSITION());
	$res;
    };
}


sub is_sort_by
{
    my $self = shift;
    my $val  = shift;
    return $self->sort_by() eq $val;
}


sub redirect
{    
    my $self = shift;
    my $cgix = flo::Standard::cgi();
    my $document = flo::Standard::current_document();
    $cgix->delete ($_) for ($cgix->param);
    $cgix->path_info ($document->path);
    print $cgix->redirect ($cgix->self_url);    
    return 'TERMINATE';
}


1;
