# -------------------------------------------------------------------------------------
# lib::Exception
# -------------------------------------------------------------------------------------
#
#       Author : Jean-Michel Hiver.
#    Copyright : (c) MKDoc Holdings Ltd., 2000.
# 
#      Unauthorized modification, use, reuse, distribution or redistribution
#      of this module is stricly forbidden.
#
#    Description:
#
#      Provides a simple, Java-ish try { ... } catch { ... }; mechanism.
#
# -------------------------------------------------------------------------------------
package lib::Exception;
use Exporter;
use strict;
use vars qw /@ISA @EXPORT $AUTOLOAD/;

@ISA    = qw /Exporter/;
@EXPORT = qw /try catch throw/;


##
# __PACKAGE__->new (@_);
# ----------------------
#   Constructs a new lib::Exception object, which is probably
#   going to be thrown somewhere. Anything in @_ is converted
#   into a hash that is blessed in __PACKAGE__.
##
sub new
{
    my $class = shift;
    $class = ref $class || $class;

    my $self = bless { @_ }, $class;

    my $i = 0;
    my $found = 0;

    # in order to provide useful information, we must rewind the stack trace
    # till we find the throw method. From then, we stop at the first method
    # which does not belong to the lib::Exception package.
    while (my @info = caller ($i++))
    {
	if ($found)
	{
	    if ( $info[3] =~ /^.*::try$/   or
		 $info[3] =~ /^.*::catch$/ or
		 $info[3] =~ /^.*::throw$/ or
		 $info[3] eq "(eval)"      or
		 $info[3] =~ /.*::__ANON__$/ )
	    {
		next;
	    }
	    else
	    {
		$self->{package}    = $info[0];
		$self->{filename}   = $info[1];
		$self->{line}       = $info[2];
		$self->{subroutine} = $info[3];
		$self->{hasargs}    = $info[4];
		$self->{wantarray}  = $info[5];
		$self->{evaltext}   = $info[6];
		$self->{is_require} = $info[7];
		last;
	    }
	}
	else
	{
	    if ($info[3] =~ /^.*::throw$/) { $found = 1 }
	}
    }

    return $self;
}


##
# try BLOCK;
# ----------
#   Same as eval BLOCK. See perldoc -f eval.
#
# try BLOCK catch BLOCK;
# ----------------------
#   Executes the code in the try BLOCKED. if
#   an exception is raised, executes the
#   catch block and passes the exception as
#   an argument.
##
sub try (&@)
{
    my ($try, $catch) = (shift, shift);
    
    $@ = undef;
    eval { &$try };
    if ($@)
    {
	unless (ref $@ and ref $@ eq 'lib::Exception')
	{
	    $@ = new lib::Exception ( code => "RUNTIME_ERROR",
				      info => $@ );
	}
	defined $catch or throw $@;
	$catch->($@);
    }
    $@ = undef;
}


# doesn't do much but provides a nice syntaxic sugar.
sub catch (&) { return shift }


##
# throw ($exception)
# ------------------
#   Throws $exception away. if $exception is not an object,
#   wraps it in a lib::Exception object and throws it away.
##
sub throw (@)
{
    my $exception = shift;
    unless (ref $exception and $exception->isa ("lib::Exception"))
    {
	$exception = new lib::Exception ( type => "runtime_error",
					  info => $exception );
    }
    die $exception;
}


##
# $obj->stack_trace;
# ------------------
#   Returns the stack trace string.
##
sub stack_trace
{
    my $i = 0;
    while (my @info = caller ($i++))
    {
	print join "\t", @info;
	print "\n";
    }
}


sub AUTOLOAD
{
    my $self = shift;
    my $name = $AUTOLOAD =~ /.*::(.*)/;
    if (@_ == 0) { return $self->{$name} }
    else         { $self->{$name} = shift }
}


1;



=head1 SYNOPSIS

------------------------------------------------------------------------------------
lib::Exception
------------------------------------------------------------------------------------

      Author : Jean-Michel Hiver.
   Copyright : (c) MKDoc Holdings Ltd., 2000.
 
   This file is part of MKDoc. 
   
   MKDoc is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   MKDoc is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with MKDoc; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

   Description:

      Provides a simple, Java-ish try { ... } catch { ... }; mechanism.

------------------------------------------------------------------------------------

=head2 overview

lib::Exception is a simple module that was designed to implement
a nice looking try / catch error handling system with Perl.

=head2 in a nutshell

	package Foo;
	use lib::Exception;

	sub some_code
	{
		try
		{
			something_dangerous();
		}
		catch
		{
			# do something with this
			my $exception = shift;
			use Data::Dumper;
			print Dumper ($exception);
		};
	}


	sub something_dangerous
	{
		blah blah blah...
		code code code...
		# something is wrong
		throw (new lib::Exception ( code => "SOMETHING_WRONG",
					    info => $@ ) );
	}

=head2 new

	new lib::Exception ( %hash );

new is the constructor for a lib::Exception object. Whenever new is
invoked, it creates a lib::Exception and sets the object with the
following attributes:

package, filename, line, subroutine, hasargs, wantarray, evaltext, and is_require

These attributes are wrapped with accessors, which means that instead of writing:

	my $package = $exception->{package}

You can write

	my $package = $exception->package;


Any extra attributes that you pass in when constructing a lib::Exception
becomes accessible as well, i.e.

	my $exception = new lib::Exception ( foo => bar, baz => buz );
	my $foo = $exception->foo; # foo now contains bar

If course this has some limitations: you cannot give any attributes with the
following names:

new, try, catch, throw, stack_trace, AUTOLOAD, package, filename, line,
subroutine, hasargs, wantarray, evaltext, and is_require.


=head2 try, catch, throw


These functions are prototyped and exported into any namespace that
uses lib::Exception.

Raising an exception can be done using throw:

	sub exception_raise
	{
		throw ( new lib::Exception ( code  => "BIG_PROBLEM",
					     info  => "Python sucks",
					     troll => 1 ) );
	}

Trying something dangerous with a try / catch block. These can be nested indeed.

	sub dangerous
	{
		try
		{
			something_dangerous();
		}
		catch
		{
			my $exception = shift;
			if ($exception->troll)
			{
				try
				{
					something();
				}
				catch
				{
					some_other_thing();
				};
			}
		};
	}

Please note that the syntax is try BLOCK catch BLOCK; Do not forget the semicolon!
(Unless you're at the end of a block, thanks to Perl smartness).


=head2 stack_trace

Does exactly this:

	my $i = 0;
	while (my @info = caller ($i++))
	{
		print join "\t", @info;
		print "\n";
	}
