use strict;
use warnings;
use Test::More qw(no_plan);
use MKDoc;
use DateTime;
use DateTime::TimeZone;

# make sure SITE_DIR is set since MKDoc->init needs it
die "SITE_DIR isn't set.  Please source mksetenv.sh from an installed MKDoc site ".
  "and try again.\n" 
  unless $ENV{SITE_DIR};

# initialize MKDoc, needed for database connection
MKDoc->init;

use_ok('flo::editor::TimeRange');

my $tr = flo::editor::TimeRange->new();
isa_ok($tr, 'flo::editor::TimeRange');

# create from now to an hour from now
my $now = time;
my $later = $now + 60 * 60;
my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = localtime($now);

# get current timezone from DateTime
my $tz = DateTime->now(time_zone => "local")->time_zone->name;
ok($tz);
isa_ok(DateTime::TimeZone->new(name => $tz), 'DateTime::TimeZone');

# setup dates and make sure accessors are working
$tr->set_from_year($year + 1900);
is($tr->from_year(), $year + 1900);
$tr->set_from_month($mon + 1);
is($tr->from_month(), $mon + 1);
$tr->set_from_day($mday);
is($tr->from_day(), $mday);
$tr->set_from_hour($hour);
is($tr->from_hour(), $hour);
$tr->set_from_minute($min);
is($tr->from_minute(), $min);
$tr->set_from_tz($tz);
is($tr->from_tz(), $tz);

# check that from_datetime is working
my $dt = $tr->from_datetime;
isa_ok($dt, 'DateTime');
foreach my $field (qw(year month day hour minute)) {
    my $meth = "from_$field";
    is($dt->$field, $tr->$meth, "Check that $field matches in from_datetime");
}

($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = localtime($later);

$tr->set_to_year($year + 1900);
is($tr->to_year(), $year + 1900);
$tr->set_to_month($mon + 1);
is($tr->to_month(), $mon + 1);
$tr->set_to_day($mday);
is($tr->to_day(), $mday);
$tr->set_to_hour($hour);
is($tr->to_hour(), $hour);
$tr->set_to_minute($min);
is($tr->to_minute(), $min);
$tr->set_to_tz($tz);
is($tr->to_tz(), $tz);

# check that to_datetime is working
$dt = $tr->to_datetime;
isa_ok($dt, 'DateTime');
foreach my $field (qw(year month day hour minute)) {
    my $meth = "to_$field";
    is($dt->$field, $tr->$meth, "Check that $field matches in to_datetime");
}

# test the month selector
my $select = $tr->month_select("from");
is(@$select, 12);
my @selected = grep { $_->{is_selected} } @$select;
is(@selected, 1);
is($selected[0]->{value}, $tr->from_month);

# test the day selector
$select = $tr->day_select("from");
is(@$select, 31);
@selected = grep { $_->{is_selected} } @$select;
is(@selected, 1);
is($selected[0]->{value}, $tr->from_day);

# test the year selector
$select = $tr->year_select("from");
is($select->[0]{value}, (localtime)[5] + 1900);
@selected = grep { $_->{is_selected} } @$select;
is(@selected, 1);
is($selected[0]->{value}, $tr->from_year);

# test the hour selector
$select = $tr->hour_select("from");
is(@$select, 24);
@selected = grep { $_->{is_selected} } @$select;
is(@selected, 1);
is($selected[0]->{value}, $tr->from_hour);

# test the minute selector
$select = $tr->minute_select("from");
is(@$select, 60);
@selected = grep { $_->{is_selected} } @$select;
is(@selected, 1);
is($selected[0]->{value}, $tr->from_minute);

# test the tz selector
$select = $tr->tz_select("from");
ok(@$select > 10);
@selected = grep { $_->{is_selected} } @$select;
is(@selected, 1);
is($selected[0]->{value}, $tr->from_tz);

# test validation
my $bad = flo::editor::TimeRange->new();
isa_ok($tr, 'flo::editor::TimeRange');

# empty title is an error
has_error($bad, 'component/timerange/title_empty');

# test equality detection, both should default to now
$bad->set_title('foo');
has_error($bad, 'component/timerange/dates_equal');

# test reversal detection
$bad->set_from_hour($bad->from_hour + 1);
has_error($bad, 'component/timerange/dates_reversed');

# now it should be ok
$bad->set_to_hour($bad->to_hour + 2);
$bad->validate;
ok(not $bad->has_errors);

# check that a nonsense date fails
$bad->set_from_month(2);
$bad->set_from_day(31);
has_error($bad, "component/timerange/from_invalid");


# set it back good
$bad->set_from_month($mon + 1);
$bad->set_from_day($mday);

# try to write to the index
ok($bad->write_index());
ok($bad->Document_TimeRange_ID);

# check that data is correct
my $dbh = lib::sql::DBH->get();
my ($db_from, $db_to) = $dbh->selectrow_array('SELECT FromDate, ToDate 
                                               FROM Document_TimeRange
                                               WHERE ID = ?', undef, 
                                              $bad->Document_TimeRange_ID);
ok($db_from =~ m/^(\d{4})-(\d{2})-(\d{2})\s+(\d{2}):(\d{2}):(\d{2})$/);
my ($db_year, $db_month, $db_day, $db_hour, $db_minute, $db_second) =
  ($1, $2, $3, $4, $5, $6);

# the date should match the current date in UTC
my $utc = $bad->from_datetime;
$utc->set_time_zone("UTC");
cmp_ok($db_year,  '==', $utc->year);
cmp_ok($db_month, '==', $utc->month);
cmp_ok($db_day,   '==', $utc->day);
cmp_ok($db_hour,  '==', $utc->hour);
cmp_ok($db_second,'==', $utc->second);

# cleanup test data
$dbh->do('DELETE FROM Document_TimeRange WHERE ID = ?', 
         undef, $bad->Document_TimeRange_ID);

#
# Utility Routines
#

# has_error($obj, $key) - calls $obj->validate(), checks that the
# object has an error and that one of the errors matches the specified
# key.  Clears the list of errors before returning.
sub has_error {
    my ($obj, $key) = @_;
    $bad->validate();
    ok($obj->has_errors, "Checking for errors at line " . (caller)[2]);
    ok((grep { $_->is($key) } @{$obj->errors}), 
       "Looking for the $key error at line " . (caller)[2]);
    $obj->{'.errors'} = [];   
}
