# ----------------------------------------------------------------------------
# MKDoc::Handler::GroupAuthz
# ----------------------------------------------------------------------------
# Author: Sam Tregar
# Copyright: (c) MKDoc Holdings Ltd, 2005
#
# This file is part of MKDoc. 
# 
# MKDoc is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# MKDoc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with MKDoc; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# ---------------------------------------------------------------------------
package MKDoc::Handler::GroupAuthz;
use strict;
use warnings;
use Apache::Constants qw/:common/;
use flo::Standard;
use strict;
use Carp;

=head1 NAME

MKDoc::Handler::GroupAuthz - enforce group-based security

=head1 SYNOPSIS

In httpd.conf for an authenticating host (like the default users.* setup):

  PerlAuthzHandler MKDoc::Handler::GroupAuthz

In httpd.conf for a non-authenticating host (like the default www.* setup):

  <Location />
    PerlModule MKDoc::Handler::GroupAuthz
    PerlAuthenHandler MKDoc::Handler::GroupAuthz->null_authen_handler
    PerlAuthzHandler  MKDoc::Handler::GroupAuthz
    AuthName "Group Authorization"
    AuthType GroupAuthz
    require valid-group
  </Location>

=head1 DESCRIPTION

Checks for group authorization in MKDoc's Grp tables.  If the document
requested is assigned to one or more groups in Document_Grp then the
user must be assigned to one of the groups in Editor_Grp.

Users may be assigned to groups in the user editor by the admin.
Groups are managed directly in the database (for now).

=head1 AUTHOR

Sam Tregar <sam@tregar.com>

=cut

sub handler {
    my $r    = shift;
    my $user = $::MKD_USER;

    # get the current document, locally setting PATH_INFO which is
    # needed by the current_document() code but not yet setup by
    # Apache at this stage
    my $document = do {
        local $ENV{PATH_INFO} = $r->path_info;
        flo::Standard::current_document();
    };

    # if it wasn't found then this request can't be group protected
    # (could be a CSS link, an image, etc)
    return OK unless $document;

    # get groups for this document, searching up the tree as needed
    my @doc_group_ids = _find_groups($document);

    # no results means this document is available to all
    return OK unless @doc_group_ids;

    # if the user isn't correctly logged in then they can't see this
    # page
    return FORBIDDEN unless $user and ref $user and $user->can('id');

    # get a list of the user's groups.  (It would be nice to just do a
    # query against Editor_Grp like (editor_id = ? AND (grp_id = ? OR
    # ...)) but I can't figure out how to get lib::sql to do anything
    # that complicated.  SQL replacements are great, huh?)
    my $editor_grp_t = flo::Standard::table('Editor_Grp');
    my $con          = lib::sql::Condition->new(Editor_ID => $user->id);
    my @groups       = $editor_grp_t->select(cols  => 'Grp_ID',
                                             where => $con)->fetch_all;

    # no results means the user wasn't in any of the groups, denied
    return FORBIDDEN unless @groups;

    # allow through if the user is in one of the document's groups
    my %groups = map { ($_->{Grp_ID}, 1) } @groups;
    return OK if grep { $groups{$_} } @doc_group_ids;

    # otherwise, no dice
    return FORBIDDEN;
}

# find all groups relevent to a document, searching up the tree to the
# root
sub _find_groups {
    my $document = shift;
    my $document_grp_t = flo::Standard::table('Document_Grp');

    # get list of all documents to check
    my @documents = ($document, $document->ancestors);

    # get results for each document 
    my %groups;
    foreach my $doc (@documents) {
        my @res = $document_grp_t->select (
	cols => 'Grp_ID',
        where => lib::sql::Condition->new(Document_ID => $doc->id)
                                      )->fetch_all();
        $groups{$_->{Grp_ID}} = 1 for @res;
    }

    return keys %groups;
}


# an authen handler that does nothing.  This is needed to allow the
# group authz mechanism to work on www.* which doesn't do
# authentication.
sub null_authen_handler ($$) { OK }

1;
