# -----------------------------------------------------------------------------
# MKDoc::Site::ConfigWriter::Httpd_Conf
# -----------------------------------------------------------------------------
#       Author : Jean-Michel Hiver
#    Copyright : Copyright (c) 2002 MKDoc Holdings Ltd.
#
# This file is part of MKDoc. 
# 
# MKDoc is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# MKDoc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with MKDoc; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#   Description:
#
#       Writes the global.cfg file using the site config object.
#
# -----------------------------------------------------------------------------
package MKDoc::Site::ConfigWriter::Httpd_Conf;
use strict;
use warnings;
use Carp;

use MKDoc::Site::Config;
use base qw /MKDoc::Site::ConfigWriter/;
use vars qw /$PUBLIC_DOMAIN $USER_DOMAIN $ADMIN_DOMAIN $SERVER_ADMIN $SITE_DIR $MKDOC_DIR/;


sub class
{
    my $class = shift;
    return ref $class || $class;
}


##
# $class->setup();
# ----------------
# Writes the httpd.conf file.
##
sub setup
{
    my $class  = shift;
    
    my $conf = new MKDoc::Site::Config;
    local $PUBLIC_DOMAIN = $class->site_public_domain;
    local $USER_DOMAIN   = $class->site_user_domain;
    local $SERVER_ADMIN  = $conf->{SERVER_ADMIN};
    local $SITE_DIR      = $conf->{SITE_DIR};
    local $MKDOC_DIR     = $conf->{MKDOC_DIR};

    write_httpd_include();
    write_httpd_conf();
    write_httpd_static_conf();
    write_httpd_authenticate_conf();
    write_httpd_env_conf ($conf);
    write_httpd_mkdoc();
}


sub write_httpd_mkdoc
{
    my $data = <<EOF;
#Editor: vim:syn=apache
<Directory $MKDOC_DIR/cgi>
  # directives for mod_perl
  PerlSetEnv PERL5LIB $MKDOC_DIR
  SetHandler perl-script
  PerlHandler Apache::Registry
  PerlSendHeader On

  # directives for CGI. These should work everywhere
  # do not uncomment that bit of text
  Options ExecCGI
  AddHandler cgi-script .cgi
  order allow,deny
  allow from all
  AllowOverride none
</Directory>
EOF

    open FP, ">$SITE_DIR/httpd/httpd-mkdoc.conf" || die "Cannot write-open $SITE_DIR/httpd/httpd-mkdoc.conf. Reason: $@";
    print FP $data;
    close FP;
}


sub write_httpd_include
{
    my $data = <<EOF;
#Editor: vim:syn=apache
Include $SITE_DIR/httpd/httpd.conf
EOF

    open FP, ">$SITE_DIR/httpd.conf" || die "Cannot write-open $SITE_DIR/httpd.conf. Reason: $@";
    print FP $data;
    close FP;
}


sub write_httpd_static_conf
{
    my $data = <<EOF;
#Editor: vim:syn=apache
# This is the file and image uploads directory.
<Directory $SITE_DIR/static>
    order allow,deny
    allow from all
    AllowOverride none
</Directory>
<Directory $MKDOC_DIR/skin>
    # The next 3 lines enable SSI and MultiViews for the default error documents
    Options Includes MultiViews
    AddHandler server-parsed .shtml
    AddType "text/html; charset=UTF-8" .shtml
    order allow,deny
    allow from all
    AllowOverride none
</Directory>
# Error documents in the following directory will only be used if the 
# /.static/css Alias in the httpd.conf file in this directory points to
# $SITE_DIR/resources/skin 
<Directory $SITE_DIR/resources/skin>
    # The next 3 lines enable SSI and MultiViews for site error documents
    Options Includes MultiViews
    AddHandler server-parsed .shtml
    AddType "text/html; charset=UTF-8" .shtml
    order allow,deny
    allow from all
    AllowOverride none
</Directory>
EOF

    open FP, ">$SITE_DIR/httpd/httpd-static.conf" || die "Cannot write-open $SITE_DIR/httpd/httpd-static.conf. Reason: $@";
    print FP $data;
    close FP;
}


sub write_httpd_env_conf
{
    my $conf = shift;

    my @opt  = ();
    $::CACHE_DIR && do {
        push @opt, "# where to cache stuff";
        push @opt, "SetEnv\tMKD__CACHE_DIR\t\"$::CACHE_DIR\"";
        push @opt, "";
    };

    $::TEMP_DIR && do {
        push @opt, "# where to do temp stuff";
        push @opt, "SetEnv\tTMPDIR\t\"$::TEMP_DIR\"";
        push @opt, "";
    };
    my $opt = join "\n", @opt;
    
    my $data = <<EOF;
#Editor: vim:syn=apache
# those env vars were generated by the Makefile, don't touch them!
PerlSetEnv  PERL5LIB                $MKDOC_DIR
PerlSetEnv  MKDOC_DIR               $MKDOC_DIR
PerlSetEnv  SITE_DIR                $SITE_DIR
SetEnv      MKDOC_DIR               $MKDOC_DIR
SetEnv      SITE_DIR                $SITE_DIR
SetEnv      PUBLIC_DOMAIN           $conf->{PUBLIC_DOMAIN}
SetEnv      USER_DOMAIN             $conf->{USER_DOMAIN}

$opt

# Optional stuff...

# Uncomment if you want to change the HTML tidy command.
SetEnv MKD__TIDY                    $MKDOC_DIR/tools/whitelist.sh

# Uncomment if you want to change the number of posts per page on discussion boards.
# SetEnv MKD__PLUGIN_DISCUSSION_PER_PAGE    50

# Uncomment if you want to change the width / height of the photo thumbnails
# SetEnv MKD__PLUGIN_PHOTO_THUMB_X    320
# SetEnv MKD__PLUGIN_PHOTO_THUMB_Y    200

# Uncomment if you want to change the width / height of the photo scaled images
# SetEnv MKD__PLUGIN_PHOTO_SCALED_X   640
# SetEnv MKD__PLUGIN_PHOTO_SCALED_Y   480

# uncomment if you want to change the JPEG quality of photo thumbnails
# SetEnv MKD__PLUGIN_PHOTO_QUALITY    50

# Uncomment if you want to change the amount of returned results
# per page on the search page
# SetEnv MKD__PLUGIN_SEARCH_PER_PAGE  10

# Uncomment if you want to change the depth of the sitemap
# SetEnv MKD__PLUGIN_SITEMAP_DEPTH    2

# Uncomment if you want the following port numbers (comma separated) to be stripped from URLs.
# SetEnv MKD__URL_PORT_STRIP        "80,8080"

# Uncomment if you want port numbers matching the following regex to be stripped from URLs.
# SetEnv MKD__URL_PORT_STRIP_REGEX  "80\\d*"

# Uncomment if you want to change the list of allowed tags in HTML components
# SetEnv MKD__XHTML_FILTER_LIST       $MKDOC_DIR/conf/xhtml_filter.conf

# Uncomment if you want to change the HTML tidy configuration file.
# SetEnv MKD__TIDY_CONF               $MKDOC_DIR/custom_tidy_config.conf

# Uncomment this if you want to change the timezone for this site
# SetEnv TZ                          Europe/London

# Uncomment if you want users to have a page they can edit created when they sign up.
# SetEnv MKD__ACCOUNT_HOMEPAGE_LOCATION /%Y/%B/%L
# Uncomment if you want some default text for the users home page.
# SetEnv MKD__ACCOUNT_HOMEPAGE_CONTENT  $MKDOC_DIR/conf/homepage.txt

# Uncomment this if you want external links to open in new windows.
# SetEnv MKD__EXTERNAL_LINKS_OPEN_IN_NEW_WINDOW  TRUE

EOF

    open FP, ">$SITE_DIR/httpd/httpd-env.conf" || die "Cannot write-open $SITE_DIR/httpd/httpd-env.conf. Reason: $@";
    print FP $data;
    close FP;
}


sub write_httpd_authenticate_conf
{
    my $data = <<EOF;
#Editor: vim:syn=apache
# The following Location directive prevents the custom error
# documents being accessed where as the LocationMatch 
# directives that follow allow anon access to /.static/css
#<Location />
#  PerlAuthenHandler MKDoc::Handler::Authenticate
#  PerlAuthzHandler  MKDoc::Handler::GroupAuthz
#  AuthName "Please enter your user credentials"
#  AuthType Basic
#  require valid-user
#</Location>
<LocationMatch ^\/\$>
  PerlAuthenHandler MKDoc::Handler::Authenticate
  PerlAuthzHandler  MKDoc::Handler::GroupAuthz
  AuthName "Please enter your user credentials"
  AuthType Basic
  require valid-user
</LocationMatch>
<LocationMatch ^\/[a-zA-Z0-9]>
  PerlAuthenHandler MKDoc::Handler::Authenticate
  PerlAuthzHandler  MKDoc::Handler::GroupAuthz
  AuthName "Please enter your user credentials"
  AuthType Basic
  require valid-user
</LocationMatch>
<LocationMatch ^\/\.(comment|account|admin|sitemap|print|headlines|meta|search)>
  PerlAuthenHandler MKDoc::Handler::Authenticate
  PerlAuthzHandler  MKDoc::Handler::GroupAuthz
  AuthName "Please enter your user credentials"
  AuthType Basic
  require valid-user
</LocationMatch>
EOF

    open FP, ">$SITE_DIR/httpd/httpd-authenticate.conf" || die "Cannot write-open $SITE_DIR/httpd/httpd-authenticate.conf. Reason: $@";
    print FP $data;
    close FP;
}


sub write_httpd_conf
{
    my $data = <<EOF;
#Editor: vim:syn=apache
<VirtualHost *>
  ServerName            $PUBLIC_DOMAIN
  ServerAdmin           $SERVER_ADMIN
  CustomLog             $::LOG_DIR/www_combined_log combined
  ErrorLog              $::LOG_DIR/www_error_log
  # If you want to use customised skins uncomment the following line
  # and comment the line after it.
  #Alias /.static/css    $SITE_DIR/resources/skin
  Alias /.static/css    $MKDOC_DIR/skin
  # If you don't want the standard file type icons comment the following line
  Alias /.static/css/mime    $MKDOC_DIR/skin/mime
  Alias /.static        $SITE_DIR/static
  Alias /               $MKDOC_DIR/cgi/mkdoc.cgi/
  Include               $SITE_DIR/httpd/httpd-env.conf
  UseCanonicalName      On

  SetEnv        MKD__PLUGIN_LIST        $MKDOC_DIR/conf/www.conf
  PerlInitHandler                       MKDoc::Handler::Initialize
  Include                               $SITE_DIR/httpd/httpd-static.conf
  Include                               $SITE_DIR/httpd/httpd-mkdoc.conf

  <Location />
    PerlModule MKDoc::Handler::GroupAuthz
    PerlAuthenHandler MKDoc::Handler::GroupAuthz->null_authen_handler
    PerlAuthzHandler  MKDoc::Handler::GroupAuthz
    AuthName "Group Authorization"
    AuthType GroupAuthz
    require valid-group
  </Location>

  # Error documents use MultiViews and Includes
  ErrorDocument 401 /.static/css/401
  ErrorDocument 404 /.static/css/404
  ErrorDocument 403 /.static/css/403
  ErrorDocument 500 /.static/css/500

</VirtualHost>

<VirtualHost *>
  ServerName            $USER_DOMAIN
  ServerAdmin           $SERVER_ADMIN
  CustomLog             $::LOG_DIR/users_combined_log combined
  ErrorLog              $::LOG_DIR/users_error_log
  # If you want to use customised skins uncomment the collowing line
  # and comment the line after it.
  #Alias /.static/css    $SITE_DIR/resources/skin
  Alias /.static/css    $MKDOC_DIR/skin
  Alias /.static        $SITE_DIR/static
  Alias /               $MKDOC_DIR/cgi/mkdoc.cgi/
  Include               $SITE_DIR/httpd/httpd-env.conf
  UseCanonicalName      On

  SetEnv        MKD__PLUGIN_LIST        $MKDOC_DIR/conf/users.conf
  PerlInitHandler                       MKDoc::Handler::Initialize
  Include                               $SITE_DIR/httpd/httpd-authenticate.conf
  Include                               $SITE_DIR/httpd/httpd-static.conf
  Include                               $SITE_DIR/httpd/httpd-mkdoc.conf

  # Error documents use MultiViews and Includes
  ErrorDocument 401 /.static/css/401
  ErrorDocument 404 /.static/css/404
  ErrorDocument 403 /.static/css/403
  ErrorDocument 500 /.static/css/500

</VirtualHost>
EOF

    open FP, ">$SITE_DIR/httpd/httpd.conf" || die "Cannot write-open $SITE_DIR/httpd/httpd.conf. Reason: $@";
    print FP $data;
    close FP;
}


1;


__END__
