# -------------------------------------------------------------------------------------
# flo::Record::Document
# -------------------------------------------------------------------------------------
# Author : Jean-Michel Hiver.
# Copyright : (c) MKDoc Holdings Ltd, 2003.
#
# This file is part of MKDoc. 
# 
# MKDoc is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# MKDoc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with MKDoc; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#
# This class models MKDoc document objects, and holds methods related to document
# hierarchy and metadata validation.
# -------------------------------------------------------------------------------------
package flo::Record::Document;
use Time::Local;
use flo::RedirectManager;
use flo::Standard;
use flo::Editor;
use MKDoc::Util::TemplateFinder;
use MKDoc::Config;
use MKDoc::Ouch;
use Text::Unidecode;
use strict;
use Carp;

# memoized, modperl compatible
our %DC_CREATED_W3C = ();


# avoid silly warning
use Petal;
sub sillyness { @Petal::BASE_DIR };

use constant SORT_CREATED              => 'Date_Created';
use constant SORT_LAST_MODIFIED        => 'Date_Last_Modified';
use constant SORT_TITLE                => 'Title';
use constant SORT_POSITION             => 'Sibling_Position';

use constant ORDER_DESC                => 1;
use constant ORDER_ASC                 => 0;

use constant SORT_METHOD_CREATED       => 1;
use constant SORT_METHOD_LAST_MODIFIED => 2;
use constant SORT_METHOD_TITLE         => 3;
use constant SORT_METHOD_POSITION      => 4;

use constant MAX_NAME_LENGTH           => 46;

use constant MAX_PATH_LENGTH           => 255;
use constant MAX_TITLE_LENGTH          => 255;
use constant MAX_DESC_LENGTH           => 1024;
use constant MAX_KEYWD_LENGTH          => 1024;



##
# $self->showable_class();
# ------------------------
# Returns 'not-showable' if this document is not showable,
# undef otherwise.
##
sub showable_class
{
    my $self = shift;
    return $self->is_showable() ? 'showable' : 'not-showable';
}


##
# $self->is_showable()
# --------------------
# Returns FALSE if this object is hidden or a versioned backup, TRUE otherwise.
##
sub is_showable
{
    my $self = shift;
    return not ( $self->is_hidden()           ||
		 $self->is_modified_version() ||
		 $self->is_deleted_version() );
}

=pod

  $self->is_showable_to_editor()

Returns FALSE if this object is a versioned backup, TRUE otherwise.

=cut

sub is_showable_to_editor
{
    my $self = shift;
    return not ( $self->is_modified_version() ||
		 $self->is_deleted_version() );
}


##
# $self->is_hidden();
# -------------------
# Returns TRUE if this document is private, FALSE otherwise.
##
sub is_hidden
{
    my $self = shift;
    my $path = $self->path();
    my $re   = MKDoc::Config->URI_HIDDEN_REGEX;
    return $path =~ /$re/;
}


##
# $self->is_modified();
# ---------------------
# Returns TRUE if this document is private, FALSE otherwise.
##
sub is_modified_version
{
    my $self = shift;
    my $path = $self->path();
    my $re   = MKDoc::Config->URI_MODIFIED_REGEX;
    return $path =~ /$re/;
}


##
# $self->is_deleted();
# ---------------------
# Returns TRUE if this document is private, FALSE otherwise.
##
sub is_deleted_version
{
    my $self = shift;
    my $path = $self->path();
    my $re   = MKDoc::Config->URI_DELETED_REGEX;
    return $path =~ /$re/;
}


##
# $self->is_root();
# -----------------
# Returns TRUE if this object is the root document,
# returns FALSE otherwise.
##
sub is_root
{
    my $self = shift;
    my $parent_id = $self->parent_id();
    if (defined $parent_id) { return 0 }
    else                    { return 1 }
}


##
# $self->first();
# ---------------
# Returns the first sibling. If this document has no
# sibling, is the first sibling, or is an only child,
# returns undef.
##
sub first
{
    my $self = shift;
    
    exists $self->{'.first'} and return $self->{'.first'};
    $self->{'.first'} = do {
	my $parent   = $self->parent();
	if ($parent)
	{
	    my @siblings = $parent->children();
	    my $first    = shift (@siblings);
	    if ($first)
	    {
		$first->equals ($self) ? undef : $first;
	    }
	    else
	    {
		undef;
	    }
	}
	else
	{
	    undef;
	}
    };
}


##
# $self->first_showable();
# ------------------------
# Returns the first showable document, if any.
##
sub first_showable
{
    my $self  = shift;
    my $first = $self->first() || return;
    return $first if ($first->is_showable());
    return $first->next_showable();
}


##
# $self->last();
# --------------
# Returns the first sibling. If this document has no
# sibling, is the first sibling, or is an only child,
# returns undef.
##
sub last
{
    my $self = shift;
    
    exists $self->{'.last'} and return $self->{'.last'};
    $self->{'.last'} = do {
	my $parent   = $self->parent();
	if ($parent)
	{
	    my @siblings = $parent->children();
	    @siblings    = reverse (@siblings);
	    my $last     = shift (@siblings);
	    if ($last)
	    {
		$last->equals ($self) ? undef : $last;
	    }
	    else
	    {
		undef;
	    }
	}
	else
	{
	    undef;
	}
    };
}


##
# $self->last_showable();
# ------------------------
# Returns the last showable document, if any.
##
sub last_showable
{
    my $self  = shift;
    my $last = $self->last() || return;
    return $last if ($last->is_showable());
    return $last->prev_showable();
}


##
# $self->next();
# --------------
# Returns the next sibling related to this document.
# If the document has no next sibling, returns undef.
##
sub next
{
    my $self = shift;
    exists $self->{'.next'} and return $self->{'.next'};
    $self->{'.next'} = do {
	my $res = undef;
	my $parent = $self->parent() || return;
	my @siblings = $parent->children();
	while (@siblings)
	{
	    $_   = shift (@siblings);
	    $res = shift (@siblings) if ($self->equals ($_));
	}
	$res;
    };
    return $self->{'.next'};
}


##
# $self->next_showable();
# -----------------------
# Returns the next showable sibling.
##
sub next_showable
{
    my $self = shift;
    my $next = $self->next() || return;
    return $next if ($next->is_showable());
    return $next->next_showable();
}


##
# $self->prev();
# --------------
# Returns the previous sibling related to this document.
# If the document has no previous sibling, returns undef.
##
sub prev
{
    my $self = shift;
    exists $self->{'.prev'} and return $self->{'.prev'};
    $self->{'.prev'} = do {
	my $res = undef;
	my $parent = $self->parent() || return;
	my @siblings = reverse $parent->children();
	while (@siblings)
	{
	    $_   = shift (@siblings);
	    $res = shift (@siblings) if ($self->equals ($_));
	}
	$res;
    };
    return $self->{'.prev'};
}


##
# $self->prev_showable();
# -----------------------
# Returns the previous showable sibling.
##
sub prev_showable
{
    my $self = shift;
    my $prev = $self->prev() || return;
    return $prev if ($prev->is_showable());
    return $prev->prev_showable();
}


sub components
{
    my $self = shift;
    
    # Hack...
    local $::MKD_flo_Editor_ETERNAL;
    $::MKD_flo_Editor_ETERNAL = do {
	my $editor = _new flo::Editor();
	$editor->parse_xml ($self->{Body}, $self->id());
	$editor;
    };
    
    my @res = $::MKD_flo_Editor_ETERNAL->access (@_);
    return wantarray ? @res : \@res;
}


sub add_component
{
    my $self = shift;
    my $component = shift;
    
    # Hack...
    local $::MKD_flo_Editor_ETERNAL;
    $::MKD_flo_Editor_ETERNAL = do {
	my $editor = _new flo::Editor();
	$editor->parse_xml ($self->{Body});
	$editor;
    };

    my $editor = $::MKD_flo_Editor_ETERNAL;
    $editor->add_component ($self, $component);
    $self->{Body} = $editor->generate_xml();
}


sub del_component
{
    my $self = shift;
    my $component = shift;
    
    # Hack...
    local $::MKD_flo_Editor_ETERNAL;
    $::MKD_flo_Editor_ETERNAL = do {
	my $editor = _new flo::Editor();
	$editor->parse_xml ($self->{Body});
	$editor;
    };
    
    my $editor = $::MKD_flo_Editor_ETERNAL;
    $editor->del_component ($component);
    $self->{Body} = $editor->generate_xml();
}


##
# $self->new (%args);
# -------------------
# Instanciates a new flo::Record::Document object and validates it.
# If the document is validated, it is saved and returned.
# Otherwise, undef is returned.
##
sub new
{
    my $class = shift;
    $class = ref $class || $class;
    my $self = bless { @_ }, $class;
    
    $self->set_name ($self->_find_name ($self->name() || $self->title()));
    if ($self->validate())
    {
	$self->save();
	return $self;
    }
    
    return;
}


##
# $self->save();
# --------------
# Saves the current document. Warning, does not perform any validation.
##
sub save
{
    my $self = shift;
    my $document_t = flo::Standard::table ('Document');
    if ($self->id())
    {
	$document_t->modify ($self);
    }
    else
    {
	$self->{ID} = $document_t->insert ($self);
    }
}


##
# $self->delete ($redirect_object);
# ---------------------------------
# Deletes the current document, including its associated records,
# child documents, etc.
##
sub delete
{
    my $self = shift;    
    $self->is_root() and return;

    # call delete handler
    $self->on_delete();
    
    $_->delete() for ($self->children());
    $self->_delete_redirect (@_);
    $self->_delete_base_doc (@_);
    $self->_delete_self (@_);
    return 1;
}


##
# $self->copy_to ($parent);
# -------------------------
# Copies $self into $parent, overriding children if necessary.
##
sub copy_to
{
    my $self   = shift;
    my $target = shift;
    my $child  = $target->get_child ($self->name());
    
    (defined $child) ?
        $self->copy_to__shallow_overwrite ($child) :
	$self->copy_to__shallow_new ($target);
}


sub copy_to__shallow_overwrite
{
    my $self  = shift;
    my $child = shift;
    my $id    = $child->id();
    my $p_id  = $child->parent_id();
    my $name  = $child->name();
    my $dc    = $child->date_created();
    my $eci   = $child->editor_created_id();
    
    for my $key (keys %{$self})
    {
	($key =~ /^\w/) and do { $child->{$key} = $self->{$key} }
    }
    
    $child->set_id ($id);
    $child->set_parent_id ($p_id);
    $child->set_name ($name);
    $child->set_date_created ($dc);
    $child->set_editor_created_id ($eci);
    $child->save();
    return $child;
}


sub copy_to__shallow_new
{
    my $self   = shift;
    my $parent = shift;
    
    my $self_path = $self->path();
    my $parent_path = $parent->path();
    
    delete $parent->{'.children'};
    my %args = map { /^\./ ? () : ( $_ => $self->{$_} ) } keys %{$self};
    
    delete $args{ID};
    delete $args{Full_Path};
    $args{Parent_ID} = $parent->id();
    
    my $class = ref $parent;
    
    my @errors = ();
    my $name = $self->name();
    
    local $MKDoc::Ouch::CALLBACK;
    $MKDoc::Ouch::CALLBACK = sub { push @errors, @_ };
    my $obj  = $class->new (%args);
    
    use Data::Dumper;
    die Dumper (@errors) if (scalar @errors);
    return $obj;
}


##
# $self->sync_to ($target);
# -------------------------
# Syncs $self tree onto $target tree so that $target tree
# becomes a perfect copy of $self tree. Also tries to avoid
# work which has become unnecessary.
##
sub sync_to
{
    my $source = shift;
    my $target = shift;
    
    # syncs content from $source node to $target node
    if ($source->date_last_modified() gt $target->date_last_modified())
    {
	$source->sync_to__overwrite ($target);
    }
    else
    {
	$source->sync_to__unchanged ($target);
    }
    
    # removes $target children nodes which do not exist
    # as children of $source
    foreach my $target_child ($target->children())
    {
	my $target_child_name = $target_child->name();
	my $source_child      = $source->get_child ($target_child_name) || $source->sync_to__delete ($target_child);
    }
    
    # inserts $source children nodes which do not exists
    # as children of $target
    foreach my $source_child ($source->children())
    {
	my $source_child_name = $source_child->name();
	my $target_child      = $target->get_child ($source_child_name) || $source->sync_to__insert ($source_child, $target);
    }
    
    # at this point $source and $target should have the same
    # amount of children, and those children need to be synced
    # together
    foreach my $source_child ($source->children())
    {
	delete $target->{'.children'};
	my $source_child_name = $source_child->name();
	my $target_child      = $target->get_child ($source_child_name) || do {
	    warn "Cannot fetch " . $target->path() . $source_child_name . "/";
	    next;
	};
	
	$source_child->sync_to ($target_child);
    }
    
    return 1;
}


sub sync_to__overwrite
{
    my $source = shift;
    my $target = shift;
    $source->copy_to__shallow_overwrite ($target);
}


sub sync_to__unchanged
{
    my $source = shift;
    my $target = shift;
}


sub sync_to__delete
{
    my $source = shift;
    my $target_child = shift;
    $target_child->delete();
}


sub sync_to__insert
{
    my $source = shift;
    my $source_child = shift;
    my $target = shift;
    $source_child->copy_to ($target);
}


##
# $self->move_to ($target_object);
# --------------------------------
# Moves a document as a child of another document.
##
sub move_to
{
    my $self = shift;
    my $parent = shift;

    # cannot move the root document
    $self->is_root() and do {
	new MKDoc::Ouch 'document/move/self_is_root';
	return 0;
    };
    
    # cannot move a document into itself
    $parent->equals ($self) and do {
	new MKDoc::Ouch 'document/move/parent_equals_self';
	return 0;
    };
    
    # cannot move a document into its parent, that where it
    # already is...
    $parent->equals ($self->parent()) and do {
	new MKDoc::Ouch 'document/move/parent_equals_self_parent';
	return 0;
    };
    
    # cannot move a document into one of its descendants
    $parent->is_descendant_of ($self) and do {
	new MKDoc::Ouch 'document/move/parent_is_descendant_of_self';
	return 0;
    };
    
    # cannot move a document into another document which has a child
    # with the same name has the current document
    $parent->get ($self->name()) and do {
	new MKDoc::Ouch 'document/move/name_already_exists';
	return 0;
    };
    
    # cannot move a document into another document if the 'Full_Path' attribute
    # became too long
    my $parent_path  = $parent->path();
    my $self_name    = $self->name();
    my $path_to_be   = $parent_path . $self_name . '/';
    (length ($path_to_be) > MAX_PATH_LENGTH) and do {
	new MKDoc::Ouch 'document/move/path_too_long';
	return 0;
    };
    
    flo::RedirectManager->document_moved ($self, $parent->path());
    $self->set_parent ($parent);
    $self->save();
}


sub is_ancestor_of
{
    my $self = shift;
    my $obj  = shift;
    return $obj->is_descendant_of ($self);
}


sub is_descendant_of
{
    my $self = shift;
    my $obj  = shift;
    
    my $self_path = $self->path();
    my $obj_path  = $obj->path();
    
    return index ($self_path, $obj_path) == 0;
}


sub _delete_redirect
{
    my $self = shift;
    my $doc_to = shift || return;
    flo::RedirectManager->document_deleted ($self, $doc_to->path());
}


sub _delete_base_doc
{
    my $self = shift;
    my $base_document_t = flo::Standard::table ('Base_Document');
    $base_document_t->delete (Document_ID => $self->id());
}


sub _delete_self
{
    my $self = shift;
    my $document_t = flo::Standard::table ('Document');
    $document_t->delete (ID => $self->id());
}


##
# $self->validate();
# ------------------
# Validates this document to check wether it's OK
# for insertion / modification.
##
sub validate
{
    my $self = shift;
    return $self->validate_parent_id()               &
	   $self->validate_name()                    &
	   $self->validate_path()                    &
	   $self->validate_title()                   &
	   $self->validate_description()             &
	   $self->validate_keywords()                &
	   $self->validate_template()                &
	   $self->validate_lang()                    &
	   $self->validate_sort_method()             &
	   $self->validate_editor_created_id()       &
	   $self->validate_editor_last_modified_id() &
	   $self->validate_cache_control()           &
	   $self->validate_date_created()            &
	   $self->validate_date_last_modified();
}


sub _find_name
{
    my $self = shift;
    my $name = $self->normalize_name (shift) || 'document';
    my $document_table = flo::Standard::table ('Document');
    
    if (defined $document_table->get ( Parent_ID => $self->parent_id, Name => $name ))
    {
	my $count = 2;
	my $new_name = $name . '-' . $count;
	while (defined $document_table->get ( Parent_ID => $self->parent_id, Name => $new_name ))
	{
	    $count++;
	    $new_name = $name . '-' . $count;
	}
	$name = $new_name;
    }
    
    return $name;
}


##
# $self->id();
# ------------
# Returns the ID of this document.
##
sub id
{
    my $self = shift;
    return $self->{ID};
}


##
# $self->set_id ($id);
# --------------------
# Sets the id of this document. It's here because it's
# nice and symetric, but really you should never be using
# this method.
##
sub set_id
{
    my $self = shift;
    $self->{ID} = shift;
}


##
# $self->parent_id();
# -------------------
# Returns the parent id of this document.
##
sub parent_id
{
    my $self = shift;
    return $self->{Parent_ID};
}


##
# $self->set_parent_id ($parent_id);
# ----------------------------------
# Sets the parent id of this document.
##
sub set_parent_id
{
    my $self = shift;
    $self->{Parent_ID} = shift;
}


sub validate_parent_id
{
    my $self = shift;
    my $parent_id = $self->parent_id();
    return 1 unless (defined $parent_id);
    
    my $document_t = flo::Standard::table ('Document');
    my $parent = $document_t->get ($parent_id);
    return 1 if (defined $parent);
    
    new MKDoc::Ouch 'document/parent_id';
    return 0;
}


##
# $self->name;
# ------------
# Returns the name ('URI name') of this document.
##
sub name
{
    my $self = shift;
    return $self->{Name};
}


##
# $self->set_name ($name);
# ------------------------
# Sets the name ('URI name') of this document.
##
sub set_name
{
    my $self = shift;
    $self->{Name} = $self->normalize_name (shift, $self->lang());
}


sub validate_name
{
    my $self  = shift;
    my $name  = $self->name();
    
    # if $self is the root document and $name is defined, then
    # it's an error because the root document is the only document
    # which hasn't got a name.
    if (not $self->parent())
    {
	return 1 unless (defined $name and $name);
	new MKDoc::Ouch 'document/name/defined_but_root';
	return 0;
    }
    
    # if the method has not returned yet at this point, then the
    # document is not a root document. We need to check that $name
    # is defined...
    unless (defined $name and $name)
    {
	new MKDoc::Ouch 'document/name/empty';
	return 0;
    }
    
    # ... and we need to checked that the normalized version of
    # $name has no sibling with the same name.
    $name = $self->normalize_name ($name, $self->lang);
    if ($self->parent)
    {
	foreach my $document ($self->parent->children)
	{
	    next if (defined $self->id() and $document->id() == $self->id());
	    if ($document->name() eq $name)
	    {
		new MKDoc::Ouch 'document/name/exists';
		return 0;
	    }
	}
    }
    
    return 1;
}


##
# $self->normalize_name ($name);
# ------------------------------
# Returns the nornalized $name, i.e.
# --Hello, World !!! would return hello-world.
#
# $name is the name to be normalized.
##
sub normalize_name
{
    my $self = shift;
    my $name = unidecode (shift); # was MKDoc:: Lang
    my $lang = $self->lang();
        
    # turn anything that's weird into hyphens
    $name =~ s/_/-/g;
    $name =~ s/\s/-/g;
    $name =~ s/[^A-Za-z0-9-]//g;
    
    # lowercase everything
    $name = lc ($name);
    
    # temp until 'new document' is refactored, when that's
    # done then MAX_NAME_LENGTH should be a constant in the
    # flo::Record::Document class
    $name = substr ($name, 0, MAX_NAME_LENGTH());
    
    # remove beginning hyphens
    $name =~ s/^\-*//;
    
    # remove ending hyphens
    $name =~ s/\-*$//;
    
    # squash consecutive hyphens
    $name =~ s/\-+/-/g;
    
    return $name;
}


##
# $self->parent;
# --------------
# Returns the parent document object of the current object. If the current
# object has no parents (in which case it's the root document), returns
# undef.
##
sub parent
{
    my $self = shift;
    my $path = $self->path();
    return if (defined $path and $path eq '/');
    
    $self->{'.parent'} ||= do {
	my $document_t = flo::Standard::table ('Document');
	$document_t->get ($self->parent_id());
    };
    
    return $self->{'.parent'};
}


sub parent_showable
{
    my $self = shift;
    return $self->parent (@_);
}



##
# $self->set_parent ($parent);
# ----------------------------
# Sets the parent document of this document to $parent.
##
sub set_parent
{
    my $self = shift;
    my $parent = shift;
    $self->set_parent_id ($parent->id);
}


sub components_list
{
    my $self = shift;
    warn "'components_list' is deprecated - use 'components' instead";
    return $self->components (@_);
}


##
# $self->get ($name);
# -------------------
# Returns this document's child object, wether it's a document or a component,
# called $name. If there is no child object called $name, returns undef.
##
sub get
{
    my $self = shift;
    my $name = shift;
    if ($name =~ /\./) { return $self->get_component ($name) }
    else               { return $self->get_child     ($name) }
}


sub get_child
{
    my $self = shift;
    my $name = shift;
    my @res  = map { ($_->name() eq $name) ? $_ : () } $self->children();
    (scalar @res) ? return shift (@res) : return;
}


sub get_component
{
    my $self = shift;
    my $name = shift;
    
    foreach my $component ($self->components())
    {
	next unless ($component->can ('name'));
	return $component if ($component->name() eq $name);
    }
    return;
}


sub uri_name
{
    my $self = shift;
    return $self->{Name};
}


##
# $self->replaces();
# ------------------
# Returns a list of URIs which this document replaces. This means that
# HTTP get on any of the returned URI should perform a redirect to the
# URI() of this document.
##
sub replaces
{
    my $self = shift;
    my $redirect_table = flo::Standard::table ('Redirect');
    my @redirects =  $redirect_table->search ( New_Path => $self->{Full_Path} )->fetch_all;
    return (wantarray) ? @redirects : \@redirects;
}


##
# $self->is_current();
# --------------------
# Returns TRUE if this document is the same as the current document,
# FALSE otherwise.
##
sub is_current
{
    my $self = shift;
    my $current_document = flo::Standard::current_document();
    return $self->equals ($current_document);
}


##
# $self->equals ($document);
# --------------------------
# Returns TRUE if this document has the same id as $document,
# FALSE otherwise.
##
sub equals
{
    my $self = shift;
    my $document = shift;
    my $res = eval { $self->id == $document->id };
    $@ and Carp::confess ($@);
    return $res;
}


##
# $self->ancestors();
# -------------------
# Returns the ancestors of that document, i.e. its parent,
# grand-parent, grand-grand parent, etc... the root element
# being the first element of the array which is being returned.
##
sub ancestors
{
    my $self = shift;
    my $parent = $self->parent;
    return (wantarray) ? () : [] unless (defined $parent);
    
    my @res = ($parent->ancestors, $parent);
    return (wantarray) ? @res : \@res;
}


##
# $self->root_child();
# --------------------
# Returns the ancestor of this document that is a child of the root document.
##
sub root_child 
{
    my $self = shift;
    my $ancestors = $self->ancestors;
    return unless $ancestors;
    return $self unless $ancestors->[1];
    return $ancestors->[1];
}


##
# $self->position();
# ------------------
# Gets the sibling position of that document, i.e. where it
# ranks compared with its siblings.
##
sub position
{
    my $self = shift;
    return $self->{Sibling_Position};
}


##
# $self->set_position ($sibling_position);
# ----------------------------------------
# Sets the sibling position of this document.
##
sub set_position
{
    my $self = shift;
    $self->{Sibling_Position} = shift;
}


##
# $self->path();
# --------------
# Gets this document's path, i.e. /foo/bar/baz/.
##
sub path
{
    my $self = shift;
    return $self->{Full_Path};
}


sub validate_path
{
    my $self = shift;
    return 1 if ($self->is_root());
    
    my $path = $self->parent()->path() . $self->name() . '/';
    (length ($path) > MAX_PATH_LENGTH) and do {
	new MKDoc::Ouch 'document/path/too_long';
	return 0;
    };

    return 1;
}



##
# $self->title();
# ---------------
# Gets the title of this document.
##
sub title
{
    my $self = shift;
    return $self->{Title};
}


##
# $self->set_title();
# -------------------
# Sets the title of this document.
##
sub set_title
{
    my $self = shift;
    my $val  = shift;
    $val =~ s/^(?:\s|\n|\r)+//gsm;
    $val =~ s/(?:\s|\n|\r)+$//gsm;
    $self->{Title} = $val;
}


sub validate_title
{
    my $self  = shift;
    my $title = $self->title();
    $title =~ s/^(?:\s|\n|\r)+//gsm;
    $title =~ s/(?:\s|\n|\r)+$//gsm;

    if ($title eq '')
    {
	new MKDoc::Ouch 'document/title/empty';
	return 0;
    }
    
    if (length ($title) > MAX_TITLE_LENGTH)
    {
	new MKDoc::Ouch 'document/title/too_long';
	return 0;
    }
    
    return 1;
}


##
# $self->description();
# ---------------------
# Gets the description of this document
##
sub description
{
    my $self = shift;
    return $self->{Description};
}


##
# $self->set_description();
# -------------------------
# Sets the description of this document.
##
sub set_description
{
    my $self = shift;
    my $val  = shift;
    $val =~ s/^(?:\s|\n|\r)+//gsm;
    $val =~ s/(?:\s|\n|\r)+$//gsm;
    $self->{Description} = $val;
}


sub validate_description
{
    my $self = shift;
    my $description = $self->description();
    $description =~ s/^(?:\s|\n|\r)+//gsm;
    $description =~ s/(?:\s|\n|\r)+$//gsm;
    
    unless (defined $description and $description)
    {
	new MKDoc::Ouch 'document/description/empty';
	return 0;
    }
    
    if (length ($description) > MAX_DESC_LENGTH)
    {
	new MKDoc::Ouch 'document/description/too_long';
	return 0;
    }
    
    return 1;
}


##
# $self->subjects();
# ------------------
# Splits the keywords() string into a list of words and
# returns the list.
##
sub subjects
{
    my $self = shift;
    my $keywords = $self->keywords;
    my @keywords = split /,\s+/, $keywords;
    return wantarray ? @keywords : \@keywords;
}


##
# $self->keywords();
# ------------------
# Gets the keywords field of this document (comma separated list).
##
sub keywords
{
    my $self = shift;
    return $self->{Keywords};
}


##
# $self->set_keywords ($keywords);
# --------------------------------
# Sets the keywords field of this document.
##
sub set_keywords
{
    my $self = shift;
    my $val  = shift;
    $val =~ s/^(?:\s|\n|\r)+//gsm;
    $val =~ s/(?:\s|\n|\r)+$//gsm;
    $self->{Keywords} = $val;
}


sub validate_keywords
{
    my $self = shift;
    my $keywords = $self->keywords();
    $keywords =~ s/^(?:\s|\n|\r)+//gsm;
    $keywords =~ s/(?:\s|\n|\r)+$//gsm;

    unless (defined $keywords and $keywords)
    {
	new MKDoc::Ouch 'document/keywords/empty';
	return 0;
    }
    
    if (length ($keywords) > MAX_KEYWD_LENGTH)
    {
	new MKDoc::Ouch 'document/keywords/too_long';
	return 0;
    }
    
    return 1;
}


##
# $self->template();
# ------------------
# Gets the name of the template which is being used for this document.
##
sub template
{
    my $self = shift;
    return $self->{Template};
}


##
# $self->set_template ($template);
# --------------------------------
# Sets the name of the template which is being used for this document.
##
sub set_template
{
    my $self = shift;
    $self->{Template} = shift;
}


sub validate_template
{
    my $self = shift;
    my $template = $self->template();
    unless (defined $template and $template)
    {
	new MKDoc::Ouch 'document/template/empty';
	return 0;
    }
    
    ($template eq $_) ? return 1 : () for ($self->available_templates);
    
    new MKDoc::Ouch 'document/template/not_in_list';
    return 0;
}


sub available_templates
{
    my $class = shift;
    return MKDoc::Util::TemplateFinder::list ("document");
}


sub popular_children_template
{
    my $self = shift;
    my @children = $self->children();
    return $self->template() unless (scalar @children);
    
    my %templates = ();
    foreach my $child (@children)
    {
	my $child_template = $child->template();
	if (defined $templates{$child_template}) { $templates{$child_template}++   }
	else                                     { $templates{$child_template} = 1 }
    }
    
    my @res = sort { $templates{$b} <=> $templates{$a} } keys %templates;
    return $res[0];
}


=cut

sub popular_children_cache_control
{
    my $self = shift;
    my @children = $self->children();
    return $self->cache_control() unless (scalar @children);
    
    my %cache_controls = ();
    foreach my $child (@children)
    {
	my $child_cache_control = $child->cache_control();
	if (defined $cache_controls{$child_cache_control}) { $cache_controls{$child_cache_control}++   }
	else                                               { $cache_controls{$child_cache_control} = 1 }
    }
    
    my @res = sort { $cache_controls{$b} <=> $cache_controls{$a} } keys %cache_controls;
    return $res[0];
}

=cut


##
# $self->lang();
# --------------
# Gets the current language ISO code for this document.
##
sub lang
{
    my $self = shift;
    return $self->{Lang};
}


##
# $self->set_lang ($lang_iso_code);
# ---------------------------------
# Sets the current language ISO code for this document.
##
sub set_lang
{
    my $self = shift;
    $self->{Lang} = shift;
}


sub validate_lang
{
    my $self = shift;
    my $lang = $self->lang();
    
    unless (defined $lang and $lang ne '')
    {
	new MKDoc::Ouch 'document/lang/empty';
	return 0;
    }
    
    ($lang eq $_) ? return 1 : () for ($self->available_langs);
    
    new MKDoc::Ouch 'document/lang/not_in_list';
    return 0;
}


sub available_langs
{
    my $self    = shift;
    my $default = shift || $self->lang || 'en';

    my $cfg = new MKDoc::Config->parsefile_hashref (MKDoc::Config->LANGUAGE_LIST);
    my @res = sort { $cfg->{$a} cmp $cfg->{$b} } keys %{$cfg};
    return (wantarray) ? @res : \@res;
}


sub used_langs
{
    my $class = shift;
    my $sql   = <<'EOF';
SELECT DISTINCT (Lang)
FROM Document
WHERE Name NOT LIKE 'deleted-____-__-__-__-__-__'  AND
      Name NOT LIKE 'modified-____-__-__-__-__-__' AND
      Name NOT LIKE 'private-%'                    AND
      Name NOT LIKE 'hidden-%'
EOF
    
    my $dbh = lib::sql::DBH->get();
    $dbh->prepare ($sql);

    my @res = ();

    my $sth = $dbh->prepare ($sql);
    $sth->execute();
    while (my $array_ref = $sth->fetchrow_arrayref())
    {
        push @res, $array_ref->[0];
    }

    return wantarray ? @res : \@res;
}


sub lang_label
{
    my $self = shift;
    my $lang = shift || $self->language;
    my $hash = MKDoc::Config->parsefile_hashref (MKDoc::Config->LANGUAGE_LIST);
    return exists $hash->{$lang} ? $hash->{$lang} : return;
}


##
# $self->sort_by;
# ---------------
#   Returns the column by which to sort children document of this document.
#   The values can be any of the following:
#
#     $flo::Record::Document::SORT_CREATED
#     $flo::Record::Document::SORT_LAST_MODIFIED
#     $flo::Record::Document::SORT_TITLE
##
sub sort_by
{
    my $self = shift;
    return $self->{Sort_By};
}


##
# $self->set_sort_by ($sort_by);
# ------------------------------
#   $sort_by - Column to sort on.
#
#   Returns the column by which to sort children document of this document.
#   The values can be any of the following:
#
#     $flo::Record::Document::SORT_CREATED
#     $flo::Record::Document::SORT_LAST_MODIFIED
#     $flo::Record::Document::SORT_TITLE
##
sub set_sort_by
{
    my $self = shift;
    $self->{Sort_By} = shift;
}


##
# $self->order_by;
# ----------------
#   Gets the order by which to sort the children document of this document.
#   The values can be any of the following:
#
#     $flo::Record::Document::ORDER_DESC - Descending order
#     $flo::Record::Document::ORDER_ASC  - Ascending order
##
sub order_by
{
    my $self = shift;
    return $self->{Order_By};
}


##
# $self->set_order_by ($order_by);
# --------------------------------
#   $order_by - Order by which to sort.
#
#   Sets the order by which to sort the children document of this document.
#   The values can be any of the following:
#
#     $flo::Record::Document::ORDER_DESC - Descending order
#     $flo::Record::Document::ORDER_ASC  - Ascending order
##
sub set_order_by
{
    my $self = shift;
    $self->{Order_By} = shift;
}


##
# $self->sort_method;
# -------------------
#   Returns the current sort method for this document's children documents,
#   or undef if the current sort method is unknown.
#
#   The values can be any of the following:
#
#     $flo::Record::Document::SORT_METHOD_CREATED
#     $flo::Record::Document::SORT_METHOD_LAST_MODIFIED
#     $flo::Record::Document::SORT_METHOD_TITLE
#     $flo::Record::Document::SORT_METHOD_POSITION
#     undef
##
sub sort_method
{
    my $self = shift;
    $self->sort_by eq SORT_CREATED       and $self->order_by == ORDER_DESC and return SORT_METHOD_CREATED;
    $self->sort_by eq SORT_LAST_MODIFIED and $self->order_by == ORDER_DESC and return SORT_METHOD_LAST_MODIFIED;
    $self->sort_by eq SORT_TITLE         and $self->order_by == ORDER_ASC  and return SORT_METHOD_TITLE;
    $self->sort_by eq SORT_POSITION      and $self->order_by == ORDER_ASC  and return SORT_METHOD_POSITION;
    return;
}


##
# $self->set_sort_method ($sort_method);
# --------------------------------------
#   $sort_method - method to sort by.
#
#   Returns the column by which to sort children document of this document.
#   The values can be any of the following:
#
#     $flo::Record::Document::SORT_METHOD_CREATED
#     $flo::Record::Document::SORT_METHOD_LAST_MODIFIED
#     $flo::Record::Document::SORT_METHOD_TITLE
#     $flo::Record::Document::SORT_METHOD_POSITION
##
sub set_sort_method
{
    my $self = shift;
    my $method = shift;
    if ($method == SORT_METHOD_CREATED)
    {
	$self->set_sort_by (SORT_CREATED);
	$self->set_order_by (ORDER_DESC);
    }
    elsif ($method == SORT_METHOD_LAST_MODIFIED)
    {
	$self->set_sort_by (SORT_LAST_MODIFIED);
	$self->set_order_by (ORDER_DESC);
    }
    elsif ($method == SORT_METHOD_TITLE)
    {
	$self->set_sort_by (SORT_TITLE);
	$self->set_order_by (ORDER_ASC);
    }
    elsif ($method == SORT_METHOD_POSITION)
    {
	$self->set_sort_by (SORT_POSITION);
	$self->set_order_by (ORDER_ASC);	
    }
}


sub validate_sort_method
{
    my $self = shift;
    $self->sort_method() || $self->set_sort_method (SORT_METHOD_TITLE);
    return 1;
}


##
# $self->children;
# ----------------
# Returns a list of all sub-documents held by that document
##
sub children
{
    my $self = shift;
    my $args = { @_ };
    
    $self->{'.children'} ||= do {
	my $document_table = flo::Standard::table ('Document');
	my $sort_by = $self->sort_by || 'Sibling_Position';
	my $query = $document_table->select ( where => { Parent_ID => $self->id },
					      sort  => [ $sort_by ],
					      desc  => ($self->order_by) ? 1 : 0 );
	my $children = $query->fetch_all;
	
	# my @children = map { $_->is_showable() ? $_ : () } @{$children};
	# \@children;
	$children;
    };
    
    my @res = @{$self->{'.children'}};
    
    my $with = $args->{'with'};
    $with and do { @res = map { @{$_->components ($with) } ? $_ : () } @res };
    
    my $without = $args->{'without'};
    $without and do { @res = map { @{$_->components ($without) } ? () : $_ } @res };
    
    return wantarray ? @res : \@res;
}


##
# $self->children_showable();
# ---------------------------
# Same as children(), except returns only showable documents.
##
sub children_showable
{
    my $self = shift;
    my @res;
    if (flo::Standard::current_user && flo::Standard::current_user->is_editor)
    {
        @res  = map { $_->is_showable_to_editor() ? $_ : () } $self->children;
    }
    else
    {
        @res  = map { $_->is_showable() ? $_ : () } $self->children;
    }
    return wantarray ? @res : \@res;
}


##
# $self->children_showable_percent_each()
# ---------------------------------------
# returns 25% if there are 4 children, 12.5% if there are eight etc..
# css percentages are decimal numbers, not necessarily integers
# http://www.w3.org/TR/REC-CSS2/syndata.html#value-def-number
##
sub children_showable_percent_each
{
    my $self = shift;
    my $children_showable = $self->children_showable;
    my $number = @$children_showable || return;
    return (100 / $number) . '%';
}


##
# $self->cousins_homonyms();
# --------------------------
# Returns a list of cousins which are also homonyms.
##
sub cousin_homonyms
{
    my $self = shift;
    my $self_id        = $self->id();
    my $self_level     = $self->path() =~ tr/\//\//;
    my $self_parent_id = $self->parent_id() || return;
    my $self_name      = $self->name()      || return;
    my $args           = { @_ };
    
    my $document_t = flo::Standard::table ('Document');
    
    my $sql  = <<EOF;
SELECT ID, Parent_ID, Full_Path, Name
FROM Document
WHERE Name = ?
ORDER BY Full_Path ASC
EOF
    
    my $dbh = lib::sql::DBH->get();
    my $sth = $dbh->prepare_cached ($sql);
    $sth->execute ($self_name);
    
    my @res = ();
    while (my $h = $sth->fetchrow_hashref())
    {
	$h->{Level} = $h->{Full_Path} =~ tr/\//\//;
	push @res, $h;
    }
    
    # fetch all the cousin homonyms
    @res = map {
	(
	    ($_->{Level} == $self_level) and
	    ($_->{Name}  eq $self_name)
	) ? $document_t->get ($_->{ID}) : ()
    } @res;
    
    @res = $self->_homonyms_filter_prefix ($args, @res);
    @res = $self->_homonyms_objectify (@res);
    return wantarray ? @res : \@res;
}


##
# $self->cousin_homonyms_showable();
# ----------------------------------
# Same as cousin_homonyms(), except returns only showable documents.
##
sub cousin_homonyms_showable
{
    my $self = shift;
    my @res  = map { $_->is_showable() ? $_ : () } $self->cousin_homonyms();
    return wantarray ? @res : \@res;
}


##
# $self->_homonyms_objectify (@_);
# --------------------------------
# Loads all the hashrefs stored in @_ and returns
# them as full fledget objects.
##
sub _homonyms_objectify
{
    my $self  = shift;
    my $doc_t = flo::Standard::table ('Document');
    return map { $doc_t->get ($_->{ID}) } @_;
}


##
# $self->homonyms();
# ------------------
# Returns a list of documents which have the same name
# as this document.
##
sub homonyms
{
    my $self = shift;
    my $args = { @_ };

    my $document_t = flo::Standard::table ('Document');
    my $q = $document_t->select ( cols  => '*',
				  sort  => [ qw /Full_Path/ ],
				  desc  => 0,
				  where => { Name => $self->name() } );
    
    my @res = ();
    while (my $h = $q->next()) { push @res, $h }
 
    @res = $self->_homonyms_filter_prefix ($args, @res);
    @res = $self->_homonyms_objectify (@res);
    return wantarray ? @res : \@res;
}


##
# $self->homonyms_showable();
# ---------------------------
# Same as homonyms(), except returns only showable documents.
##
sub homonyms_showable
{
    my $self = shift;
    my @res  = map { $_->is_showable() ? $_ : () } $self->homonyms();
    return wantarray ? @res : \@res;
}


##
# $self->_homonyms_filter_prefix ($args, @res);
# ---------------------------------------------
# If $args->{prefix} is defined, returns elements
# in @res which Full_Path start with $args->{prefix}
##
sub _homonyms_filter_prefix
{
    my $self   = shift;
    my $args   = shift;
    my $prefix = $args->{prefix} || return @_;   
    return map { $_->{Full_Path} =~ /^\Q$prefix\E/ ? $_ : () } @_;
}


##
# $self->root;
# ------------
#   returns the root document
##
sub root
{
    my $document_table = flo::Standard::table ('Document');
    my $query = $document_table->select ( where => { Full_Path => '/' } ); 
    return $query->next;
}


##
# $self->contributors;
# --------------------
#   Returns an array of Editor objects that represent the editors
#   who contributed to the document at some point
##
sub contributors
{
    my $self = shift;
    my $contributor_table = flo::Standard::table ('Contributor');
    my @res = $contributor_table->search ( Document_ID => $self->id )->fetch_all;
    @res = map { ($_->editor_id == $self->editor_created_id) ? () : $_->editor } @res;
    return (wantarray) ? @res : \@res;
}


##
# $self->editor_created_id;
# -------------------------
#   Gets the ID of the editor who created that document
##
sub editor_created_id
{
    my $self = shift;
    return $self->{Editor_Created_ID};
}


##
# $self->set_editor_created_id ($editor_created_id);
# --------------------------------------------------
#   $editor_created_id - The ID of the editor who created the document
#
#   Sets the ID of the editor who created the document
##
sub set_editor_created_id
{
    my $self = shift;
    $self->{Editor_Created_ID} = shift;
}


##
# $self->editor_created;
# ----------------------
#   Returns the editor who created that document
##
sub editor_created
{
    my $self = shift;
    my $editor_table = flo::Standard::table ('Editor');
    return $editor_table->get ($self->editor_created_id);
}


##
# $self->set_editor_created ($editor);
# ------------------------------------
#   $editor - The object to extract the editor ID from
#
#   Sets the Editor_Created_ID from the $editor object
##
sub set_editor_created
{
    my $self = shift;
    my $editor = shift;
    $self->set_editor_created_id ($editor->id);
}


sub validate_editor_created_id
{
    my $self = shift;
    my $editor_created = $self->editor_created;
    unless (defined $editor_created)
    {
	new MKDoc::Ouch 'document/editor_created';
	return 0;
    }
    
    return 1;
}


##
# $self->editor_last_modified_id;
# -------------------------------
#   Gets the ID of the editor who last_modified that document
##
sub editor_last_modified_id
{
    my $self = shift;
    return $self->{Editor_Last_Modified_ID};
}


##
# $self->set_editor_last_modified_id ($editor_last_modified_id);
# --------------------------------------------------------------
#   $editor_last_modified_id - The ID of the editor who last_modified the document
#
#   Sets the ID of the editor who last_modified the document
##
sub set_editor_last_modified_id
{
    my $self = shift;
    $self->{Editor_Last_Modified_ID} = shift;
}


##
# $self->editor_last_modified;
# ----------------------------
#   Returns the editor who last_modified that document
##
sub editor_last_modified
{
    my $self = shift;
    my $editor_table = flo::Standard::table ('Editor');
    return $editor_table->get ($self->editor_last_modified_id);
}


##
# $self->set_editor_last_modified ($editor);
# ------------------------------------------
#   $editor - The object to extract the editor ID from
#
#   Sets the Editor_Last_modified_ID from the $editor object
##
sub set_editor_last_modified
{
    my $self = shift;
    my $editor = shift;
    $self->set_editor_last_modified_id ($editor->id);
}


sub validate_editor_last_modified_id
{
    my $self = shift;
    my $editor_created = $self->editor_created;
    unless (defined $editor_created)
    {
	new MKDoc::Ouch 'document/editor_last_modified';
	return 0;
    }
    
    return 1;
}


##
# $self->cache_control;
# ---------------------
#   Gets the ID of the editor who last_modified that document
##
sub cache_control
{
    my $self = shift;
    return $self->{Cache_Control};
}


##
# $self->set_cache_control ($cache_control);
# ------------------------------------------
#   $cache_control - The ID of the editor who last_modified the document
#
#   Sets the ID of the editor who last_modified the document
##
sub set_cache_control
{
    my $self = shift;
    $self->{Cache_Control} = shift;
}


##
# $self->validate_cache_control ($new_cache_control, $error);
# -----------------------------------------------------------
#   Does nothing if $new_cache_control is valid.
#   Otherwise, sets a flag into the MKDoc::Error
#   $error object and returns it.
##
sub validate_cache_control
{
    my $self = shift;
    my $cache_control = $self->cache_control();
    
    if (not defined $cache_control or $cache_control eq '')
    {
	new MKDoc::Ouch 'document/cache_control/empty';
	return 0;
    }
    
    unless ($cache_control =~ /^\d+$/)
    {
	new MKDoc::Ouch 'document/cache_control/not_integer';
	return 0;
    }
    
    return 1;
}


##
# $class->available_cache_controls;
# ---------------------------------
#   Returns a list of cache control values which can be used.
##
sub available_cache_controls
{
    my $class = shift;
    my @res = qw /0 5 10 15 30 60 120/;
    return (wantarray) ? @res : \@res;
}




##
# $self->keywords_items;
# ----------------------
#   Return the list of keywords as an array
##
sub keywords_items
{
    my $self = shift;
    my @subject = map { (/^\s*$/) ? () : $_ } split /,/, $self->keywords;
    return (wantarray) ? @subject : \@subject;
}


##
# $self->uri;
# -----------
#   Returns the uri of this document.
##
sub uri
{
    my $self = shift;
    my $plugin = shift;
    my $cgix = flo::Standard::cgi->new;
    $cgix->path_info ($self->path);
    for ($cgix->param) { $cgix->delete ($_) }
    return $cgix->self_url;
}


##
# $self->date_created;
# --------------------
#   Gets the date of creation associated with that document,
#   in MySQL format
##
sub date_created
{
    my $self = shift;
    return $self->{Date_Created};
}


##
# $self->set_date_created ($new_date);
# ------------------------------------
#   $new_date - date to set
##
sub set_date_created
{
    my $self = shift;
    $self->{Date_Created} = shift;
}


##
# $self->date_created_w3c;
# ------------------------
#   Returns the date of creation associated with that document,
#   in W3C format
##
sub date_created_w3c
{
    my $self = shift;
    my $date_created = ($self->date_created);
    
    # this function is completely memoizable, let's do it
    $DC_CREATED_W3C{$date_created} ||= do {
	my ($year, $mon, $mday, $hour, $min, $sec) = split /\D/, $date_created;
	$year -= 1900;
	$mon--;
	
	my $time = timelocal ($sec, $min, $hour, $mday, $mon, $year);
	
	my @time = gmtime ($time);
	$time[5] += 1900;
	$time[4]++;
	
	$time[4] = 0 . $time[4] if (length ($time[4]) < 2);
	$time[3] = 0 . $time[3] if (length ($time[3]) < 2);
	$time[2] = 0 . $time[2] if (length ($time[2]) < 2);
	$time[1] = 0 . $time[1] if (length ($time[1]) < 2);
	$time[0] = 0 . $time[0] if (length ($time[0]) < 2);
    
	$time[5] . '-' . $time[4] . '-' . $time[3] . 'T' . $time[2] . ':' . $time[1] . ':' . $time[0] . 'Z';
    };
    
    return $DC_CREATED_W3C{$date_created};
}


sub validate_date_created
{
    my $self = shift;
    my $date = $self->date_created;
    unless (defined $date)
    {
	new MKDoc::Ouch 'document/date_created/undefined';
	return 0;
    }
    
    unless ($date =~ /^\d\d\d\d-\d\d-\d\d \d\d:\d\d:\d\d$/)
    {
	new MKDoc::Ouch 'document/date_created/malformed';
	return 0;
    }
    
    return 1;
}


##
# $self->date_last_modified;
# --------------------------
#   Sets / Gets the last modification date associated with that document,
#   in MySQL format
##
sub date_last_modified
{
    my $self = shift;
    return $self->{Date_Last_Modified};
}


##
# $self->set_date_last_modified ($new_date);
# ------------------------------------------
#   $new_date - date to set
##
sub set_date_last_modified
{
    my $self = shift;
    $self->{Date_Last_Modified} = shift;
}


##
# $self->date_last_modified_w3c;
# ------------------------------
#   Sets / Gets the last modification date associated with that document,
#   in MySQL format
##
sub date_last_modified_w3c
{
    my $self = shift;
    my $date_last_modified = ($self->date_last_modified);
    my ($year, $mon, $mday, $hour, $min, $sec) = split /\D/, $date_last_modified;
    $year -= 1900;
    $mon--;
    
    my $time = timelocal ($sec, $min, $hour, $mday, $mon, $year);
    
    my @time = gmtime ($time);
    $time[5] += 1900;
    $time[4]++;
    
    $time[4] = 0 . $time[4] if (length ($time[4]) < 2);
    $time[3] = 0 . $time[3] if (length ($time[3]) < 2);
    $time[2] = 0 . $time[2] if (length ($time[2]) < 2);
    $time[1] = 0 . $time[1] if (length ($time[1]) < 2);
    $time[0] = 0 . $time[0] if (length ($time[0]) < 2);
    
    return $time[5] . '-' . $time[4] . '-' . $time[3] . 'T' . $time[2] . ':' . $time[1] . ':' . $time[0] . 'Z';
}


sub validate_date_last_modified
{
    my $self = shift;
    my $date = $self->date_last_modified;
    unless (defined $date)
    {
	new MKDoc::Ouch 'document/date_last_modified/undefined';
	return 0;
    }
    
    unless ($date =~ /^\d\d\d\d-\d\d-\d\d \d\d:\d\d:\d\d$/)
    {
	new MKDoc::Ouch 'document/date_last_modified/malformed';
	return 0;
    }
    
    return 1;
}


##
# $self->is_only_child;
# ---------------------
# Returns TRUE if $self is an only child, FALSE otherwise
##
sub is_only_child
{
    my $self = shift;
    return 1 unless (defined $self->parent_id); 
    my @siblings = $self->parent->children (@_); 
    return @siblings == 1;
}


##
# $self->is_only_child_showable;
# ------------------------------
# Returns TRUE if $self is an only showable child, FALSE otherwise
##
sub is_only_child_showable
{
    my $self = shift;
    return 1 unless (defined $self->parent_id); 
    my @siblings = $self->parent->children_showable (@_);
    return @siblings == 1;
}


##
# $self->is_only_homonym;
# -----------------------
# Returns TRUE if $self is an only homonym, FALSE otherwise
##
sub is_only_homonym
{
    my $self = shift;
    return 1 unless (defined $self->parent_id); 
    my @list = $self->homonyms (@_);
    return @list == 1;
}


##
# $self->is_only_homonym_showable;
# --------------------------------
# Returns TRUE if $self is an only homonym, FALSE otherwise
##
sub is_only_homonym_showable
{
    my $self = shift;
    return 1 unless (defined $self->parent_id); 
    my @list = $self->homonyms_showable (@_);
    return @list == 1;
}


##
# $self->is_only_cousin_homonym;
# -----------------------
# Returns TRUE if $self is an only cousin_homonym, FALSE otherwise
##
sub is_only_cousin_homonym
{
    my $self = shift;
    return 1 unless (defined $self->parent_id); 
    my @list = $self->cousin_homonyms (@_);
    return @list == 1;
}


##
# $self->is_only_cousin_homonym_showable;
# ---------------------------------------
# Returns TRUE if $self is an only cousin_homonym, FALSE otherwise
##
sub is_only_cousin_homonym_showable
{
    my $self = shift;
    return 1 unless (defined $self->parent_id); 
    my @list = $self->cousin_homonyms_showable (@_);
    return @list == 1;
}


##
# $self->cousin_homonym;
# ----------------------
# Returns the first cousin homonym returned by cousin_homonyms(),
# undef if none returned.
##
sub cousin_homonym
{
    my $self = shift;
    my @list = $self->cousin_homonyms (@_);
    return $list[0];
}


##
# $self->cousin_homonym_showable;
# -------------------------------
# Returns the first cousin homonym returned by cousin_homonyms(),
# undef if none returned.
##
sub cousin_homonym_showable
{
    my $self = shift;
    my @list = $self->cousin_homonyms_showable (@_);
    return $list[0];
}

    
##
# $self->homonym;
# ---------------
# Returns the first cousin homonym returned by homonyms(),
# undef if none returned.
##
sub homonym
{
    my $self = shift;
    my @list = $self->homonyms (@_);
    return $list[0];
}


##
# $self->homonym_showable;
# ------------------------
# Returns the first cousin homonym returned by homonyms(),
# undef if none returned.
##
sub homonym_showable
{
    my $self = shift;
    my @list = $self->homonyms_showable (@_);
    return $list[0];
}
   

sub language { return shift()->lang (@_) }


##
# $self->align;
# -------------
#   Returns the alignment of this object depending on the language,
#   i.e. 'left' or 'right'
##
sub align
{
    my $self   = shift;
    return ($self->direction eq 'ltr') ? 'left' : 'right';
}


##
# $self->align_opposite;
# ----------------------
#   Returns 'left' when align() returns 'right', returns 'right' otherwise
##
sub align_opposite
{
    my $self = shift;
    return ($self->direction eq 'ltr') ? 'right' : 'left';
}


##
# $self->direction;
# -----------------
#   Returns the direction depending on the language, i.e. 'left' or 'right'
##
sub direction
{
    my $self = shift;
    my $hash = MKDoc::Config->parsefile_hashref (MKDoc::Config->LANGUAGE_LIST_RTL);
    my $lang = $self->language();
    return exists $hash->{$lang} ? 'rtl' : 'ltr';
}


##
# $self->audiences;
# -----------------
#   Returns the audiences objects which are associated with this document
##
sub audiences
{
    my $self = shift;
    my $document_audience_t = flo::Standard::table ('Document_Audience');
    my $audience_t = flo::Standard::table ('Audience');
    my @res = $document_audience_t->search ( Document_ID => $self->id() )->fetch_all;
    my @res2 = map { my $var = $audience_t->get ($_->audience_id); ($var) ? $var : () } @res;
    return wantarray ? @res2 : \@res2;
}


##
# $self->matches_preference ($preference);
# ----------------------------------------
#   Returns TRUE if this document matches preference $preference
#   (disregarding its flag), FALSE otherwise.
##
sub matches_preference
{
    my $self = shift;
    my $pref = shift;
    if ($pref->name eq 'dc.audience')
    {
	foreach ($self->audiences)
	{
	    return 1 if ($pref->name eq $_->name);
	}
	return;
    }
    
    if ($pref->name eq 'dc.language')
    {
	return $pref->value eq $self->lang();
    }
    
    confess ($pref->name . " is not supported by " . ref $self . "::matches_preference");
}


##
# $self->rights;
# --------------
#   Gets the optional rights attribute for the document
##
sub rights
{
    my $self = shift;
    return $self->{Rights};
}


##
# $self->set_rights ($rights);
# ----------------------------
#   $rights - The new optional rights attribute for the document
#
#   Sets the rights attribute for the document
##
sub set_rights
{
    my $self = shift;
    $self->{Rights} = shift;
}


##
# $self->publisher;
# -----------------
#   Gets the optional publisher attribute for the document
##
sub publisher
{
    my $self = shift;
    return $self->{Publisher};
}


##
# $self->set_publisher ($publisher);
# ----------------------------------
#   $publisher - The new optional publisher attribute for the document
#
#   Sets the publisher attribute for the document
##
sub set_publisher
{
    my $self = shift;
    $self->{Publisher} = shift;
}


##
# $self->relation;
# ----------------
#   Gets the optional relation attribute for the document
##
sub relation
{
    my $self = shift;
    return $self->{Relation};
}


##
# $self->set_relation ($relation);
# --------------------------------
#   $relation - The new optional relation attribute for the document
#
#   Sets the relation attribute for the document
##
sub set_relation
{
    my $self = shift;
    $self->{Relation} = shift;
}


##
# $self->Can_Admin;
# -----------------
# Returns TRUE if the user can administrate that document,
# FALSE otherwise.
##
sub Can_Admin
{
    my $self = shift;
    my $user = flo::Standard::current_user() || return;
    foreach my $base_document ($user->base_documents())
    {
	return 1 if ($base_document->is_ancestor_of ($self));
    }
    
    return;
}


sub now_iso
{
    my @time  = localtime (time());
    
    my $year  = $time[5];
    $year    += 1900;
    
    my $month = $time[4];
    $month   += 1;
    $month    = "0$month" unless (length ($month) == 2);
    
    my $day   = $time[3];
    $day      = "0$day" unless (length ($day) == 2);
    
    my $hour  = $time[2];
    $hour     = "0$hour" unless (length ($hour) == 2);
    
    my $min   = $time[1];
    $min      = "0$min" unless (length ($min) == 2);
    
    my $sec   = $time[0];
    $sec      = "0$sec" unless (length ($sec) == 2);
    
    return "$year-$month-$day $hour:$min:$sec";
}


sub now_day_iso
{
    my $self = shift;
    my ($day, $time) = split /\s/, $self->now_iso();
    return $day;
}


sub now_time_iso
{
    my $self = shift;
    my ($day, $time) = split /\s/, $self->now_iso();
    return $time;
}

##
# $self->on_save;
#
# Handler which gets called just before saving the document.  Needed
# to clean up index tables, like Document_TimeRange.
##
sub on_save {
    my $self = shift;
    my $id   = $self->id();

    # new documents won't need pre-save cleanup
    return unless $id;

    my $dbh = lib::sql::DBH->get();
    $dbh->do('DELETE FROM Document_TimeRange WHERE Document_ID = ?',
             undef, $id);
}

##
# $self->on_delete;
#
# Handler which gets called just before deleting the document.  Needed
# to clean up index tables, like Document_TimeRange.
##
sub on_delete {
    my $self = shift;
    $self->on_save();
}

1;
