# -------------------------------------------------------------------------------------
# flo::Record::Editor
# -------------------------------------------------------------------------------------
# Author : Jean-Michel Hiver.
# Copyright : (c) MKDoc Holdings Ltd, 2003.
#
# This file is part of MKDoc. 
# 
# MKDoc is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# MKDoc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with MKDoc; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# -------------------------------------------------------------------------------------
package flo::Record::Editor;
use flo::Standard;
use flo::User::Preferences;
use MKDoc::Ouch;
use Digest::MD5;
use strict;
use 5.008_000;
use utf8;


$Petal::Hash::MODIFIERS->{'user_root:'} = sub {
    my $hash = shift;
    my $editor_t = flo::Standard::table ('Editor');
    return $editor_t->get (1);
};


##
# $class->new (%args);
# --------------------
# Instanciates a new flo::Record::Editor object.
##
sub new
{
    my $class = shift;
    $class    = ref $class || $class;
    my $self  = bless { @_ }, $class;
    return $self if (defined $self->{ID});
    
    if ($self->validate())
    {
	$self->save();
        $self->preferences->set_default_language;
	return $self;
    }
    
    return;
}


sub load
{
    my $class = shift || return;
    my $editor_t = flo::Standard::table ('Editor');
    return $editor_t->get (@_);
}


##
# $self->id;
# ----------
#   Returns the ID for this flo::Record::Editor.
##
sub id
{
    my $self = shift || return;
    return $self->{ID};
}


##
# $self->set_id ($id);
# --------------------
#   $id - New ID for this flo::Record::Editor
#
#   Sets the ID for this flo::Record::Editor object.
##
sub set_id
{
    my $self = shift || return;
    $self->{ID} = shift;
    return $self;
}


##
# $self->group;
# -------------
#   Returns the type of user this is by checking if
#   this user has a root document.  If so then 'editor'
#   is returned, otherwise 'user' is returned.
#
# Note:
#   This is a hack and should be replaced by a
#   'User_Type' field being added to the database.
##
sub group
{
    my $self = shift || return;

    # does the user have a base document?
    my $bdoc_table = flo::Standard::table ('Base_Document');
    if ($bdoc_table->get( Editor_ID => $self->id ))
    {
	return 'editor';
    }
    else
    {
	return 'user';
    }
}


sub is_admin
{
    my $self = shift;
    return $self->id() == 1;
}


sub is_editor
{
    my $self = shift;
    return $self->is_group ('editor');
}


sub is_user
{
    my $self = shift;
    return $self->is_group ('user');
}


sub is_group
{
    my $self = shift;
    my $group = shift;
    return $self->group() eq $group;
}


##
# $self->login;
# -------------
#   Gets the login of this user.
##
sub login
{
    my $self = shift || return;
    return $self->{Login};
}


##
# $self->set_login ($login);
# --------------------------
#   $login - Login to set.
#
#   Sets the login of this user.
##
sub set_login
{
    my $self = shift || return;
    $self->{Login} = shift;
    return $self;
}


##
# $self->password;
# ----------------
#   Gets the password of this user.
##
sub password
{
    my $self = shift || return;
    return $self->{Password};
}


##
# $self->set_password ($password);
# --------------------------------
#   $password - Password to set.
#
#   Sets the password of this user.
##
sub set_password
{
    my $self = shift || return;
    $self->{Password} = shift;
    return $self;
}


##
# $self->email;
# -------------
#   Gets this user's email.
##
sub email
{
    my $self = shift || return;
    return $self->{Email};
}


##
# $self->set_email ($email);
# --------------------------
#   $email - Email to set
#
#   Sets the email of this user
##
sub set_email
{
    my $self = shift || return;
    $self->{Email} = shift;
    return $self;
}


##
# $self->first_name;
# ------------------
#   Gets the user's first name
##
sub first_name
{
    my $self = shift || return;
    return $self->{First_Name};
}


##
# $self->set_first_name ($name);
# ------------------------------
#   $name - the new first name for this user.
#
#   Sets the user's first name to $name
##
sub set_first_name
{
    my $self = shift || return;
    $self->{First_Name} = shift;
    return $self;
}


sub last_name
{
    my $self = shift;
    return $self->family_name;
}


##
# $self->family_name;
# -------------------
#   Gets the user's family name
##
sub family_name
{
    my $self = shift || return;
    return $self->{Family_Name};
}


##
# $self->set_family_name ($name);
# -------------------------------
#   $name - the new family name for this user.
#
#   Sets the user's family name to $name
##
sub set_family_name
{
    my $self = shift || return;
    $self->{Family_Name} = shift;
    return $self;
}


##
# $self->real_name;
# -----------------
#   Gets the user real name
##
sub real_name
{
    my $self = shift || return;
    $self->first_name && $self->family_name and return $self->first_name . " " . $self->family_name;
    $self->first_name  and return $self->first_name;
    $self->family_name and return $self->family_name;
    return 'nobody';
}


##
# $self->set_real_name ($name);
# -----------------------------
#   $name - the new real name for this user.
#
#   Sets the real user name to $name
##
sub set_real_name
{
    my $self = shift || return;
    my $name = shift;
    my ($first_name, $last_name) = split /\s+/, $name, 2;
    $first_name ||= '';
    $last_name  ||= '';
    $self->set_first_name($first_name);
    $self->set_family_name($last_name);
    return $self;
}


##
# $self->enabled;
# ---------------
#   Returns TRUE if this user is enabled, FALSE otherwise.
##
sub enabled
{
    my $self = shift || return;
    return $self->{Disabled} != 1;
}


##
# $self->set_enabled ($enabled);
# ------------------------------
#   $enabled - a boolean value which tells wether this user is enabled or not.
#
#   Returns TRUE if this user is enabled, FALSE otherwise.
##
sub set_enabled
{
    my $self = shift || return;
    my $enabled = shift;
    $self->{Disabled} = ($enabled) ? 0 : 1;
    return $self;
}


##
# $self->is_enabled;
# ------------------
#   alias for enabled().
##
sub is_enabled
{
    my $self = shift || return;
    return $self->enabled;
}


##
# $self->is_confirmed;
# --------------------
##
sub is_confirmed
{
    my $self = shift || return;
    return $self->{Disabled} == 0;
}


sub save
{
    my $self = shift;
    my $editor_t = flo::Standard::table ('Editor');
    
    ($self->id()) ?
	$editor_t->modify ($self) :
	$self->set_id ($editor_t->insert ($self));
}


sub delete
{
    my $self = shift;
    
    my $self_id = $self->id();
    ($self_id == 1) and do {
	new MKDoc::Ouch 'editor/delete/admin';
	return 0;
    };
    
    my $base_document_t = flo::Standard::table ('Base_Document');
    $base_document_t->delete (Editor_ID => $self_id);
    
    my $contributor_t   = flo::Standard::table ('Contributor');
    $contributor_t->delete (Editor_ID => $self_id);
    
    my $document_t      = flo::Standard::table ('Document');
    $document_t->update ( { Editor_Created_ID => 1 }, { Editor_Created_ID => $self_id } );
    $document_t->update ( { Editor_Last_Modified_ID => 1 }, { Editor_Last_Modified_ID => $self_id } );
    
    my $preference_t    = flo::Standard::table ('Preference');
    $preference_t->delete (Editor_ID => $self_id);
    
    my $preference_a_t  = flo::Standard::table ('Preference_Audience');
    $preference_a_t->delete (Editor_ID => $self_id);
    
    my $preference_l_t  = flo::Standard::table ('Preference_Language');
    $preference_l_t->delete (Editor_ID => $self_id);
    
    my $editor_t        = flo::Standard::table ('Editor');
    $editor_t->delete (ID => $self_id);
    
    $self->set_id (undef);
    return 1;
}


sub preferences
{
    my $self = shift;
    return new flo::User::Preferences->set_user($self);
}


sub validate
{
    my $self = shift;
    return $self->validate_login()      &
           $self->validate_password()   &
	   $self->validate_email()      &
	   $self->validate_first_name() &
	   $self->validate_family_name();
}


sub validate_login
{
    my $self = shift;
    my $login = $self->login();

    (defined $login and $login ne '') or do {
	new MKDoc::Ouch 'user/login/empty';
	return 0;
    };
    
    (length ($login) < 15) or do {
	new MKDoc::Ouch 'user/login/too_long';
	return 0;
    };
    
   ($login =~ /[^A-Za-z0-9]/) and do {
	new MKDoc::Ouch 'user/login/malformed';
	return 0;
    };
    
    return 1 if ($self->id());
    
    my $editor_t = flo::Standard::table ('Editor');
    $editor_t->get (Login => $login) || return 1;
    
    new MKDoc::Ouch 'user/login/exists';
    return 0;
}


sub validate_password
{
    my $self = shift;
    my $pass = $self->password();
    
    (defined $pass and $pass ne '') or do {
	new MKDoc::Ouch 'user/password/empty';
	return 0;
    };
    
    (length $pass > 4) or do {
	new MKDoc::Ouch 'user/password/too_short';
	return 0;
    };
    
    (length $pass < 15) or do {
	new MKDoc::Ouch 'user/password/too_long';
	return 0;
    };
    
    return 1;
}


sub validate_email
{
    my $self = shift;
    my $mail = $self->email();
    
    (defined $mail and $mail ne '') or do {
	new MKDoc::Ouch 'user/email/empty';
	return 0;
    };
    
    (length $mail < 255) or do {
	new MKDoc::Ouch 'user/email/too_long';
	return 0;
    };
    
    return 1;
}


sub validate_first_name
{
    my $self = shift;
    my $name = $self->first_name();
    
    (defined $name and $name ne '') or do {
	new MKDoc::Ouch 'user/first_name/empty';
	return 0;
    };
    
    (length $name < 50) or do {
	new MKDoc::Ouch 'user/first_name/too_long';
	return 0;
    };
    
    return 1;
}


sub validate_family_name
{
    my $self = shift;
    my $name = $self->family_name();
    
    (defined $name and $name ne '') or do {
	new MKDoc::Ouch 'user/family_name/empty';
	return 0;
    };
    
    (length $name < 50) or do {
	new MKDoc::Ouch 'user/family_name/too_long';
	return 0;
    };
    
    return 1;
}


sub base_documents
{
    my $self = shift;
    $self->{base_documents} ||= do {
	my $base_document_t = flo::Standard::table ('Base_Document');
	my $document_t      = flo::Standard::table ('Document');
    
	my @res = map { $document_t->get ($_->{Document_ID}) }
	          $base_document_t->search ( Editor_ID => $self->id() )
		  ->fetch_all();
	
	\@res;
    };
    
    return wantarray ? @{$self->{base_documents}} : $self->{base_documents};
}


sub signature
{
    my $self     = shift;
    my $login    = $self->login;
    my $password = $self->password;
    my $disabled = $self->{Disabled};
    my $string   = "$login:$password:$disabled";
    return substr Digest::MD5::md5_hex ($string), 0, 16;
}


sub get_login
{
    my $class    = shift;
    my $editor_t = flo::Standard::table ('Editor');
    return $editor_t->get (Login => shift);
}


sub auth
{
    my $self = shift;
    my $uri  = shift;
    
    my $login    = $self->login();
    my $password = $self->password();
    
    my $auth = "$login:$password\@";
    $uri     =~ s/:\/\//:\/\/$auth/;
    return $uri;
}


1;
