# -----------------------------------------------------------------------------
# flo::RedirectManager
# -----------------------------------------------------------------------------
#       Author : Jean-Michel Hiver
#    Copyright : Copyright (c) 2002 MKDoc Holdings Ltd.
#
# This file is part of MKDoc. 
# 
# MKDoc is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# MKDoc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with MKDoc; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#
#   Description:
#
#       Module that handles redirect related operations
#
# -----------------------------------------------------------------------------
package flo::RedirectManager;
use flo::Standard;
use warnings;
use strict;
use Carp;


##
# $class->document_added ($path);
# -------------------------------
# Checks if there is a redirect from $path to another location.
# If there is none, does nothing.
# If not, set a redirect for each of the child of that document.
#
# For example:
#
#     /foo/
#     /foo/bar/
#     /foo/baz/
#     /foo/baz/buz/
#
# /foo/ is moved to /hello/, the structure becomes:
#
#     /hello/
#     /hello/bar/
#     /hello/baz/
#     /hello/baz/buz/
#
# And the following redirect is inserted:
#
#     [+] /foo/     => /hello/
#
# Now let's say a document is added at /foo/, it is necessary
# to do the following:
#
#     [-] /foo/     => /hello/
#     [+] /foo/bar/ => /hello/bar/
#     [+] /foo/baz/ => /hello/baz/
##
sub document_added
{
    my $class = shift;
    my $path  = shift;
    my $document_t = flo::Standard::table ('Document');
    my $redirect_t = flo::Standard::table ('Redirect');
    
    my $redirect = $redirect_t->get ( New_Path => $path );
    return unless (defined $redirect);
    
    my $redirect_from = $redirect->old_path();
    my $redirect_to   = $redirect->new_path();
    my $document = $document_t->get ( Full_Path => $redirect_to );
    
    $redirect_t->delete ( New_Path => $path );
    (defined $document) and do {
	foreach my $child ($document->children())
	{
	    my $new_redirect_to = $child->path();
	    my $new_redirect_from = $new_redirect_to;
	    $new_redirect_from =~ s/\Q$redirect_to\E/$redirect_from/;
	    $class->insert_redirect (
		Old_Path => $new_redirect_from,
		New_Path => $new_redirect_to
	       );
	}
    };
}


##
# $class->insert_redirect (Old_Path => 'foo', New_Path => 'bar' );
# ----------------------------------------------------------------
# Inserts a redirect in the database.
##
sub insert_redirect
{
    my $class = shift;
    my $redirect = { @_ };
    my $redirect_t = flo::Standard::table ('Redirect');
    
    return if ($redirect_t->get ($redirect));
    $redirect_t->insert ($redirect);
}


##
# $class->document_moved ($document, $new_path);
# ----------------------------------------------
# $document is about to be moved (or, for the matter,
# deleted but that doesn't make any difference) from
# $document->path() to $new_path.
#
# Any previous redirect that was leading to $document->path()
# needs to be redirected to $path. Plus a redirect from
# $document->path() to path() needs to be inserted.
#
# For example, let's consider the following redirect table:
#
#     /foo/bar/ => /hello/bar/
#     /foo/baz/ => /hello/baz/
#
# Let's say that /hello/bar/ is redirected to /hello/baz/baz/,
# this needs to become:
#
#     [-] /foo/bar/   => /hello/bar/
#     [+] /foo/bar/   => /hello/baz/
#     [+] /hello/bar/ => /hello/baz/
#     [+] /foo/bar/   => /hello/baz/
##
sub document_moved
{
    my $class = shift;
    my $document = shift;
    my $path = shift;
    $path .= $document->name() . '/';
    
    $class->_register_redirect ( $document->path() => $path );
}


##
# $class->document_renamed ($document, $new_name);
# ------------------------------------------------
# This method should be invoked when a document is about to
# be renamed.
##
sub document_renamed
{
    my $class = shift;
    my $document = shift;
    my $new_name = shift;
    
    return if ($new_name eq $document->name());
    
    my $path = $document->path();
    $path =~ s/[a-z0-9-]+\/$/$new_name\//;    
    $class->_register_redirect ( $document->path => $path );
}


##
# $class->document_deleted ($document, $new_path);
# ------------------------------------------------
# ATM an alias for $class->document_moved ($document, $new_path);
##
sub document_deleted
{
    my $class = shift;
    my $document = shift;
    my $replacement_path = shift;
    $class->_register_redirect ( $document->path() => $replacement_path );
}


##
# $class->_register_redirect ($old_path, $new_path);
# --------------------------------------------------
# Registers the fact that URIs starting with $old_path
# now need start with $new_path and perform a redirect.
##
sub _register_redirect
{
    my $class = shift;
    my $from  = shift;
    my $to    = shift;
    
    my $redirect_t = flo::Standard::table ('Redirect');
    
    # step 1.
    #    remove any redirect which points from $to to
    #    something else.
    $redirect_t->delete ( Old_Path => $to );
    $redirect_t->delete ( Old_Path => $from );
    
    # step 2.
    #    update all the redirects which point to the old location
    #    to the new location.
    $redirect_t->update (
	{ New_Path => $to   },
	{ New_Path => $from },
       );
    
    # step 3.
    #    add a redirect from the old location to the new location.
    $class->insert_redirect (
	Old_Path => $from,
	New_Path => $to
       );
}


##
# $class->translate_path ($path);
# -------------------------------
# This method is usually called because $path could not
# be found and a redirect is needed. Returns either a new
# $path or undef.
#
# For example if the redirect table is the following:
#
#     /hi/ => /hello/
#
#     $class->translate ('/hi/man/'); # returns '/hello/man/'
#     $class->translate ('/yo/');     # returns 'undef'
##
sub translate_path
{
    my $class = shift;
    my $path = shift;
    my $redirect_t = flo::Standard::table ('Redirect');
    my @redirects = sort {
	length ($b->{Old_Path}) <=>
	length ($a->{Old_Path})
    } $redirect_t->search()->fetch_all();
    
    for my $redirect (@redirects)
    {
	my $from = quotemeta ($redirect->{Old_Path});
	my $to   = $redirect->{New_Path};
	return $path if ($path =~ s/^$from/$to/);
    }
    return;
}


1;
