# -------------------------------------------------------------------------------------
# flo::editor::Comment
# -------------------------------------------------------------------------------------
# Author : Bruno Postle
# Copyright : (c) MKDoc Ltd, 2005
#
# This file is part of MKDoc. 
# 
# MKDoc is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# MKDoc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with MKDoc; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# -------------------------------------------------------------------------------------
package flo::editor::Comment;

=head1 NAME

flo::editor::Comment - an editor for comments

=head1 DESCRIPTION

The module provides an editor for Comment components.  A Comment component has
information about the time and the poster as well as the comment text itself.

By default, only a valid editor for a particular document can add, delete or
modify Comment components, enable the L<flo::plugin::Comment> plugin to allow
any user to append Comment components.

=head1 INTERFACE

=over 4

This module is a simple subclass of the L<flo::editor::Text> module.

=cut

use strict;

use flo::editor::Text;
use base qw /flo::editor::Text/;

=item C<< $self->as_xhtml() >>

Renders the comment via petal.  Essentially the same as the Text component
except that external links are given an 'untrusted' rel="nofollow" attribute
and the comment, date_posted, poster, email & login fields are available for
the template designer.

=cut

sub as_xhtml
{
    my $self = shift;
    my $data = $self->{'data'};
    local $MKDoc::Text::Structured::Inline::NoFollow = 1;
    $data = MKDoc::Text::Structured::process ($data);
    
    my @newl = $self->links_for_tagger();
    $self->{'comment'} = MKDoc::XML::Tagger::Preserve->process_data (['a'], $data, @newl);

    my $type = $self->type();
    my $template = new Petal (
        file   => "component/$type",
        lang   => $self->language(),
        input  => 'XML',
        output => 'XHTML'
       );

    my $res = $template->process (self => $self);
    Encode::_utf8_on ($res);

    return $res;
}

sub _initialize
{
    my $self = shift;
    my $cgi = $self->{cgi};
    my $param_name = $self->{param_name};

    if (defined $cgi)
    {
        $self->{date_posted} = $cgi->param ($param_name . "_date_posted") || flo::Standard::current_document()->now_iso();
        $self->{poster}      = $cgi->param ($param_name . "_poster")      || flo::Standard::current_user()->real_name();
        $self->{email}       = $cgi->param ($param_name . "_email")       || flo::Standard::current_user()->email();
        $self->{login}       = $cgi->param ($param_name . "_login")       || flo::Standard::current_user()->login();
    }
}

1;
