# -------------------------------------------------------------------------------------
# flo::editor::File
# -------------------------------------------------------------------------------------
# Author : Jean-Michel Hiver.
# Copyright : (c) MKDoc Holdings Ltd, 2003
#
# This file is part of MKDoc. 
# 
# MKDoc is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# MKDoc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with MKDoc; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# -------------------------------------------------------------------------------------
package flo::editor::File;
use MKDoc::Util::Text2HTML;
use MKDoc::Config;
use flo::Standard;
use flo::Editor;
use strict;
use 5.008_000;
use utf8;

use base qw /flo::Component
	     flo::editor::Mixin::compute_name
	     flo::editor::Mixin::normalize_name
	     flo::editor::Mixin::hr_name
	     flo::editor::Mixin::upload_file/;


sub validate
{
    my $self = shift;
    
    # set up the callback for errors
    local $MKDoc::Ouch::CALLBACK;
    $MKDoc::Ouch::CALLBACK = sub { $self->add_error (@_) };
    
    return $self->validate_title() &
           $self->validate_file();
}


sub validate_file
{
    my $self = shift;
    defined $self->{file} and $self->{file} ne '' or do {
	new MKDoc::Ouch 'component/file/no_file';
	return 0;
    };
    
    return 1;
}


sub validate_title
{
    my $self = shift;
    defined $self->{title} and $self->{title} !~ /^\s*$/ or do {
	new MKDoc::Ouch 'component/file/title_empty';
	return 0;
    };
    
    return 1;
}


sub preferred_extension
{
    my $self  = shift;
    my $file  = $self->{file} || return '';
    $file =~ s/^.*\///;
    my ($ext) = $file =~ /.*\.(.*)/;
    $ext = $1 if ($file =~ /\.([^.]+\.gz)$/);
    $ext ||= $self->type();
    return $ext;
}


sub extension
{
    my $self = shift;
    return $self->preferred_extension();
}


##
# $self->new;
# -----------
# Returns a new flo::editor::File object based on CGI input
##
sub new
{
    my $class = shift;
    $class    = ref $class || $class;
    my $self  = bless { @_ }, $class;
    
    my $cgi = $self->{cgi};
    my $param_name = $self->{param_name};
    
    # if CGI is defined, then we probably want to do some stuff
    if (defined $cgi)
    {
	# update title field
	if (defined $cgi->param ($param_name . "_title"))
	{
	    my $title = $cgi->param ($param_name . "_title");
	    $self->{title} = $title;
	}
	else
	{
	    $self->{title} = '';
	}
	    
	# if the upload field is not empty, then the user
	# just uploaded a new file
	if (defined $cgi->param ($param_name . "_upload"))
	{
	    $self->{title} = '';
	    my $new_file = $cgi->param ($param_name);
	    $new_file = "$new_file";
	    if ($new_file)
	    {
		my $new_file_no_slash = $new_file;
		$new_file_no_slash =~ s/^.*\///g;
		$new_file_no_slash =~ s/^.*\\//g;

		my $new_file_friendly = $self->normalize_name ($new_file_no_slash);
		my $dir = $self->compute_name;
		my $full_dir = MKDoc::Config->FILE_DISK_PATH . "/" . $dir;
		
		mkdir $full_dir;
		$class->upload_file ('file_disk_path', $dir, $new_file_friendly, $cgi->param ($param_name))
		    or return $self;
                
		my $self_title = $self->{title};
		$self_title ||= do {
                    my $res = $class->hr_name ($self->normalize_name_nodot ($new_file_no_slash));
		    $res;
		};
	
		my $self_file = "$dir/$new_file_friendly";
		
		$self->{file} = $self_file;
		$self->{title} = $self_title;
	    }
	}
	else
	{
	    my $file = $cgi->param ($param_name . "_file");
	    $self->{file} = $file;
	}
    }
    
    return $self;
}


sub parse_xml
{
    my $self = shift;
    my $data = shift || return;
   
    if ($data =~ /^\<perl/ or $data =~ /^\<hash/) { return $self->SUPER::parse_xml ($data, @_) }
    else
    {
        # backwards compatibility horribleness
        my ($file, $title) = $data =~ /\<a href=\"(.*?)\"\>(.*)\<\/a\> \/\>/;
        my $q_file_http_path = quotemeta (MKDoc::Config->FILE_HTTP_PATH);
        $file =~ s/^$q_file_http_path//;
        $file =~ s/^\///o;
    
        $self->{file}  = $file  || '';
        $self->{title} = $title || '';
   }
}


##
# $self->internal_location();
# ---------------------------
# Returns the internal location associated with $self.
##
sub internal_location
{
    my $self = shift;
    return MKDoc::Config->FILE_HTTP_PATH . "/" . $self->{file};
}


1;
