# -------------------------------------------------------------------------------------
# flo::editor::Link
# -------------------------------------------------------------------------------------
# Author : Jean-Michel Hiver.
# Copyright : (c) MKDoc Holdings Ltd, 2003
#
# This file is part of MKDoc. 
# 
# MKDoc is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# MKDoc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with MKDoc; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# -------------------------------------------------------------------------------------
package flo::editor::Link;
use flo::Editor;
use flo::Standard;
use MKDoc::CGI;
use MKDoc::Util::Text2HTML;
use MKDoc::Util::LinkParser;
use strict;

use base qw /flo::Component
	     flo::editor::Mixin::compute_name
	     flo::editor::Mixin::normalize_name/;

# keep a global link parser around
my $link_parser = MKDoc::Util::LinkParser->new();

# placeholder for the URL entry field
my $EMPTY_URL = 'http://';

sub preferred_extension { 'link' }


sub link
{
    return shift;
}


sub _initialize
{
    my $self = shift;
    my $args = $self->cgi_args()      || return;

    $self->{title}       = $args->{'title'}       || '';
    $self->{description} = $args->{'description'} || '';

    if ($args->{url}) {
        $self->parse_url($args->{url});
    } else {
        $self->{url} = $EMPTY_URL;
    }
}


sub parse_url {
    my ($self, $url) = @_;
    $link_parser->parse($url);

    if ($link_parser->is_internal and $link_parser->is_valid) {
        # store relevent details for an internal link (invalid
        # links are handled by validate_url)
        $self->{internal_link} = $link_parser->freeze();

        # reflect the canonical form in the UI
        $self->{url}           = $link_parser->as_string;
    } else {
        # store just the url as entered for external links
        delete $self->{internal_link};
        $self->{url} = $url;
    }
}

# template rendering for the URL
sub render_url {
    my $self = shift;
    if ($self->{internal_link}) {
        $link_parser->thaw($self->{internal_link});
        return $link_parser->as_string;
    } else {
        return $self->{url};
    }
}

sub is_internal {
    return shift->{internal_link} ? 1 : 0;
}

sub validate
{
    my $self = shift;
    
    # set up the callback for errors
    local $MKDoc::Ouch::CALLBACK;
    $MKDoc::Ouch::CALLBACK = sub { $self->add_error (@_) };
    
    return $self->validate_url()   &
           $self->validate_title() &
	   $self->validate_description();
}


sub validate_url
{
    my $self = shift;
    $self->{url} and $self->{url} ne $EMPTY_URL or do {
	new MKDoc::Ouch 'component/link/url_empty';
	return 0;
    };

    $link_parser->parse($self->{url});
    if ($link_parser->is_internal and not $link_parser->is_valid) {
        new MKDoc::Ouch 'component/link/invalid_internal_url';
        return 0;
    }
    
    return 1;
}


sub validate_title
{
    my $self = shift;
    $self->{title} or do {
	new MKDoc::Ouch 'component/link/title_empty';
	return 0;
    };
    
    return 1;
}


sub validate_description
{
    my $self = shift;
    $self->{description} or do {
	new MKDoc::Ouch 'component/link/description_empty';
	return 0;
    };
    
    return 1;
}


sub full_url
{
    my $self = shift;
    my $url  = shift;
    return $url if ($url =~ /^(http|https|mailto|news|gopher):/);
    
    if ($url =~ /^\//)
    {
	return $ENV{SERVER_NAME} . $url;
    }
    else
    {
	return $ENV{SERVER_NAME} . flo::Standard::raw_path_info() . $url;
    }
}


sub url
{
    my $self = shift;
    $self->{url} = shift || return $self->{url};
}


sub url_clean
{
    my $self = shift;
    my $url = $self->url;
    $url =~ s/\&(?!amp;)/&amp;/g;
    return $url;
}


sub title
{
    my $self = shift;
    $self->{title} = shift || return $self->{title};
}


sub description
{
    my $self = shift;
    $self->{description} = shift || return $self->{description};
}


sub parse_xml
{
    my $self = shift;
    my $data = shift || return;
   
    $data =~ /^<a href/ and do {
        # backwards compatibility horribleness
        my ($url, $description, $title) = $data =~ /\<a href=\"(.*?)\" title=\"(.*?)\"\>(.*)\<\/a\>/;
        $self->{url}         = xml2text ($url)          || '';
        $self->{description} = xml2text ($description)  || '';
        $self->{title}       = xml2text ($title)        || '';
        return $self;
    };
    
    return $self->SUPER::parse_xml ($data, @_);
}


sub uri
{
    my $self = shift;
    my $url  = $self->url();
    return $url if ($url =~ /^[A-Za-z]+\:/);
    
    $url =~ /^\// and return do {
	my $cgix = flo::Standard::cgi()->new();
	for ($cgix->param()) { $cgix->delete ($_) }
	$cgix->path_info ($url);
	$cgix->self_url;
    };
    
    return do {
	my $cgix = flo::Standard::cgi()->new();
	my $path = flo::Standard::path_info();
	my $relpath = $url;
	$path = File::Spec->canonpath ($path . $relpath);
	$cgix->path_info ($path);
	$cgix->self_url;
    };
}


1;

