# -------------------------------------------------------------------------------------
# flo::editor::Photo
# -------------------------------------------------------------------------------------
# Author : Jean-Michel Hiver.
# Copyright : (c) MKDoc Holdings Ltd, 2003
#
# This file is part of MKDoc. 
# 
# MKDoc is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# MKDoc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with MKDoc; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# -------------------------------------------------------------------------------------
package flo::editor::Photo;
use MKDoc::Util::Text2HTML;
use MKDoc::Config;
use flo::Standard;
use flo::Editor;
use strict;
use Carp;
use Time::Local;

use base qw /flo::Component
	     flo::editor::Mixin::compute_name
	     flo::editor::Mixin::normalize_name
	     flo::editor::Mixin::upload_file
	     flo::editor::Mixin::hr_name
	     /;

sub activate
{
    my $self = shift;
    return unless ($self->SUPER::activate (@_));
    eval "use Image::Magick;";
    return if (defined $@ and $@);
    return 1;
}


##
# $self->internal_location();
# ---------------------------
# Returns the internal location associated with $self.
##
sub internal_location
{
    my $self = shift;
    return MKDoc::Config->IMAGE_HTTP_PATH . "/" . $self->{image};
}


sub internal_location_scaled
{
    my $self = shift;
    my $path = $self->internal_location;
    my $filename;    
    ($path, $filename) = $path =~ /(.*)\/(.*)/;
    $path .= "/scaled/$filename";
    return $path;
}


sub internal_location_small
{
    my $self = shift;
    my $path = $self->internal_location;
    my $filename;
    ($path, $filename) = $path =~ /(.*)\/(.*)/;
    $path .= "/thumbnail/$filename";
    return $path;
}


sub preferred_extension
{
    my $self  = shift;
    my $file  = $self->{image} || return 'jpg';
    $file =~ s/^.*\///;
    my ($ext) = $file =~ /.*\.(.*)/;
    $ext = $1 if ($file =~ /\.([^.]+\.gz)$/);
    $ext ||= $self->type();
    return $ext;
}


sub validate
{
    my $self = shift;
    
    # set up the callback for errors
    local $MKDoc::Ouch::CALLBACK;
    $MKDoc::Ouch::CALLBACK = sub { $self->add_error (@_) };
    
    return $self->validate_image()       &
           $self->validate_title()       &
	   $self->validate_description() &
	   $self->validate_creator()     &
	   $self->validate_date_created();
}


sub validate_image
{
    my $self = shift;
    (defined $self->{image} and $self->{image} ne '') or do {
	new MKDoc::Ouch 'component/photo/no_image';
	return 0;
    };
    
    return 1;
}


sub validate_title
{
    my $self = shift;
    defined $self->{alt} and $self->{alt} !~ /^\s*$/ or do {
	new MKDoc::Ouch 'component/photo/title_empty';
	return 0;
    };
    
    return 1;
}


sub validate_description
{
    my $self = shift;
    defined $self->{description} and $self->{description} !~ /^\s*$/ or do {
	new MKDoc::Ouch 'component/photo/description_empty';
	return 0;
    };
    
    return 1;
}


sub validate_creator
{
    my $self = shift;
    defined $self->{creator} and $self->{creator} !~ /^\s*$/ or do {
	new MKDoc::Ouch 'component/photo/creator_empty';
	return 0;
    };
    
    return 1;
}


sub validate_date_created
{
    my $self = shift;
    my $date = $self->{date_created} or do {
	new MKDoc::Ouch 'component/photo/date_created_empty';
	return 0;
    };

    $date =~ /^\d\d\d\d-\d\d-\d\d$/ or do {
	new MKDoc::Ouch 'component/photo/date_created_malformed';
	return 0;
    };
    
    my ($year, $month, $day) = split /-/, $date;
    $year  =~ s/^0+//;
    $month =~ s/^0+//;
    $day   =~ s/^0+//;
    
    $year  = 0 if ($year eq '');
    $day   = 0 if ($day eq '');
    
    $month == 0 and do {
	new MKDoc::Ouch 'component/photo/date_created_invalid';
	return 0;
    };
    
    $month--;
    $year -= 1900;
    eval { timelocal ( 0, 0, 0, $day, $month, $year) };
    
    $@ and do {
	new MKDoc::Ouch 'component/photo/date_created_invalid';
	return;
    };
    
    return 1;
}


sub new
{
    my $class = shift;
    $class    = ref $class || $class;
    my $self  = bless { @_ }, $class;
    
    my $cgi = $self->{cgi};
    my $param_name = $self->{param_name};   
    
    # if CGI is defined, then we probably want to do some stuff
    if (defined $cgi)
    {
	my $alt         = $cgi->param ($param_name . "_alt");
	my $description = $cgi->param ($param_name . "_description");
	my $creator     = $cgi->param ($param_name . "_creator")      || flo::Standard::current_user()->real_name();
	my $date        = $cgi->param ($param_name . "_date_created") || flo::Standard::current_document()->now_day_iso();
	my $rights      = $cgi->param ($param_name . "_rights");
	my $coverage    = $cgi->param ($param_name . "_coverage");
	
	$self->{alt} = $alt;
	$self->{description} = $description;
	$self->{creator} = $creator;
	$self->{date_created} = $date;
	$self->{rights} = $rights;
	$self->{coverage} = $coverage;
	
	# if the upload field is not empty, then the user
	# just uploaded a new image
	if (defined $cgi->param ($param_name . "_upload"))
	{
	    my $new_image = $cgi->param ($param_name);
	    if ($new_image)
	    {
		my $new_image_no_slash  = $new_image;
		$new_image_no_slash =~ s/^.*\///g;
		$new_image_no_slash =~ s/^.*\\//g;
		
		my $new_image_friendly  = $self->normalize_name ($new_image_no_slash);
                return $self unless $new_image_friendly =~ /\.(jpeg|jpg|jfif|jpe)$/i;
		
		my $dir           = $self->compute_name;
		my $dir_thumbnail = "$dir/thumbnail";
		my $dir_scaled    = "$dir/scaled";
		
		my $full_dir           = MKDoc::Config->IMAGE_DISK_PATH . "/" . $dir;
		my $full_dir_thumbnail = MKDoc::Config->IMAGE_DISK_PATH . "/" . $dir_thumbnail;
		my $full_dir_scaled    = MKDoc::Config->IMAGE_DISK_PATH . "/" . $dir_scaled;
		
		mkdir $full_dir           or confess "Cannot create $dir";
		mkdir $full_dir_thumbnail or confess "Cannot create $full_dir";
		mkdir $full_dir_scaled    or confess "Cannot create $dir_scaled";
		
		$class->upload_file ('image_disk_path', $dir, $new_image_friendly, $cgi->param ($param_name))
		    or return $self;

		
		# we uploaded the photo, now we need to make the scaled
		# and thumbnailed versions of the image
		eval {
		    my $config = new MKDoc::Config ('flo.plugin.Photo');
		    my $scaled_max_x = $config->get ('scaled_max_x');
		    my $scaled_max_y = $config->get ('scaled_max_y');
		    my $thumb_max_x  = $config->get ('thumb_max_x');
		    my $thumb_max_y  = $config->get ('thumb_max_y');
                    my $quality      = $config->get ('quality');
		    
		    my $image = new Image::Magick;
		    my $retcode = $image->Read ("$full_dir/$new_image_friendly");
		    die $retcode if (defined $retcode and $retcode =~ /exception/i);
		    
                    $image->Strip; # requires $Image::Magick::VERSION >= 5.5.8
                    $image->Set (quality => $quality);
		    $image->Resize (geometry => $scaled_max_x . 'x' . $scaled_max_y);
		    $image->Write ("$full_dir_scaled/$new_image_friendly");
		    $image->Resize (geometry => $thumb_max_x  . 'x' . $thumb_max_y);
		    $image->Write ("$full_dir_thumbnail/$new_image_friendly");
		    
		    $self->{image} = "$dir/$new_image_friendly";
		};
		if (defined $@ and $@)
		{
		    carp ("Cannot upload image: $@");
		    $self->{image} = '';
		}

		$self->{alt} ||=
		    $cgi->param ($param_name . "_alt")    ||
		    $class->hr_name ($self->normalize_name_nodot ($new_image_no_slash)) ||
		    'image_title';
	    }
	}
	else
	{
	    $self->{image} = $cgi->param ($param_name . "_img");
	}
    }
    
    return $self;
}


sub uri_small
{
    my $self  = shift;
    my $small = MKDoc::Config->PHOTO_SMALL_HINT;
    return $self->is_saved() ?
        $self->SUPER::uri (@_) . ",$small" :
	$self->Thumbnail_URI (@_);
}


sub uri_scaled
{
    my $self = shift;
    my $scaled = MKDoc::Config->PHOTO_SCALED_HINT;
    return $self->is_saved() ?
        $self->SUPER::uri (@_) . ",$scaled" :
	$self->Scaled_URI (@_);
}


sub uri_html
{
    my $self = shift;
    return $self->SUPER::uri (@_) . ",html";
}


sub uri
{
    my $self = shift;
    return $self->is_saved() ?
        $self->SUPER::uri (@_) :
	$self->Original_URI (@_);
}


sub Thumbnail_URI
{
    my $self = shift;
    my $file = $self->{image};
    return unless (defined $file);
    
    my @path = split /(?:\/|\\)/, $file;
    my $filename = pop (@path);
    push @path, 'thumbnail';
    push @path, $filename;
    my $new_file = join '/', @path;
    return MKDoc::Config->IMAGE_HTTP_PATH . "/" . $new_file;
}


sub Scaled_URI
{
    my $self = shift;
    my $file = $self->{image};
    return unless (defined $file);
    
    my @path = split /(?:\/|\\)/, $file;
    my $filename = pop (@path);
    push @path, 'scaled';
    push @path, $filename;
    my $new_file = join '/', @path;
    return MKDoc::Config->IMAGE_HTTP_PATH . "/" . $new_file;
}


sub Original_URI
{
    my $self = shift;
    my $file = $self->{image};
    return MKDoc::Config->IMAGE_HTTP_PATH . "/" . $file;
}


sub URI
{
    my $self = shift;
    return $self->DC_Identifier;
}


sub title
{
    my $self = shift;
    return $self->{alt};
}


sub DC_Identifier
{
    my $self = shift;
    # dangerous
    my $path_info = $self->Original_URI;
    my $cgix = cgi->new;
    $cgix->path_info ($path_info);
    return $cgix->self_url;
}


sub parse_xml
{
    my $self  = shift;
    my $class = ref $self || $self;
    my $data  = shift;
    if ($data)
    {
	if ($data =~ /<img src/)
	{
	    my ($img, $alt) = $data =~ /<img src=\"(.*?)\" alt=\"(.*)\" \/>/;
	    $self->{image}  = $img;
	    $self->{alt}    = html2text ($alt) || $self->{image};
	}
	else
	{
            my $hash = $self->xml_cache_get ($data) || do {
                my $res = MKDoc::XML::Dumper->xml2perl ($data);
                $self->xml_cache_set ($data, $res);
                $res;
            };

	    for (keys %{$hash}) { $self->{$_} = $hash->{$_} }
	    
	    # more data backwards compatibility options
	    my $day   = delete $hash->{created_day};
	    my $month = delete $hash->{created_month};
	    my $year  = delete $hash->{created_year};
	    
	    $day && $month && $year and do {
		my $date_created = "$year-$month-$day";
		$self->{date_created} = $date_created;
	    };
	}
    }
}


sub html
{
    my $self = shift;
    return '<img src="' . $self->Thumbnail_URI . '" alt="' . text2html ($self->{alt}) . '" />';
}


sub date_created
{
    my $self = shift;
    return $self->{date_created};
}


sub set_date_created
{
    my $self = shift;
    $self->{date_created} = shift;
}


1;

