# -------------------------------------------------------------------------------------
# flo::plugin::Account::Preferences
# -------------------------------------------------------------------------------------
# Author      : Steve Purkis <spurkis@mkdoc.com>,
# Last-Update : Jean-Michel Hiver
# Copyright   : MKDoc Holdings Ltd, 2003
#
# This file is part of MKDoc. 
# 
# MKDoc is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# MKDoc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with MKDoc; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#
# This plugin lets an existing user edit their Language, Audience, newsletter
# preferences, and their user details.
# -------------------------------------------------------------------------------------
package flo::plugin::Account::Preferences;
use MKDoc::Ouch;
use flo::Standard;
use flo::Plugin;
use Carp;
use Petal;
use strict;
use 5.008_000;
use utf8;


use base qw /flo::Plugin/;


sub template_path { return '/account/preferences' };


sub http_get
{
    my $self = shift;
    my $cgi = flo::Standard::cgi();
    ($ENV{REQUEST_METHOD} =~ /^(?:GET|HEAD)$/) and do {
	$cgi->param ('password', $self->user()->password());
	$cgi->param ('password_verify', $self->user()->password());
    };
    $self->render_http (
	self       => $self,
	object     => $self->parent,
	__input__  => 'XML',
	__output__ => 'XHTML',
       );
    return 'TERMINATE';
}


sub http_post
{
    my $self  = shift;
    my $cgi   = $self->cgi();
    $self->_update_user_from_cgi();
    $self->_update_preferences_from_cgi();

    if ($cgi->param ('goto'))
    {
        print $cgi->redirect ($cgi->param ('goto'));
        return 'TERMINATE';
    }

    return $self->http_get();
}


sub _update_user_from_cgi
{
    my $self   = shift;
    my $user   = $self->user;
    my $cgi    = $self->cgi;
    
    $user->set_email ($cgi->param ('email'));
    $user->set_first_name ($cgi->param ('first_name'));
    $user->set_family_name ($cgi->param ('family_name'));
    
    my $pwd1 = $cgi->param ('password')        || '';
    my $pwd2 = $cgi->param ('password_verify') || '';
    ($pwd1 eq $pwd2) or do {
	new MKDoc::Ouch 'user/password/mismatch';
	return;
    };
    
    $user->set_password ($pwd1);
    $user->validate() || return;
    $user->save();
}


sub _update_preferences_from_cgi
{
    my $self   = shift;
    my $prefs  = $self->preferences;
    my $cgi    = $self->cgi;

    foreach my $param ($cgi->param)
    {
	if ($param =~ /^((?:general)|(?:language)|(?:audience))_([\w\-\.\:]+)$/)
	{
	    my $type      = $1;
	    my $name      = $2;
	    my $new_value = $cgi->param ($param);
	    
	    $cgi->delete($param);
	    next unless defined($new_value);
	    
	    my $get_method = "$type\_preference";
	    my $set_method = "set_$type\_preference";
	    my $old_value  = $prefs->$get_method ($name);
	    unless (defined $old_value)
	    {
		$self->_notify_error ($type, $name);
	    }
	    else
	    {
		if ($new_value ne $old_value)
		{
		    unless ($prefs->$set_method ($name, $new_value))
		    {
			$self->_notify_error ($type, $name);
		    }
		}
	    }
	}
    }
}


sub _notify_error
{
    my $class = shift;
    my $type  = shift;
    my $name  = shift;
  CASE:
    local $_ = "$type.$name";
    for ($_)
    {
	/email\.not_set/        and do { new MKDoc::Error ( template => '/error/email.not_set/'      ); last CASE };
	/first_name\.not_set/   and do { new MKDoc::Error ( template => '/error/first_name.not_set/' ); last CASE };
	/family_name\.not_set/  and do { new MKDoc::Error ( template => '/error/last_name.not_set/'  ); last CASE };
	/password\.not_set/     and do { new MKDoc::Error ( template => '/error/password.not_set/'   ); last CASE };
	/password\.too_short/   and do { new MKDoc::Error ( template => '/error/password.length/'    ); last CASE };
	/password\.mismatch/    and do { new MKDoc::Error ( template => '/error/password.mismatch/'  ); last CASE };
	new MKDoc::Error ( template => '/error/', info => "preferences.$type.$name not defined" );
    }
}


sub preferences ($)
{
    my $self = shift;
    return $self->user()->preferences();
}


sub activate
{
    my $self = shift;
    return 0 unless ($self->SUPER::activate(@_));
    return 0 unless ($self->user);
    return $self;
}


1;


__END__
