# -------------------------------------------------------------------------------------
# flo::plugin::Account::Subscribe
# -------------------------------------------------------------------------------------
# Author : Steve Purkis <spurkis@mkdoc.com>, Jean-Michel Hiver
# Copyright : (c) MKDoc Holdings Ltd, 2003.
#
# This file is part of MKDoc. 
# 
# MKDoc is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# MKDoc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with MKDoc; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# -------------------------------------------------------------------------------------
package flo::plugin::Account::Subscribe;
use flo::Record::Ticket;
use flo::Record::Editor;
use flo::Standard;

# pull in Mail::CheckUser and remember whether it worked or not
use vars qw /$HAVE_CHECKMAIL/;
BEGIN {
    eval "use Mail::CheckUser";
    $HAVE_CHECKMAIL = 1 unless ($@);
}

use Carp;
use Petal;
use Petal::Mail;

use MKDoc::Config;
use MKDoc::Ouch;

use strict;
use 5.008_000;
use utf8;

use constant ACTION => 'subscribe';
use constant DEFAULT_EXPIRY_TIME => 7*24*60*60;

use base qw /flo::Plugin/;

sub template_path { '/account/subscribe' }


##
# $self->http_get;
# ----------------
# Displays the form which lets the editor choose which audiences
# the current document relates to.
##
sub http_get
{
    my $self = shift;
    $self->render_http (
	self       => $self,
	object     => $self->parent,
	__input__  => 'XML',
	__output__ => 'XHTML',
       );    
    return 'TERMINATE';
}


##
# $self->http_post;
# -----------------
# Reflect the changes on the database and redirects to
# current document.
##
sub http_post
{
    my $self = shift;
    my $cgix = flo::Standard::cgi()->new;
    my $email  = $cgix->param ('email');
    
    (defined $email and $email) or do {
	new MKDoc::Ouch 'subscribe/no_email';
	return $self->http_get();
    };
    
    ($email =~ /[^A-Za-z0-9'@._+-]/) and do {
	new MKDoc::Ouch 'subscribe/malformed_email';
	return $self->http_get();
    };

    if ($HAVE_CHECKMAIL) {
        # setup Mail::CheckUser options
        local $Mail::CheckUser::Timeout = 5;
        
        # check the address using Mail::CheckUser
        unless(Mail::CheckUser::check_email($email)) {
            new MKDoc::Ouch 'user/email/advanced_checks';
            return $self->http_get();
        }
    }

    $self->{email} = $email;
    $self->{ticket} = $self->_create_ticket ($email);
    $self->_send_mail ($self->{ticket});

    return $self->http_get;
}


##
# $self->_create_ticket;
# ----------------------
# Creates the ticket and inserts it in the database.
# Returns the newly created ticket.
##
sub _create_ticket
{
    my $self  = shift;
    my $email = shift;
    
    # clear old tickets & remove previous tickets
    # this is so that a user can re-ask for a ticket
    # rather than wait for his ticket to have expired
    flo::Record::Ticket::clear_old_tickets_from_db();
    my $ticket_table = flo::Standard::table ('Tickets');
    $ticket_table->delete ( Email => $email, Action => ACTION );
    
    # create a new ticket and returns it
    my $ticket = new flo::Record::Ticket;
    eval {
	$ticket->generateId;
	$ticket->Email ($email);
	$ticket->ExpiryDate (time + DEFAULT_EXPIRY_TIME);
	$ticket->Action ( ACTION );
	$ticket->saveToDb();
    };
    
    defined $@ and $@ and do {
	my $error = new MKDoc::Error (
	    template => '/error',
	    lang     => $self->lang(),
	    info     => $@,
	   );
    };
    
    return $ticket;
}


##
# $self->_send_mail ($ticket);
# ----------------------------
# Sends the email...
##
sub _send_mail
{
    my $self   = shift;
    
    my $xml_mail = eval {
        my $mail = new Petal::Mail (
            file => '/account/emails/subscribe',
            language => $self->language(),
        );

	$mail->send (
	    self       => $self,
	    object     => $self->parent(),
	    __input__  => 'XML',
	    __output__ => 'XML',
	);
    };
    defined $@ and $@ and do {
	warn $@;
	new MKDoc::Ouch 'mail/cannot_create';
	return 0;
    };

    return 1;
}


1;


__END__
