# -------------------------------------------------------------------------------------
# flo::plugin::Admin::Copy
# -------------------------------------------------------------------------------------
# Author : Jean-Michel Hiver
# Copyright : (c) MKDoc Holdings Ltd, 2003
#
# This file is part of MKDoc. 
# 
# MKDoc is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# MKDoc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with MKDoc; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#
# Lets an editor delete the current document. Redirects to the parent document.
# -------------------------------------------------------------------------------------
package flo::plugin::Admin::Copy;
use MKDoc::Config;
use MKDoc::Error;
use flo::Standard;
use Encode;
use strict;
use warnings;
use 5.008_000;
use Carp;

use base qw /flo::Plugin/;

sub template_path { 'admin/copy' }


##
# $self->activate;
# ----------------
# Returns TRUE if this plugin can be activated, FALSE
# otherwise.
##
sub activate
{
    my $self = shift;
    return unless ($self->SUPER::activate (@_));
    my $current_document = flo::Standard::current_document();
    return $current_document->Can_Admin;
}


sub http_get
{
    my $self = shift;
    my $doc  = flo::Standard::current_document();
    $self->render_http (
	self       => $self,
	__input__  => 'XML',
	__output__ => 'XHTML',
       );
    
    return 'TERMINATE';
}


sub _validate_defined
{
    my $self = shift;
    my $to = shift;
    defined $to and $to or do {
	new MKDoc::Ouch 'document/copy/no_target';
	return 0;
    };
    return 1;
}


sub _validate_exists
{
    my $self = shift;
    my $to   = shift;
    my $document_t = flo::Standard::table ('Document');
    $document_t->get ($to) or do {
	new MKDoc::Ouch 'document/copy/no_target';
	return 0;
    };
    return 1;
}


sub _validate_perms
{
    my $self = shift;
    my $to   = shift;
    my $document_t = flo::Standard::table ('Document');
    my $doc  = $document_t->get ($to);
    $doc->Can_Admin() or do {
	new MKDoc::Ouch 'document/copy/denied';
	return 0;
    };
    return 1;
}


sub http_post
{
    my $self  = shift;
    my $cgix  = flo::Standard::cgi()->new();
    my $to    = $cgix->param ('to');
    my $doc   = flo::Standard::current_document();
    my $lang  = $doc->language();
    
    $self->_validate_defined ($to) || return $self->http_get();
    $self->_validate_exists  ($to) || return $self->http_get();
    $self->_validate_perms   ($to) || return $self->http_get();

    my $document_t = flo::Standard::table ('Document');
    my $new_parent = $document_t->get ($to);
    my $new_doc    = $doc->copy_to ($new_parent);
    
    $self->has_errors() and return $self->http_get();
    
    # redirects to the new place that $doc has been copied to.
    $new_doc ||= $new_parent;
    $cgix->delete ($cgix->param());
    $cgix->path_info ($new_doc->path());
    print $cgix->redirect ($cgix->self_url);
    return 'TERMINATE';
}


sub copiable_document_list
{
    my $self = shift;
    
    my $document_t = flo::Standard::table ('Document');
    my $query = $document_t->select (
	cols   => [ qw /ID Parent_ID Cache_Control Template Description Keywords
			Date_Created Date_Last_Modified Editor_Created_ID Editor_Last_Modified_ID
			Name Full_Path Title Lang Rights Publisher Source Relation Coverage
			Sibling_Position Sort_By Order_By/ ], 
	sort   => [ 'Full_Path' ],
	desc   => 0
       );
    
    my @res = ();
    while (my $doc = $query->next())
    {
	push @res, $doc if ($doc->Can_Admin());
    }
    
    return wantarray ? @res : \@res;
}


1;
