package flo::plugin::Admin::Group;
use strict;
use warnings;

=head1 NAME

flo::plugin::Admin::Group - administrative interface for groups

=head1 SYNOPSIS

Link to this plugin like so:

  <a href=".admin.group">Groups</a>

=head1 DESCRIPTION

This plugin provides an administrative interface for groups.

=head1 AUTHOR

Sam Tregar <sam@tregar.com>

=head1 COPYRIGHT

Copyright MKDoc Holdings Ltd, 2005

=head1 LICENSE

MKDoc is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free
Software Foundation; either version 2 of the License, or (at your
option) any later version.

MKDoc is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License
along with MKDoc; if not, write to the Free Software Foundation, Inc.,
59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

=cut

use flo::Standard;
use base qw /flo::Plugin/;

# only admin can access this plugin
sub activate {
    my $self = shift;
    $self->SUPER::activate(@_) || return;
    $self->user()->id() == 1   || return;
    return 1;
}

# setup the template for this runmode
sub template_path { 
    my $self = shift;
    my $rm   = $self->runmode;
    return 'admin/group_list' if $rm eq 'list';
    return 'admin/group_edit' if $rm eq 'edit';
    return 'admin/group_add'  if $rm eq 'add';
    return 'admin/group_confirm_delete'  if $rm eq 'confirm_delete';
    die "Couldn't find a template for runmode $rm!";
}

# HTTP form handlers
sub http_get  { 
    my $self = shift;
    my $rm   = $self->runmode;
    if ($rm eq 'list') {
        return $self->output();
    } elsif ($rm eq 'edit') {
        return $self->output();
    } elsif ($rm eq 'add') {
        return $self->output();
    } elsif ($rm eq 'save') {
        return $self->save();
    } elsif ($rm eq 'confirm_delete') {
        return $self->output();
    } elsif ($rm eq 'delete') {
        return $self->delete_group();
    } else {
        die "Unknown runmode $rm.";
    }
}
sub http_post { shift()->http_get(@_) }

# render the template for this runmode
sub output {
    my $self = shift;

    # render the template
    $self->render_http(
                       self       => $self,
                       object     => $self->parent,
                       __input__  => 'XML',
                       __output__ => 'XHTML',
                      );
    return 'TERMINATE';
}

# delete a group
sub delete_group {
    my $self = shift;
    my $dbh = lib::sql::DBH->get();
    my $cgi = flo::Standard::cgi();
    my $id  = $cgi->param('id');

    # wipe out the group
    $dbh->do('DELETE FROM Editor_Grp WHERE Grp_ID = ?',   undef, $id);
    $dbh->do('DELETE FROM Document_Grp WHERE Grp_ID = ?', undef, $id);
    $dbh->do('DELETE FROM Grp WHERE ID = ?',              undef, $id);

    # send the user back to list view
    $self->runmode('list');
    return $self->output();
}

# save data from edit or add forms
sub save {
    my $self = shift;
    my $dbh = lib::sql::DBH->get();
    my $cgi = flo::Standard::cgi();
    my $id          = $cgi->param('id');
    my $name        = $cgi->param('name');
    my $description = $cgi->param('description');

    # make sure name isn't empty
    unless (defined $name and length $name) {
        new MKDoc::Ouch 'admin/group/missing_name';
    }

    # make sure the name isn't taken
    if ($id) {
        if ($dbh->selectrow_array('SELECT 1 FROM Grp 
                                   WHERE Name = ? AND ID != ?', 
                                  undef, $name, $id)) {
            new MKDoc::Ouch 'admin/group/name_conflict';
        }
    } else {
        if ($dbh->selectrow_array('SELECT 1 FROM Grp WHERE Name = ?',
                                  undef, $name)) {
            new MKDoc::Ouch 'admin/group/name_conflict';
        }
    }

    # send the user back to edit/add if there are problems
    if ($self->has_errors) {
        $self->runmode($id ? 'edit' : 'add');
        return $self->output();
    }
    
    # do the update/insert
    if ($id) {
        $dbh->do('UPDATE Grp SET Name = ?, Description = ? WHERE ID = ?',
                 undef, $name, $description, $id);
    } else {
        $dbh->do('INSERT INTO Grp (Name, Description) VALUES (?,?)',
                 undef, $name, $description);
    }

    # send the user back to list
    $self->runmode('list');
    return $self->output();
}

sub groups {
    my $dbh = lib::sql::DBH->get();
    my $results = $dbh->selectall_arrayref(
      'SELECT ID, Name, Description FROM Grp ORDER BY name');
    return [ map { { id          => $_->[0],
                     name        => $_->[1],
                     description => $_->[2] } } @$results ];
}

# return group data for the current group if ID is set.  If ID is
# unset it just echos back the query data.
sub group {
    my $self = shift;
    my $cgi = flo::Standard::cgi();
    my $group_id = $cgi->param('id');

    my ($name, $description) = 
      ($cgi->param('name'), $cgi->param('description'));

    if ($group_id) {
        my $dbh = lib::sql::DBH->get();
        my $results = $dbh->selectrow_arrayref(
                        'SELECT Name, Description FROM Grp WHERE ID = ?', 
                        undef, $group_id);
        $name        = defined $name        ? $name        : $results->[0];
        $description = defined $description ? $description : $results->[1];
    }

    return { id          => $group_id,
             name        => $name,
             description => $description };
}

# gets and sets the current runmode;
sub runmode {
    my $self = shift;
    my $cgi = flo::Standard::cgi();
    $cgi->param(rm => shift) if @_;
    return $cgi->param('rm') || 'list';
}

1;
